package net.sf.csutils.groovy;

import java.util.Collection;
import java.util.Locale;
import java.util.Map;

import net.sf.csutils.core.utils.RegistryObjects;

public class GroovyObjects {
    /**
     * Converts the given arguments into a locale.
     * @param pMethodName The method name; will be used for error messages, if required.
     * @param pArgs The method arguments.
     * @return A locale matching the arguments.
     */
    public static Locale getLocaleArg(String pMethodName, Object[] pArgs) {
        if (pArgs == null) {
            return null;
        }
        final Object o;
        final Object[] args = pArgs;
        if (args.length == 0) {
            return null;
        }
        if (args.length > 1) {
            throw new IllegalArgumentException("The method "
                    + pMethodName
                    + " accepts a string argument, but not multiple arguments.");
        }
        o = args[0];
        if (o instanceof Locale) {
            return (Locale) o;
        }
        return RegistryObjects.asLocale(o.toString());
    }

    /**
	 * Converts the given arguments into a locale.
     * @param pMethodName The method name; will be used for error messages, if required.
     * @param pArgs The method arguments.
     * @return A locale matching the arguments.
	 */
	public static Locale getLocaleArg(String pMethodName, Object pArgs) {
		if (pArgs == null) {
			return null;
		}
		final Object o;
		if (pArgs.getClass().isArray()  &&  !pArgs.getClass().getComponentType().isPrimitive()) {
			final Object[] args = (Object[]) pArgs;
			if (args.length == 0) {
				return null;
			}
			if (args.length > 1) {
				throw new IllegalArgumentException("The method "
						+ pMethodName
						+ " accepts a string argument, but not multiple arguments.");
			}
			o = args[0];
		} else if (pArgs instanceof Collection) {
			final Collection<?> collection = (Collection<?>) pArgs;
			o = collection.isEmpty() ? null : collection.iterator().next();
		} else if (pArgs instanceof Map) {
			throw new IllegalArgumentException("The method "
					+ pMethodName
					+ " accepts a string argument, but not multiple arguments.");
		} else {
			o = pArgs;
		}
		if (o == null) {
			return null;
		}
		if (o instanceof Locale) {
			return (Locale) o;
		}
		return RegistryObjects.asLocale(o.toString());
	}

	/**
	 * Converts the given arguments into a string.
     * @param pMethodName The method name; will be used for error messages, if required.
     * @param pArgs The method arguments.
     * @return The converted arguments.
	 */
	public static String getStringArg(String pMethodName, Object pArgs) {
		if (pArgs == null) {
			return null;
		}
		final Object o;
		if (pArgs.getClass().isArray()  &&  !pArgs.getClass().getComponentType().isPrimitive()) {
			final Object[] args = (Object[]) pArgs;
			if (args.length == 0) {
				return null;
			}
			if (args.length > 1) {
				throw new IllegalArgumentException("The method "
						+ pMethodName
						+ " accepts a string argument, but not multiple arguments.");
			}
			o = args[0];
		} else if (pArgs instanceof Collection) {
			final Collection<?> collection = (Collection<?>) pArgs;
			o = collection.isEmpty() ? null : collection.iterator().next();
		} else if (pArgs instanceof Map) {
			throw new IllegalArgumentException("The method "
					+ pMethodName
					+ " accepts a string argument, but not multiple arguments.");
		} else {
			o = pArgs;
		}
		if (o == null) {
			return null;
		}
		return o.toString();
	}

	/**
	 * Checks, whether the given argument is suitable for a method taking
	 * no arguments.
     * @param pMethodName The method name; will be used for error messages, if required.
     * @param pArgs The method arguments.
	 */
	public static void checkNoArgs(String pMethodName, Object pArgs) {
		if (pArgs == null) {
			return;
		}
		if (pArgs.getClass().isArray()) {
			if (pArgs.getClass().getComponentType().isPrimitive()) {
				throw new IllegalStateException("The method "
						+ pMethodName + " accepts no primitive arguments.");
			}
			final Object[] args = (Object[]) pArgs;
			if (args.length == 0) {
				return;
			}
		} else if (pArgs instanceof Collection) {
			final Collection<?> collection = (Collection<?>) pArgs;
			if (collection.isEmpty()) {
				return;
			}
		}
		throw new IllegalStateException("The method "
				+ pMethodName + " accepts no arguments.");
	}

    /**
     * Checks, whether the given argument is suitable for a method taking
     * no arguments.
     * @param pMethodName The method name; will be used for error messages, if required.
     * @param pArgs The method arguments.
     */
    public static void checkNoArgs(String pMethodName, Object[] pArgs) {
        if (pArgs == null) {
            return;
        }
        if (pArgs.length == 0) {
            return;
        }
        throw new IllegalStateException("The method "
                + pMethodName + " accepts no arguments.");
    }

    /**
     * Checks, whether the given signature contains a single string.
     * @param pMethodName The method name; will be used for error messages, if required.
     * @param pArgs The method arguments.
     * @return The string, which has been found in the argument list.
     */
    public static String checkStringArg(String pMethodName, Object[] pArgs) {
        if (pArgs != null  &&  pArgs.length == 1) {
            return String.valueOf(pArgs[0]);
        }
        throw new IllegalStateException("The method "
                + pMethodName + " accepts exactly one string argument.");
    }
}
