package net.sf.csutils.groovy;

import groovy.text.GStringTemplateEngine;
import groovy.text.SimpleTemplateEngine;
import groovy.text.Template;
import groovy.text.TemplateEngine;
import groovy.text.XmlTemplateEngine;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilterInputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import net.sf.csutils.core.registry.DefaultConnectionProvider;


/**
 * Main class of a template processor.
 */
public class Main {
	public abstract static class Opt {
		private final String name;
		private final String shortDescription, longDescription;
		private final boolean isRequired;
		private boolean isSet;

		public Opt(String pName, String pShortDescription, String pLongDescription,
				boolean pRequired) {
			name = pName;
			shortDescription = pShortDescription;
			longDescription = pLongDescription;
			isRequired = pRequired;
		}

		protected void setSet(boolean pSet) {
			isSet = pSet;
		}

		public boolean isSet() {
			return isSet;
		}

		public boolean isRequired() {
			return isRequired;
		}

		public String getName() {
			return name;
		}

		public String getShortDescription() {
			return shortDescription;
		}

		public String getLongDescription() {
			return longDescription;
		}
	}

	public abstract static class StringOpt extends Opt {
		public StringOpt(String pName, String pShortDescription, String pLongDescription, boolean pRequired) {
			super(pName, pShortDescription, pLongDescription, pRequired);
		}
		public abstract void setValue(String pValue);
	}

	public abstract static class BooleanOpt extends Opt {
		public BooleanOpt(String pName, String pShortDescription, String pLongDescription) {
			super(pName, pShortDescription, pLongDescription, false);
		}
		public abstract void setValue(boolean pValue);
	}

	public abstract static class FileOpt extends Opt {
		private boolean mustExist;

		public FileOpt(String pName, String pShortDescription, String pLongDescription,
				boolean pRequired, boolean pMustExist) {
			super(pName, pShortDescription, pLongDescription, pRequired);
			mustExist = pMustExist;
		}

		public boolean isMustExist() {
			return mustExist;
		}
		
		public abstract void setValue(File pValue);
	}

	public abstract static class URLOpt extends Opt {
		public URLOpt(String pName, String pShortDescription, String pLongDescription,
				boolean pRequired) {
			super(pName, pShortDescription, pLongDescription, pRequired);
		}
		public abstract void setValue(URL pValue);
	}

	public static class UsageException extends RuntimeException {
		private static final long serialVersionUID = -7702096820277892721L;

		public UsageException(String message) {
			super(message);
		}
	}
	
	public enum TemplateType {
		/**
		 * Uses {@link SimpleTemplateEngine}.
		 */
		simple,
		/**
		 * Uses {@link GStringTemplateEngine}.
		 */
		gstring,
		/**
		 * Uses {@link XmlTemplateEngine}.
		 */
		gxml,
		/**
		 * Uses {@link net.sf.csutils.groovy.xml.XmlTemplateEngine}.
		 */
		xml
	}
	
	private String csUrl, csUser, csPassword, inputEncoding, outputEncoding;
	private File inputFile, outputFile;
	private TemplateType templateType = TemplateType.gstring;

	public TemplateType getTemplateType() {
		return templateType;
	}

	public void setTemplateType(TemplateType templateType) {
		this.templateType = templateType;
	}

	public File getInputFile() {
		return inputFile;
	}

	public void setInputFile(File pInputFile) {
		inputFile = pInputFile;
	}

	public File getOutputFile() {
		return outputFile;
	}

	public void setOutputFile(File pOutputFile) {
		outputFile = pOutputFile;
	}

	public URL getInputURL() {
		return inputURL;
	}

	public void setInputURL(URL pInputURL) {
		inputURL = pInputURL;
	}

	private URL inputURL;

	public String getInputEncoding() {
		return inputEncoding;
	}

	public void setInputEncoding(String pInputEncoding) {
		inputEncoding = pInputEncoding;
	}

	public String getOutputEncoding() {
		return outputEncoding;
	}

	public void setOutputEncoding(String pOutputEncoding) {
		outputEncoding = pOutputEncoding;
	}

	public String getCsUrl() {
		return csUrl;
	}

	public void setCsUrl(String pUrl) {
		csUrl = pUrl;
	}

	public String getCsUser() {
		return csUser;
	}

	public void setCsUser(String pUser) {
		csUser = pUser;
	}

	public String getCsPassword() {
		return csPassword;
	}

	public void setCsPassword(String pPassword) {
		csPassword = pPassword;
	}

	public List<Opt> getOptions() {
		final Opt[] opts = new Opt[]{
			new StringOpt("csUrl", "CentraSite URL", "Sets the CentraSite URL.", true){
				@Override
				public void setValue(String pValue) {
					setCsUrl(pValue);
				}
			},
			new StringOpt("csUser", "CentraSite user", "Sets the CentraSite user.", true){
				@Override
				public void setValue(String pValue) {
					setCsUser(pValue);
				}
			},
			new StringOpt("csPassword", "CentraSite password", "Sets the CentraSite users password.", true){
				@Override
				public void setValue(String pValue) {
					setCsPassword(pValue);
				}
			},
			new StringOpt("inputEncoding", "Encoding", "Sets the templates encoding, defaults to UTF8.", false){
				@Override
				public void setValue(String pValue) {
					setInputEncoding(pValue);
				}
			},
			new StringOpt("outputEncoding", "Encoding", "Sets the output files encoding, defaults to UTF8.", false){
				@Override
				public void setValue(String pValue) {
					setOutputEncoding(pValue);
				}
			},
			new FileOpt("inputFile", "Input file", "Sets the input templates file.", false, true){
				@Override
				public void setValue(File pValue) {
					if (getInputURL() != null) {
						throw new UsageException("The options inputFile and inputURL are mutually exclusive.");
					}
					setInputFile(pValue);
				}
			},
			new URLOpt("inputUrl", "Input URL", "Sets the input templates URL.", false){
				@Override
				public void setValue(URL pValue) {
					if (getInputFile() != null) {
						throw new UsageException("The options inputFile and inputURL are mutually exclusive.");
					}
					setInputURL(pValue);
				}
			},
			new FileOpt("outputFile", "Output file", "Sets the output file.", false, false){
				@Override
				public void setValue(File pValue) {
					setOutputFile(pValue);
				}
			},
			new StringOpt("templateType", "Template type", "Sets the template type, either of simple|gstring|gxml|xml, defaults to gstring.", false){
				@Override
				public void setValue(String pValue) {
					try {
						setTemplateType(TemplateType.valueOf(pValue.toLowerCase()));
					} catch (IllegalArgumentException e) {
						throw new UsageException("Invalid value for option templateType: " +
								pValue + ", expexted simple|gstring|gxml|xml.");
								
					}
				}
			},
			new BooleanOpt("help", "Usage message", "Prints the usage message and exits with error status."){
				@Override
				public void setValue(boolean pValue) {
					usage(null, getOptions());
				}
			}
		};
		return new ArrayList<Opt>(Arrays.asList(opts));
	}

	private String getIntro(Opt pOpt) {
		if (pOpt instanceof StringOpt) {
			return("  -" + pOpt.getName() + " <s>");
		} else if (pOpt instanceof FileOpt) {
			return("  -" + pOpt.getName() + " <f>");
		} else if (pOpt instanceof URLOpt) {
			return("  -" + pOpt.getName() + " <u>");
		} else if (pOpt instanceof BooleanOpt) {
			return("  -" + pOpt.getName());
		} else {
			throw new IllegalStateException("Invalid option type: " + pOpt.getClass().getName());
		}
	}

	private void show(int pIntroLength, List<Opt> pOptions) {
		for (Opt opt : pOptions) {
			final StringBuilder intro = new StringBuilder(getIntro(opt));
			while (intro.length() < pIntroLength) {
				intro.append(' ');
			}
			System.err.println(intro + opt.getLongDescription());
		}
	}
	
	protected void usage(String pMessage, List<Opt> pOptions) {
		if (pMessage != null) {
			System.err.println(pMessage);
			System.err.println();
		}
		System.err.println("Usage: java " + getClass().getName() + " <options>");
		System.err.println();
		final List<Opt> allOptions = new ArrayList<Opt>(pOptions);
		Collections.sort(allOptions, new Comparator<Opt>(){
			public int compare(Opt o1, Opt o2) {
				return o1.getName().compareToIgnoreCase(o2.getName());
			}
		});
		final List<Opt> requiredOptions = new ArrayList<Opt>();
		final List<Opt> optionalOptions = new ArrayList<Opt>();
		int maxIntroLength = 0;
		for (Opt opt : allOptions) {
			int introLength = getIntro(opt).length();
			if (introLength > maxIntroLength) {
				maxIntroLength = introLength;
			}
			if (opt.isRequired()) {
				requiredOptions.add(opt);
			} else {
				optionalOptions.add(opt);
			}
		}
		System.err.println("Required options are: ");
		show(maxIntroLength+2, requiredOptions);
		System.err.println("Other options are: ");
		show(maxIntroLength+2, optionalOptions);
		System.exit(1);
	}

	private OutputStream getNonCloseableOutputStream(OutputStream pOut) {
	    return new FilterOutputStream(pOut){
            @Override
            public void close() throws IOException {
                super.close();
            }
	    };
	}

	private InputStream getNonCloseableInputStream(InputStream pIn) {
	    return new FilterInputStream(pIn){
	        @Override
	        public void close() throws IOException {
	            super.close();
	        }
	    };
	}

	protected Writer getOutputWriter() throws IOException {
	    OutputStream stream = null;
		try {
	        final File out = getOutputFile();
	        if (out == null  ||  "-".equals(out.getName())) {
	            stream = getNonCloseableOutputStream(System.out);
	        } else {
	            stream = new FileOutputStream(getOutputFile());
	        }
			final String encoding = getOutputEncoding() == null ? "UTF8" : getOutputEncoding();
			final Writer writer = new OutputStreamWriter(stream, encoding);
			stream = null;
			return writer;
		} finally {
			if (stream != null) { try { stream.close(); } catch (Throwable t) { /* Ignore me */ } }
		}
	}

	protected Reader getInput() throws IOException {
		InputStream stream = null;
		try {
			final File f = getInputFile();
			if (f == null) {
				final URL u = getInputURL();
				if (u == null) {
				    stream = getNonCloseableInputStream(System.in);
				} else {
    				try {
    					stream = u.openStream();
    				} catch (IOException e) {
    					throw new IOException("Failed to open input URL " + u
    							+ ": " + e.getMessage(), e);
    				}
				}
			} else if ("-".equals(f.getName())) {
			    stream = getNonCloseableInputStream(System.in);
			} else {
				try {
					stream = new FileInputStream(f);
				} catch (IOException e) {
					throw new IOException("Failed to open input file "
							+ f.getPath() + ": " + e.getMessage(), e);
				}
			}
			final String encoding = getInputEncoding() == null ? "UTF8" : getInputEncoding();
			final Reader result = new InputStreamReader(stream, encoding);
			stream = null;
			return result;
		} finally {
			if (stream != null) { try { stream.close(); } catch (Throwable t) { /* Ignore me */ } }
		}
	}

	protected TemplateEngine getTemplateEngine() throws Exception {
		switch (getTemplateType()) {
			case gstring:
				return new GStringTemplateEngine();
			case simple:
				return new SimpleTemplateEngine();
			case gxml:
				return new XmlTemplateEngine();
			case xml:
				return new net.sf.csutils.groovy.xml.XmlTemplateEngine();
			default:
				throw new IllegalStateException("Invalid template type: " + getTemplateType());
		}
	}

	public void run() throws Exception {
		final URL dbUrl;
		try {
			dbUrl = new URL(getCsUrl());
		} catch (Exception e) {
			throw new UsageException("Invalid database URL: " + getCsUrl());
		}
		Reader reader = null;
		Writer writer = null;
		try {
			reader = getInput();
			final TemplateEngine engine = getTemplateEngine();
			final Template template = engine.createTemplate(reader);
            writer = getOutputWriter();
			QueryEngine.run(template, writer, new DefaultConnectionProvider(dbUrl, getCsUser(), getCsPassword()));
			writer.close();
			writer = null;
			reader.close();
			reader = null;
		} finally {
			if (writer != null) { try { writer.close(); } catch (Throwable t) { /* Ignore me */ } }
			if (reader != null) { try { reader.close(); } catch (Throwable t) { /* Ignore me */ } }
		}
	}
	
	public static void main(String[] pArgs) throws Exception {
		final Main main = new Main();
		main(main, pArgs);
	}

	public static void main(final Main pMain, String[] pArgs) throws Exception {
		final List<Opt> options = pMain.getOptions();
		try {
			final List<String> args = new ArrayList<String>(Arrays.asList(pArgs));
			while (!args.isEmpty()) {
				final String arg = args.remove(0);
				final String optName;
				if (arg.startsWith("--")) {
					optName = arg.substring(2);
				} else if (arg.startsWith("-")) {
					optName = arg.substring(1);
				} else {
					throw new UsageException("Invalid option: " + arg);
				}
				Opt opt = null;
				for (Opt o : options) {
					if (o.getName().equals(optName)) {
						opt = o;
						break;
					}
				}
				if (opt == null) {
					throw new UsageException("Unknown option: " + arg);
				}
				if (opt instanceof StringOpt) {
					if (args.isEmpty()) {
						throw new UsageException("Option " + arg
								+ " requires an argument (" + opt.getShortDescription() + ")");
					}
					((StringOpt) opt).setValue(args.remove(0));
				} else if (opt instanceof FileOpt) {
					if (args.isEmpty()) {
						throw new UsageException("Option " + arg
								+ " requires an argument (" + opt.getShortDescription() + ")");
					}
					final File f = new File(args.remove(0));
					final FileOpt fileOpt = (FileOpt) opt;
					if (fileOpt.isMustExist()  &&  !f.isFile()) {
						throw new UsageException("The file " + f.getPath()
								+ ", specified by option " + arg
								+ " does not exist.");
					}
					fileOpt.setValue(f);
				} else if (opt instanceof URLOpt) {
					if (args.isEmpty()) {
						throw new UsageException("Option " + arg
								+ " requires an argument (" + opt.getShortDescription() + ")");
					}
					final String s = args.remove(0);
					final URL url;
					try {
						url = new URL(s);
					} catch (MalformedURLException e) {
						throw new UsageException("Invalid URL " + s
								+ ", specified by option " + arg);
					}
					final URLOpt urlOpt = (URLOpt) opt;
					urlOpt.setValue(url);
				} else {
					throw new IllegalStateException("Invalid option type: "
							+ opt.getClass().getName());
				}
				opt.setSet(true);
			}
			for (Opt opt : options) {
				if (opt.isRequired()  &&  !opt.isSet()) {
					throw new UsageException("The mandatory option " + opt.getName()
							+ " is missing.");
				}
			}
			pMain.run();
		} catch (UsageException e) {
			pMain.usage(e.getMessage(), options);
		}
	}
}
