package net.sf.csutils.groovy;

import groovy.lang.Closure;
import groovy.lang.GroovyObject;
import groovy.lang.GroovySystem;
import groovy.lang.MetaClass;
import groovy.lang.MetaClassRegistry;
import groovy.text.Template;

import java.io.IOException;
import java.io.Writer;
import java.lang.reflect.UndeclaredThrowableException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.registry.Connection;
import javax.xml.registry.JAXRException;
import javax.xml.registry.infomodel.InternationalString;
import javax.xml.registry.infomodel.LocalizedString;
import javax.xml.registry.infomodel.RegistryObject;

import net.sf.csutils.core.model.ROMetaModel;
import net.sf.csutils.core.query.CsqlParser;
import net.sf.csutils.core.query.CsqlStatement;
import net.sf.csutils.core.query.StatementUtils;
import net.sf.csutils.core.registry.ConnectionProvider;
import net.sf.csutils.core.registry.RegistryFacade;
import net.sf.csutils.groovy.policy.GroovyEntityMetaClass;
import net.sf.csutils.groovy.proxy.ROMetaModelProxy;
import net.sf.csutils.groovy.proxy.ROModelAccessorProxy;
import net.sf.csutils.groovy.proxy.RegistryFacadeProxy;

import com.centrasite.jaxr.CentraSiteRegistryService;
import com.softwareag.centrasite.policy.model.GroovyEntity;


/**
 * The query engine is used to read registry objects.
 */
public abstract class QueryEngine implements GroovyObject {
	private MetaClass metaClass;

	public QueryEngine() {
		setMetaClass(GroovySystem.getMetaClassRegistry().getMetaClass(getClass()));
	}

	protected abstract ROMetaModel getMetaModel();
	protected abstract RegistryFacade getRegistryFacade();

	public MetaClass getMetaClass() {
		return metaClass;
	}

	public void setMetaClass(MetaClass metaClass) {
		this.metaClass = metaClass;
	}
	
	private String getQuery(Object pArgs, List<Object> pParameters) {
		if (pArgs != null) {
			if (pArgs.getClass().isArray()  &&
				!pArgs.getClass().getComponentType().isPrimitive()) {
				final Object[] args = (Object[]) pArgs;
				if (args.length > 0) {
					final String query = args[0] == null ? null : args[0].toString();
					for (int i = 1;  i < args.length;  i++) {
						pParameters.add(args[i]);
					}
					return query;
				}
			} else if (pArgs instanceof Collection) {
				final Collection<?> args = (Collection<?>) pArgs;
				if (!args.isEmpty()) {
					final Iterator<?> iter = args.iterator();
					final Object o = iter.next();
					final String query = o == null ? null : o.toString();
					while (iter.hasNext()) {
						pParameters.add(iter.next());
					}
					return query;
				}
			}
		}
		throw new IllegalArgumentException("Missing argument: query string.");
	}
	
	private CsqlStatement query(Object pArgs) throws JAXRException {
		final List<Object> parameters = new ArrayList<Object>();
		final String query = getQuery(pArgs, parameters);
		final CsqlStatement stmt = new CsqlParser(getMetaModel()).parse(getRegistryFacade(), query);
		StatementUtils.setParameters(stmt, parameters);
		return stmt;
	}
	
	public Object invokeMethod(String pName, Object pArgs) {
		try {
			if ("query".equals(pName)) {
				final CsqlStatement st = query(pArgs);
				final List<javax.xml.registry.infomodel.RegistryObject> list = st.executeQuery();
				final List<RegistryObject> result = new ArrayList<RegistryObject>(list.size());
				for (int i = 0;  i < list.size();  i++) {
				    result.add(list.get(i));
				}
				return result;
			} else if ("queryArray".equals(pName)) {
				final CsqlStatement st = query(pArgs);
				final List<javax.xml.registry.infomodel.RegistryObject[]> list = st.executeArrayQuery();
				final List<RegistryObject[]> result = new ArrayList<RegistryObject[]>(list.size());
				for (int i = 0;  i < list.size();  i++) {
					final javax.xml.registry.infomodel.RegistryObject[] row = list.get(i);
					final RegistryObject[] ros = new RegistryObject[row.length];
					for (int j = 0;  j < ros.length;  j++) {
					    ros[j] = row[j];
					}
					result.add(ros);
				}
				return result;
			} else if ("createQuery".equals(pName)) {
				return query(pArgs);
			} else {
				throw new IllegalArgumentException("Invalid method name: " + pName);
			}
		} catch (JAXRException e) {
			throw new UndeclaredThrowableException(e);
		}
	}

	public Object getProperty(String pPropertyName) {
		throw new IllegalArgumentException("Invalid property name: " + pPropertyName);
	}

	public void setProperty(String pPropertyName, Object pNewValue) {
		throw new IllegalArgumentException("Invalid property name: " + pPropertyName);
	}

    public static void run(Template pTemplate, Writer pWriter, ConnectionProvider pProvider) throws IOException {
        final Map<String,Object> map = new HashMap<String,Object>();
        run(map, pTemplate, pWriter, pProvider);
    }

    public static void run(Closure pClosure, RegistryObject pObject) throws JAXRException {
        final CentraSiteRegistryService rs = (CentraSiteRegistryService) pObject.getLifeCycleManager().getRegistryService();
        final Connection conn = rs.getConnection();
        final ConnectionProvider provider = new ConnectionProvider() {
            public Connection getConnection() throws JAXRException {
                return conn;
            }
        };
        run(pClosure, provider, false);
    }

    public static void run(Closure pClosure, ConnectionProvider pProvider, boolean pCloseConnection) {
        RegistryFacadeProxy registryFacadeProxy = new RegistryFacadeProxy(pProvider);
        try {
            final RegistryFacadeProxy rfProxy = registryFacadeProxy;
            final ROMetaModelProxy roMetaModelProxy = new ROMetaModelProxy(rfProxy);
            final MetaClassRegistry registry = GroovySystem.getMetaClassRegistry();
            registry.setMetaClassCreationHandle(new MetaClassRegistry.MetaClassCreationHandle(){
                @Override
                protected MetaClass createNormalMetaClass(@SuppressWarnings("rawtypes") Class theClass, MetaClassRegistry pRegistry) {
                    final MetaClass metaClass = super.createNormalMetaClass(theClass, registry);
                    if (LocalizedString.class.isAssignableFrom(theClass)) {
                        return new LocalizedStringMetaClass(metaClass);
                    }
                    if (InternationalString.class.isAssignableFrom(theClass)) {
                        return new InternationalStringMetaClass(metaClass);
                    }
                    if (GroovyEntity.class.isAssignableFrom(theClass)) {
                        return new GroovyEntityMetaClass(metaClass){
                            @Override
                            protected ROMetaModel getMetaModel() {
                                return roMetaModelProxy.getROMetaModel();
                            }
                        };
                    }
                    if (RegistryObject.class.isAssignableFrom(theClass)) {
                        return new RegistryObjectMetaClass(metaClass){
                            @Override
                            protected ROMetaModel getMetaModel() {
                                return roMetaModelProxy.getROMetaModel();
                            }
                        };
                    }
                    return metaClass;
                }
            });
            final MetaClass groovyEntityMetaClass = registry.getMetaClass(GroovyEntity.class);
            if (!(groovyEntityMetaClass instanceof GroovyEntityMetaClass)) {
                registry.setMetaClass(GroovyEntity.class, registry.getMetaClassCreationHandler().create(GroovyEntityMetaClass.class, registry));
            }
            final QueryEngine queryEngine = new QueryEngine(){
                @Override
                protected RegistryFacade getRegistryFacade() {
                    return rfProxy.getRegistryFacade();
                }
                @Override
                protected ROMetaModel getMetaModel() {
                    return roMetaModelProxy.getROMetaModel();
                }
            };
            pClosure.call(new Object[]{
                registryFacadeProxy, roMetaModelProxy, queryEngine
            });
            if (pCloseConnection) {
                registryFacadeProxy.close();
            }
            registryFacadeProxy = null;
        } finally {
            if (pCloseConnection && registryFacadeProxy != null) {
                try { registryFacadeProxy.close(); } catch (Throwable t) { /* Ignore me */ }
            }
        }
    }
    
    public static void run(Map<String,Object> pMap, Template pTemplate, Writer pWriter, ConnectionProvider pProvider) throws IOException {
	    RegistryFacadeProxy registryFacadeProxy = new RegistryFacadeProxy(pProvider);
	    try {
    	    final RegistryFacadeProxy rfProxy = registryFacadeProxy;
    	    pMap.put("registryFacade", rfProxy);
            final ROMetaModelProxy roMetaModelProxy = new ROMetaModelProxy(rfProxy);
            pMap.put("registryMetaModel", roMetaModelProxy);
            final ROModelAccessorProxy modelAccProxy = new ROModelAccessorProxy(rfProxy);
            pMap.put("modelAccessor", modelAccProxy);
            final ROModelAccessorProxy metaModelAccProxy = new ROModelAccessorProxy(rfProxy);
            pMap.put("metaModelAccessor", metaModelAccProxy);
            final MetaClassRegistry registry = GroovySystem.getMetaClassRegistry();
            registry.setMetaClassCreationHandle(new MetaClassRegistry.MetaClassCreationHandle(){
                @Override
                protected MetaClass createNormalMetaClass(@SuppressWarnings("rawtypes") Class theClass, MetaClassRegistry pRegistry) {
                    final MetaClass metaClass = super.createNormalMetaClass(theClass, registry);
                    if (LocalizedString.class.isAssignableFrom(theClass)) {
                        return new LocalizedStringMetaClass(metaClass);
                    }
                    if (InternationalString.class.isAssignableFrom(theClass)) {
                        return new InternationalStringMetaClass(metaClass);
                    }
                    if (RegistryObject.class.isAssignableFrom(theClass)) {
                        return new RegistryObjectMetaClass(metaClass){
                            @Override
                            protected ROMetaModel getMetaModel() {
                                return roMetaModelProxy.getROMetaModel();
                            }
                        };
                    }
                    return metaClass;
                }
                
            });
            final QueryEngine queryEngine = new QueryEngine(){
                @Override
                protected RegistryFacade getRegistryFacade() {
                    return rfProxy.getRegistryFacade();
                }
                @Override
                protected ROMetaModel getMetaModel() {
                    return roMetaModelProxy.getROMetaModel();
                }
            };
            pMap.put("queryEngine", queryEngine);
            pTemplate.make(pMap).writeTo(pWriter);
            registryFacadeProxy.close();
            registryFacadeProxy = null;
	    } finally {
	        if (registryFacadeProxy != null) { try { registryFacadeProxy.close(); } catch (Throwable t) { /* Ignore me */ } }
	    }
	}
}
