package net.sf.csutils.groovy;

import groovy.lang.DelegatingMetaClass;
import groovy.lang.MetaClass;

import java.lang.reflect.UndeclaredThrowableException;
import java.math.BigDecimal;
import java.text.Format;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;

import javax.xml.registry.BusinessLifeCycleManager;
import javax.xml.registry.JAXRException;
import javax.xml.registry.infomodel.Association;
import javax.xml.registry.infomodel.Classification;
import javax.xml.registry.infomodel.Concept;
import javax.xml.registry.infomodel.ExternalLink;
import javax.xml.registry.infomodel.InternationalString;
import javax.xml.registry.infomodel.RegistryEntry;
import javax.xml.registry.infomodel.RegistryObject;
import javax.xml.registry.infomodel.Slot;
import javax.xml.registry.infomodel.URIValidator;
import javax.xml.registry.infomodel.Versionable;

import net.sf.csutils.core.model.QName;
import net.sf.csutils.core.model.ROAttribute;
import net.sf.csutils.core.model.ROClassification;
import net.sf.csutils.core.model.ROFile;
import net.sf.csutils.core.model.ROMetaModel;
import net.sf.csutils.core.model.RORelation;
import net.sf.csutils.core.model.ROType;
import net.sf.csutils.core.registry.centrasite.CentraSiteRegistryInfo;
import net.sf.csutils.core.utils.Generics;
import net.sf.csutils.core.utils.RegistryObjects;

import org.apache.ws.commons.util.XsDateFormat;
import org.apache.ws.commons.util.XsDateTimeFormat;
import org.apache.ws.commons.util.XsTimeFormat;
import org.codehaus.groovy.runtime.InvokerHelper;

import com.centrasite.jaxr.infomodel.CentraSiteRegistryObject;
import com.centrasite.jaxr.type.CentraSiteSlotType;


/**
 * Meta class for {@link RegistryObject}.
 */
public abstract class RegistryObjectMetaClass extends DelegatingMetaClass {
    private static final TimeZone GMT = TimeZone.getTimeZone("GMT");
    private static final XsDateTimeFormat dateTimeFormat = new XsDateTimeFormat();
    private static final XsDateFormat dateFormat = new XsDateFormat();
    private static final XsTimeFormat timeFormat = new XsTimeFormat();

    protected abstract ROMetaModel getMetaModel();
    
    /**
     * Creates a new instance.
     * @param pDelegate The proxied meta class.
     */
    public RegistryObjectMetaClass(MetaClass pDelegate) {
        super(pDelegate);
    }

    private ROType getType(javax.xml.registry.infomodel.RegistryObject pRegistryObject) throws JAXRException {
        final QName qName = QName.valueOf(getObjectType(pRegistryObject).getValue());
        final ROType type = getMetaModel().getROType(qName);
        if (type == null) {
            throw new IllegalArgumentException("Unknown registry object type: " + qName);
        }
        return type;
    }

    @Override
    public void setProperty(Object pObject, String pProperty, Object pValues) {
        try {
            final RegistryObject ro = (RegistryObject) pObject;
            final ROAttribute attr = getType(ro).getAttribute(pProperty);
            if (attr == null) {
                setStringSlot(ro, pProperty, pValues, CentraSiteSlotType.XS_STRING);
            } else {
                switch (attr.getType()) {
                    case bool:
                        setBooleanSlot(ro, pProperty, pValues);
                        break;
                    case date:
                        setDateSlot(ro, pProperty, pValues);
                        break;
                    case dateTime:
                        setDateTimeSlot(ro, pProperty, pValues);
                        break;
                    case duration:
                        setDurationSlot(ro, pProperty, pValues);
                        break;
                    case emailAddress:
                        setStringSlot(ro, pProperty, pValues, CentraSiteSlotType.EMAILADDRESS);
                        break;
                    case integer:
                        setIntegerSlot(ro, pProperty, pValues);
                        break;
                    case ipv4Address:
                        setStringSlot(ro, pProperty, pValues, CentraSiteSlotType.IPV4);
                        break;
                    case ipv6Address:
                        setStringSlot(ro, pProperty, pValues, CentraSiteSlotType.IPV6);
                        break;
                    case longInteger:
                        setLongSlot(ro, pProperty, pValues);
                        break;
                    case number:
                        setNumberSlot(ro, pProperty, pValues);
                        break;
                    case relation:
                        setRelationSlot((RORelation) attr, ro, pProperty, pValues);
                        break;
                    case string:
                        setStringSlot(ro, pProperty, pValues, CentraSiteSlotType.XS_STRING);
                        break;
                    case uri:
                        setStringSlot(ro, pProperty, pValues, CentraSiteSlotType.XS_ANYURI);
                        break;
                    case time:
                        setTimeSlot(ro, pProperty, pValues);
                        break;
                    default:
                        throw new IllegalStateException("Invalid slot type: " + attr.getType());
                }
            }
        } catch (JAXRException e) {
            throw new UndeclaredThrowableException(e);
        }
    }

    private void setStringSlot(final RegistryObject pObject, String pProperty,
            Object pValues, String pType) throws JAXRException {
        final List<?> objectList = InvokerHelper.asList(pValues);
        final List<String> stringList = new ArrayList<String>(objectList.size());
        for (Object o : objectList) {
            stringList.add(String.valueOf(o));
        }
        setSlot(pProperty, pObject, stringList, pType);
    }

    private void setBooleanSlot(RegistryObject pObject, String pProperty, Object pValues) throws JAXRException {
        final List<?> objectList = InvokerHelper.asList(pValues);
        final List<String> stringList = new ArrayList<String>(objectList.size());
        for (Object o : objectList) {
            final String s;
            if (o == null) {
                s = Boolean.FALSE.toString();
            } else if (o instanceof Boolean) {
                s = o.toString();
            } else {
                s = String.valueOf(Boolean.parseBoolean(o.toString()));
            }
            stringList.add(s);
        }
        setSlot(pProperty, pObject, stringList, CentraSiteSlotType.XS_BOOLEAN);
    }

    private void setIntegerSlot(RegistryObject pObject, String pProperty, Object pValues) throws JAXRException {
        final List<?> objectList = InvokerHelper.asList(pValues);
        final List<String> stringList = new ArrayList<String>(objectList.size());
        for (Object o : objectList) {
            final String s;
            if (o == null) {
                s = "0";
            } else if (o instanceof Number) {
                s = String.valueOf(((Number) o).intValue());
            } else {
                try {
                    s = String.valueOf(Integer.parseInt(String.valueOf(o)));
                } catch (Exception e) {
                    throw new IllegalArgumentException("Invalid integer value for slot "
                            + pProperty + ": " + o);
                }
            }
            stringList.add(s);
        }
        setSlot(pProperty, pObject, stringList, CentraSiteSlotType.XS_INTEGER);
    }

    private void setNumberSlot(RegistryObject pObject, String pProperty, Object pValues) throws JAXRException {
        final List<?> objectList = InvokerHelper.asList(pValues);
        final List<String> stringList = new ArrayList<String>(objectList.size());
        for (Object o : objectList) {
            final String s;
            if (o == null) {
                s = "0";
            } else if (o instanceof Number) {
                s = String.valueOf(((Number) o).doubleValue());
            } else {
                try {
                    s = String.valueOf(Double.parseDouble(String.valueOf(o)));
                } catch (Exception e) {
                    throw new IllegalArgumentException("Invalid number value for slot "
                            + pProperty + ": " + o);
                }
            }
            stringList.add(s);
        }
        setSlot(pProperty, pObject, stringList, CentraSiteSlotType.XS_DOUBLE);
    }

    private void setLongSlot(RegistryObject pObject, String pProperty, Object pValues) throws JAXRException {
        final List<?> objectList = InvokerHelper.asList(pValues);
        final List<String> stringList = new ArrayList<String>(objectList.size());
        for (Object o : objectList) {
            final String s;
            if (o == null) {
                s = "0";
            } else if (o instanceof Number) {
                s = String.valueOf(((Number) o).longValue());
            } else {
                try {
                    s = String.valueOf(Long.parseLong(String.valueOf(o)));
                } catch (Exception e) {
                    throw new IllegalArgumentException("Invalid long value for slot "
                            + pProperty + ": " + o);
                }
            }
            stringList.add(s);
        }
        setSlot(pProperty, pObject, stringList, CentraSiteSlotType.XS_INTEGER);
    }


    private void setDateSlot(RegistryObject pObject, String pProperty, Object pValues) throws JAXRException {
        setDateTimeSlot(pObject, pProperty, pValues, dateFormat, CentraSiteSlotType.XS_DATE);
    }

    private void setDateTimeSlot(RegistryObject pObject, String pProperty, Object pValues) throws JAXRException {
        setDateTimeSlot(pObject, pProperty, pValues, dateTimeFormat, CentraSiteSlotType.XS_DATETIME);
    }

    private void setTimeSlot(RegistryObject pObject, String pProperty, Object pValues) throws JAXRException {
        setDateTimeSlot(pObject, pProperty, pValues, timeFormat, CentraSiteSlotType.XS_TIME);
    }

    private void setRelationSlot(RORelation pRelation, RegistryObject pObject, String pProperty, Object pValues) throws JAXRException {
        throw new IllegalStateException("Relations not yet supported.");
    }

    private void setDurationSlot(RegistryObject pObject, String pProperty, Object pValues) throws JAXRException {
        throw new IllegalStateException("Durations not implemented.");
    }

    private void setDateTimeSlot(RegistryObject pObject, String pProperty, Object pValues, Format pFormat, String pType) throws JAXRException {
        final List<?> objectList = InvokerHelper.asList(pValues);
        final List<String> stringList = new ArrayList<String>(objectList.size());
        for (Object o : objectList) {
            final String s;
            if (o == null) {
                final Calendar cal = Calendar.getInstance(GMT);
                cal.setTimeInMillis(0);
                s = pFormat.format(cal);
            } else if (o instanceof Calendar) {
                s = pFormat.format(o);
            } else if (o instanceof Date) {
                final Calendar cal = Calendar.getInstance(GMT);
                cal.setTime((Date) o);
                s = pFormat.format(cal);
            } else {
                try {
                    s = pFormat.format(pFormat.parseObject(o.toString()));
                } catch (Exception e) {
                    throw new IllegalArgumentException("Invalid " + pType + " value for slot "
                            + pProperty + ": " + o);
                }
            }
            stringList.add(s);
        }
        setSlot(pProperty, pObject, stringList, pType);
    }


    @Override
    public Object getProperty(Object pObject, String pProperty) {
        try {
            final RegistryObject ro = (RegistryObject) pObject;
            final ROAttribute attr = getType(ro).getAttribute(pProperty);
            return getAttributeValue(ro, attr, pProperty);
        } catch (JAXRException e) {
            throw new UndeclaredThrowableException(e);
        } catch (ParseException e) {
            throw new UndeclaredThrowableException(e);
        }
    }

    private Object getAttributeValue(final RegistryObject pObject,
            final ROAttribute pAttr, String pProperty) throws JAXRException,
            ParseException {
        if (pAttr == null) {
            final Slot slot = getSlot(pProperty, pObject);
            if (slot == null) {
                return null;
            }
            return asStringArray(slot);
        }
        final boolean isMultiple = pAttr.getMaxOccurs() == -1  ||  pAttr.getMaxOccurs() > 1;
        switch (pAttr.getType()) {
        case bool:
            return isMultiple ?
                    asBooleanArray(getSlot(pProperty, pObject))
                    : asBoolean(getSlot(pProperty, pObject));
        case date:
            return isMultiple ?
                    asDateArray(getSlot(pProperty, pObject))
                    : asDate(getSlot(pProperty, pObject));
        case dateTime:
            return isMultiple ?
                    asDateTimeArray(getSlot(pProperty, pObject))
                    : asDateTime(getSlot(pProperty, pObject));
        case time:
            return isMultiple ?
                    asTimeArray(getSlot(pProperty, pObject))
                    : asTime(getSlot(pProperty, pObject));
        case duration:
            throw new IllegalStateException("Durations are not yet supported.");
        case emailAddress:
        case ipv4Address:
        case ipv6Address:
        case string:
        case uri:
            return isMultiple ?
                    asStringArray(getSlot(pProperty, pObject))
                    : asString(getSlot(pProperty, pObject));
        case integer:
            return isMultiple ?
                    asIntegerArray(getSlot(pProperty, pObject))
                    : asInteger(getSlot(pProperty, pObject));
        case longInteger:
            return isMultiple ?
                    asLongArray(getSlot(pProperty, pObject))
                    : asLong(getSlot(pProperty, pObject));
        case number:
            return isMultiple ?
                    asBigDecimalArray(getSlot(pProperty, pObject))
                    : asBigDecimal(getSlot(pProperty, pObject));
        case relation:
            return isMultiple ?
                    asTargetArray(pObject, (RORelation) pAttr)
                    : asTarget(pObject, (RORelation) pAttr);
        case classification:
            return isMultiple ?
                    asClassificationArray(pObject, (ROClassification) pAttr)
                    : asClassification(pObject, (ROClassification) pAttr);
        case file:
            return isMultiple ?
                    asFileArray(pObject, (ROFile) pAttr)
                    : asFile(pObject, (ROFile) pAttr);
        default:
            throw new IllegalStateException("Unsupported attribute type: " + pAttr.getType());
        }
    }

    private Slot getSlot(String pProperty, final RegistryObject pObject)
            throws JAXRException {
        return pObject.getSlot(getSlotName(pProperty, pObject));
    }

    private void setSlot(String pProperty, RegistryObject pObject, Collection<String> pValues, String pType) throws JAXRException {
        final String slotName = getSlotName(pProperty, pObject);
        final Slot slot = pObject.getSlot(slotName);
        if (slot == null) {
            final BusinessLifeCycleManager blcm = ((CentraSiteRegistryObject) pObject).getRegistryService().getBusinessLifeCycleManager();
            pObject.addSlot(blcm.createSlot(slotName, pValues, pType));
        } else {
            slot.setValues(pValues);
            slot.setSlotType(pType);
        }
    }
    
    private String getSlotName(String pProperty, final RegistryObject pObject)
            throws JAXRException {
        final String objectType = getObjectType(pObject).getValue();
        int offset = objectType.indexOf('}');
        final String slotName;
        if (offset == -1) {
            slotName = pProperty;
        } else {
            slotName = objectType.substring(0, offset+1) + pProperty;
        }
        return slotName;
    }

    private String asString(final Slot pSlot) throws JAXRException {
        if (pSlot == null) {
            return null;
        }
        final Collection<?> values = pSlot.getValues();
        if (values == null  ||  values.isEmpty()) {
            return null;
        }
        final Object o = values.iterator().next();
        return o == null ? null : o.toString();
    }

    private String[] asStringArray(final Slot pSlot) throws JAXRException {
        if (pSlot == null) {
            return new String[0];
        }
        final Collection<?> values = pSlot.getValues();
        final String[] result = new String[values.size()];
        int i = 0;
        for (Iterator<?> iter = values.iterator();  iter.hasNext();  ) {
            Object o = iter.next();
            result[i++] = o == null ? null : o.toString();
        }
        return result;
    }

    private Integer asInteger(final Slot pSlot) throws JAXRException {
        if (pSlot == null) {
            return null;
        }
        final Collection<?> values = pSlot.getValues();
        if (values == null  ||  values.isEmpty()) {
            return null;
        }
        final Object o = values.iterator().next();
        return o == null ? null : Integer.valueOf(o.toString());
    }

    private Integer[] asIntegerArray(final Slot pSlot) throws JAXRException {
        if (pSlot == null) {
            return new Integer[0];
        }
        final Collection<?> values = pSlot.getValues();
        final Integer[] result = new Integer[values.size()];
        int i = 0;
        for (Iterator<?> iter = values.iterator();  iter.hasNext();  ) {
            Object o = iter.next();
            result[i++] = o == null ? null : Integer.valueOf(o.toString());
        }
        return result;
    }

    private BigDecimal asBigDecimal(final Slot pSlot) throws JAXRException {
        if (pSlot == null) {
            return null;
        }
        final Collection<?> values = pSlot.getValues();
        if (values == null  ||  values.isEmpty()) {
            return null;
        }
        final Object o = values.iterator().next();
        return o == null ? null : new BigDecimal(o.toString());
    }

    private BigDecimal[] asBigDecimalArray(final Slot pSlot) throws JAXRException {
        if (pSlot == null) {
            return new BigDecimal[0];
        }
        final Collection<?> values = pSlot.getValues();
        final BigDecimal[] result = new BigDecimal[values.size()];
        int i = 0;
        for (Iterator<?> iter = values.iterator();  iter.hasNext();  ) {
            Object o = iter.next();
            result[i++] = o == null ? null : new BigDecimal(o.toString());
        }
        return result;
    }

    private Long asLong(final Slot pSlot) throws JAXRException {
        if (pSlot == null) {
            return null;
        }
        final Collection<?> values = pSlot.getValues();
        if (values == null  ||  values.isEmpty()) {
            return null;
        }
        final Object o = values.iterator().next();
        return o == null ? null : Long.valueOf(o.toString());
    }

    private Long[] asLongArray(final Slot pSlot) throws JAXRException {
        if (pSlot == null) {
            return new Long[0];
        }
        final Collection<?> values = pSlot.getValues();
        final Long[] result = new Long[values.size()];
        int i = 0;
        for (Iterator<?> iter = values.iterator();  iter.hasNext();  ) {
            Object o = iter.next();
            result[i++] = o == null ? null : Long.valueOf(o.toString());
        }
        return result;
    }

    private Boolean asBoolean(final Slot pSlot) throws JAXRException {
        if (pSlot == null) {
            return null;
        }
        final Collection<?> values = pSlot.getValues();
        if (values == null  ||  values.isEmpty()) {
            return null;
        }
        final Object o = values.iterator().next();
        return o == null ? null : Boolean.valueOf(o.toString());
    }

    private Boolean[] asBooleanArray(final Slot pSlot) throws JAXRException {
        if (pSlot == null) {
            return new Boolean[0];
        }
        final Collection<?> values = pSlot.getValues();
        final Boolean[] result = new Boolean[values.size()];
        int i = 0;
        for (Iterator<?> iter = values.iterator();  iter.hasNext();  ) {
            Object o = iter.next();
            result[i++] = o == null ? null : Boolean.valueOf(o.toString());
        }
        return result;
    }

    private ExternalLink[] asFileArray(RegistryObject pObject, ROFile pAttr) throws JAXRException {
        final List<ExternalLink> list = asFileList(pObject, pAttr);
        return list.toArray(new ExternalLink[list.size()]);
    }

    private ExternalLink asFile(RegistryObject pObject, ROFile pAttr) throws JAXRException {
        final List<ExternalLink> list = asFileList(pObject, pAttr);
        return list.isEmpty() ? null : list.get(0);
    }

    private List<ExternalLink> asFileList(RegistryObject pObject, ROFile pAttr) throws JAXRException {
        final Collection<ExternalLink> els = Generics.cast(pObject.getExternalLinks());
        final List<ExternalLink> list = new ArrayList<ExternalLink>();
        for (ExternalLink el : els) {
            final Slot slot = el.getSlot(CentraSiteRegistryInfo.SLOT_NAME_ATTRIBUTES_KEY);
            if (slot != null) {
                Collection<?> values = slot.getValues();
                if (values != null) {
                    final Iterator<?> iter = values.iterator();
                    if (iter.hasNext()  &&  iter.next().equals(pAttr.getAttributeKey())) {
                        list.add(el);
                    }
                }
            }
        }
        return list;
    }

    private Concept asClassification(RegistryObject pObject, ROClassification pAttr) throws JAXRException {
        final List<Concept> list = asClassificationList(pObject, pAttr);
        return list == null || list.isEmpty() ? null : list.get(0);
    }

    private Concept[] asClassificationArray(RegistryObject pObject, ROClassification pAttr) throws JAXRException {
        final List<Concept> list = asClassificationList(pObject, pAttr);
        return list.toArray(new Concept[list.size()]);
    }

    private List<Concept> asClassificationList(RegistryObject pObject, ROClassification pAttr) throws JAXRException {
        final Collection<Classification> cls = Generics.cast(pObject.getClassifications());
        final List<Concept> list = new ArrayList<Concept>();
        for (Classification cl : cls) {
            final Slot slot = cl.getSlot(CentraSiteRegistryInfo.SLOT_NAME_ATTRIBUTES_KEY);
            if (slot != null) {
                Collection<?> values = slot.getValues();
                if (values != null) {
                    final Iterator<?> iter = values.iterator();
                    if (iter.hasNext()  &&  iter.next().equals(pAttr.getAttributeKey())) {
                        list.add(cl.getConcept());
                    }
                }
            }
        }
        return list;
    }

    private RegistryObject asTarget(RegistryObject pObject, RORelation pAttr) throws JAXRException {
        final List<RegistryObject> list = asTargetList(pObject, pAttr);
        return list == null || list.isEmpty() ? null : list.get(0);
    }
    
    private RegistryObject[] asTargetArray(RegistryObject pObject, RORelation pAttr) throws JAXRException {
        final List<RegistryObject> list = asTargetList(pObject, pAttr);
        return list.toArray(new RegistryObject[list.size()]);
    }

    private List<RegistryObject> asTargetList(RegistryObject pObject, RORelation pAttr) throws JAXRException {
        final Collection<Association> assocs = Generics.cast(pObject.getAssociations());
        final List<RegistryObject> list = new ArrayList<RegistryObject>();
        for (Association assoc : assocs) {
            final Slot slot = assoc.getSlot(CentraSiteRegistryInfo.SLOT_NAME_ATTRIBUTES_KEY);
            if (slot != null) {
                Collection<?> values = slot.getValues();
                if (values != null) {
                    final Iterator<?> iter = values.iterator();
                    if (iter.hasNext()  &&  iter.next().equals(pAttr.getAttributeKey())) {
                        list.add(assoc.getTargetObject());
                    }
                }
            }
        }
        return list;
    }
    
    private Calendar asDate(final Slot pSlot) throws JAXRException, ParseException {
        return asCalendar(pSlot, dateFormat);
    }

    private Calendar[] asDateArray(final Slot pSlot) throws JAXRException, ParseException {
        return asCalendarArray(pSlot, dateFormat);
    }

    private Calendar asDateTime(final Slot pSlot) throws JAXRException, ParseException {
        return asCalendar(pSlot, dateTimeFormat);
    }

    private Calendar[] asDateTimeArray(final Slot pSlot) throws JAXRException, ParseException {
        return asCalendarArray(pSlot, dateTimeFormat);
    }

    private Calendar asTime(final Slot pSlot) throws JAXRException, ParseException {
        return asCalendar(pSlot, timeFormat);
    }

    private Calendar[] asTimeArray(final Slot pSlot) throws JAXRException, ParseException {
        return asCalendarArray(pSlot, timeFormat);
    }

    private Calendar asCalendar(final Slot pSlot, Format pFormat) throws JAXRException, ParseException {
        if (pSlot == null) {
            return null;
        }
        final Collection<?> values = pSlot.getValues();
        if (values == null  ||  values.isEmpty()) {
            return null;
        }
        final Object o = values.iterator().next();
        return o == null ? null : (Calendar) pFormat.parseObject(o.toString());
    }

    private Calendar[] asCalendarArray(final Slot pSlot, Format pFormat) throws JAXRException, ParseException {
        if (pSlot == null) {
            return new Calendar[0];
        }
        final Collection<?> values = pSlot.getValues();
        final Calendar[] result = new Calendar[values.size()];
        int i = 0;
        for (Iterator<?> iter = values.iterator();  iter.hasNext();  ) {
            Object o = iter.next();
            result[i++] = o == null ? null : (Calendar) pFormat.parseObject(o.toString());
        }
        return result;
    }

    private String getKey(RegistryObject ro) throws JAXRException {
        return ro.getKey().getId();
    }

    private String getName(RegistryObject ro, Locale pLocale) throws JAXRException {
        return pLocale == null ?
                RegistryObjects.getName(ro)
                : RegistryObjects.getName(ro, pLocale);
    }

    private InternationalString getNames(RegistryObject ro) throws JAXRException {
        return ro.getName();
    }

    private String getDescription(RegistryObject ro, Locale pLocale) throws JAXRException {
        return pLocale == null ?
                RegistryObjects.getDescription(ro)
                : RegistryObjects.getDescription(ro, pLocale);
    }

    private InternationalString getDescriptions(RegistryObject ro) throws JAXRException {
        return ro.getDescription();
    }

    private Integer getStatus(RegistryObject ro) throws JAXRException {
        return (ro instanceof RegistryEntry) ?
                Integer.valueOf(((RegistryEntry) ro).getStatus()) : null;
    }

    private String getStatusName(RegistryObject ro) throws JAXRException {
        if (ro instanceof RegistryEntry) {
            final int status = ((RegistryEntry) ro).getStatus();
            switch (status) {
                case RegistryEntry.STATUS_APPROVED:
                    return "APPROVED";
                case RegistryEntry.STATUS_DEPRECATED:
                    return "DEPRECATED";
                case RegistryEntry.STATUS_SUBMITTED:
                    return "SUBMITTED";
                case RegistryEntry.STATUS_WITHDRAWN:
                    return "WITHDRAWN";
                default:
                    return String.valueOf(status);
            }
        }
        return null;
    }

    private Integer getStability(RegistryObject ro) throws JAXRException {
        return (ro instanceof RegistryEntry) ?
                Integer.valueOf(((RegistryEntry) ro).getStability()) : null;
    }

    private String getStabilityName(RegistryObject ro) throws JAXRException {
        if (ro instanceof RegistryEntry) {
            final int stability = ((RegistryEntry) ro).getStability();
            switch (stability) {
                case RegistryEntry.STABILITY_DYNAMIC:
                    return "DYNAMIC";
                case RegistryEntry.STABILITY_DYNAMIC_COMPATIBLE:
                    return "DYNAMIC_COMPATIBLE";
                case RegistryEntry.STABILITY_STATIC:
                    return "STATIC";
                default:
                    return String.valueOf(stability);
            }
        }
        return null;
    }

    private Integer getMajorVersion(RegistryObject ro) throws JAXRException {
        return (ro instanceof Versionable) ?
                Integer.valueOf(((Versionable) ro).getMajorVersion()) : null;
    }

    private Integer getMinorVersion(RegistryObject ro) throws JAXRException {
        return (ro instanceof Versionable) ?
                Integer.valueOf(((Versionable) ro).getMinorVersion()) : null;
    }

    private String getUserVersion(RegistryObject ro) throws JAXRException {
        return (ro instanceof Versionable) ?
                ((Versionable) ro).getUserVersion() : null;
    }

    private Concept getObjectType(final RegistryObject pRo) throws JAXRException {
        return pRo.getObjectType();
    }

    private Object getValue(final RegistryObject pRo) throws JAXRException {
        return (pRo instanceof Concept) ? ((Concept) pRo).getValue() : null;
    }

    private Boolean getValidateURI(final RegistryObject ro)
            throws JAXRException {
        return (ro instanceof URIValidator) ? Boolean.valueOf(((URIValidator) ro).getValidateURI()) : null;
    }

    private String getExternalURI(final RegistryObject pRo)
            throws JAXRException {
        return (pRo instanceof ExternalLink) ? ((ExternalLink) pRo).getExternalURI() : null;
    }

    @Override
    public Object invokeMethod(Object pObject, String pMethodName, Object[] pArgs) {
        final RegistryObject ro = (RegistryObject) pObject;
        try {
            if (pMethodName.equals("attribute")) {
                final String propertyName = GroovyObjects.checkStringArg(pMethodName, pArgs);
                return getProperty(pObject, propertyName);
            }
            if (pMethodName.equals("key")) {
                GroovyObjects.checkNoArgs(pMethodName, pArgs);
                return getKey(ro);
            }
            if (pMethodName.equals("name")) {
                return getName(ro, GroovyObjects.getLocaleArg(pMethodName, pArgs));
            }
            if (pMethodName.equals("names")) {
                GroovyObjects.checkNoArgs(pMethodName, pArgs);
                return getNames(ro);
            }
            if (pMethodName.equals("description")) {
                return getDescription(ro, GroovyObjects.getLocaleArg(pMethodName, pArgs));
            }
            if (pMethodName.equals("descriptions")) {
                GroovyObjects.checkNoArgs(pMethodName, pArgs);
                return getDescriptions(ro);
            }
            if (pMethodName.equals("status")) {
                GroovyObjects.checkNoArgs(pMethodName, pArgs);
                return getStatusName(ro);
            }
            if (pMethodName.equals("statusInt")) {
                GroovyObjects.checkNoArgs(pMethodName, pArgs);
                return getStatus(ro);
            }
            if (pMethodName.equals("stability")) {
                GroovyObjects.checkNoArgs(pMethodName, pArgs);
                return getStabilityName(ro);
            }
            if (pMethodName.equals("stabilityInt")) {
                GroovyObjects.checkNoArgs(pMethodName, pArgs);
                return getStability(ro);
            }
            if (pMethodName.equals("minorVersion")) {
                GroovyObjects.checkNoArgs(pMethodName, pArgs);
                return getMinorVersion(ro);
            }
            if (pMethodName.equals("majorVersion")) {
                GroovyObjects.checkNoArgs(pMethodName, pArgs);
                return getMajorVersion(ro);
            }
            if (pMethodName.equals("userVersion")) {
                GroovyObjects.checkNoArgs(pMethodName, pArgs);
                return getUserVersion(ro);
            }
            if (pMethodName.equals("objectType")) {
                GroovyObjects.checkNoArgs(pMethodName, pArgs);
                return getObjectType(ro);
            }
            if (pMethodName.equals("value")) {
                GroovyObjects.checkNoArgs(pMethodName, pArgs);
                return getValue(ro);
            }
            if (pMethodName.equals("externalURI")) {
                GroovyObjects.checkNoArgs(pMethodName, pArgs);
                return getExternalURI(ro);
            }
            if (pMethodName.equals("validateURI")) {
                GroovyObjects.checkNoArgs(pMethodName, pArgs);
                return getValidateURI(ro);
            }
            return super.invokeMethod(pObject, pMethodName, pArgs);
        } catch (JAXRException e) {
            throw new UndeclaredThrowableException(e);
        }
    }
}
