package net.sf.csutils.groovy.xml;

import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamConstants;
import javax.xml.stream.XMLStreamException;

import net.sf.csutils.groovy.xml.XmlTemplateEngineStreamReader.Event;
import net.sf.csutils.groovy.xml.XmlTemplateEngineStreamReader.StartElementEvent;
import net.sf.csutils.groovy.xml.XmlTemplateEngineStreamReader.TextEvent;

import org.xml.sax.helpers.AttributesImpl;


public class XmlTemplateEngineParser {
	private static final QName ATTR_NAME = new QName(XmlTemplateEngine.NS_URI, "name");
	private static final QName ATTR_SIMPLE_NAME = new QName("name");
	private static final QName ATTR_SIMPLE_VALUE = new QName("value");
	private final XmlTemplateEngineStreamReader xsr;
	private final Writer writer;

	public XmlTemplateEngineParser(Reader pReader, Writer pWriter) throws XMLStreamException {
		writer = pWriter;
		xsr = new XmlTemplateEngineStreamReader(pReader);
	}

	private void write(String pValue) throws IOException {
		writer.write(pValue);
	}

	private void writeEscaped(String pValue) throws IOException {
		writer.write('"');
		for (int i = 0;  i < pValue.length();  i++) {
			char c = pValue.charAt(i);
			switch(c) {
			case '"':
				writer.write("\\\"");
				break;
			case '\\':
				writer.write("\\\\");
				break;
			case '\n':
				writer.write("\\n");
				break;
			case '\r':
				writer.write("\\r");
				break;
			case '\t':
				writer.write("\\t");
				break;
			default:
				writer.write(c);
				break;
			}
		}
		writer.write('"');
	}
	
	private void createHeader() throws IOException {
		write("/* Generated by ");
		write(XmlTemplateEngine.class.getName());
		write(" */\n");
		write("import ");
		write(AttributesImpl.class.getName());
		write("\ndef xsw = new ");
		write(XmlTemplateEngineStreamWriter.class.getName());
		write("(out)\n");
		write("def attrs = null\n");
		write("def buffer = null\n");
		write("def qName = null\n");
	}

	private void createFooter() {
		// Nothing to do
	}

	public void parse() throws XMLStreamException, IOException {
		createHeader();
		parseElement();
		createFooter();
	}

	private boolean isWhitespace(TextEvent pEvent) {
		final String data = pEvent.getText();
		for (int i = 0;  i < data.length();  i++) {
			if (!Character.isWhitespace(data.charAt(i))) {
				return false; 
			}
		}
		return true;
	}

	private boolean isAttributeEvent(Event pEvent) {
		switch(pEvent.getEventType()) {
		  case XMLStreamConstants.SPACE:
		  case XMLStreamConstants.COMMENT:
			return true;
		  case XMLStreamConstants.CDATA:
		  case XMLStreamConstants.CHARACTERS:
			return isWhitespace((TextEvent) pEvent);
		  case XMLStreamConstants.START_ELEMENT:
			final StartElementEvent startElementEvent = (StartElementEvent) pEvent;
			final QName qName = startElementEvent.getName();
			return XmlTemplateEngine.NS_URI.equals(qName.getNamespaceURI())
				&&  "scriptlet".equals(qName.getLocalPart());
	      default:
		    	return false;
		}
	}
	
	private void applyAttributeEvent(Event pEvent) throws XMLStreamException, IOException {
		switch(pEvent.getEventType()) {
	      default:
			throw new XMLStreamException("Invalid event type: " + pEvent.getEventType());
		  case XMLStreamConstants.CDATA:
		  case XMLStreamConstants.CHARACTERS:
		  case XMLStreamConstants.SPACE:
			handleCharacters(pEvent.getEventType(), ((TextEvent) pEvent).getText());
			break;
		  case XMLStreamConstants.START_ELEMENT:
			final StartElementEvent startElementEvent = (StartElementEvent) pEvent;
			handleStartElement(startElementEvent);
			break;
		}
	}

	private String getElementText() throws XMLStreamException {
		final StringBuilder sb = new StringBuilder();
		while (xsr.hasNext()) {
			int type = xsr.next();
			switch (type) {
				case XMLStreamConstants.END_ELEMENT:
					return sb.toString();
				case XMLStreamConstants.CHARACTERS:
				case XMLStreamConstants.CDATA:
				case XMLStreamConstants.SPACE:
					sb.append(xsr.getText());
					break;
				case XMLStreamConstants.COMMENT:
				case XMLStreamConstants.PROCESSING_INSTRUCTION:
					throw new XMLStreamException("Unexpected event: " + type
							+ ", expected EndElement|Characters|CData|Space|Comment");
			}
		}
		throw new XMLStreamException("Unexpected end of stream, missing EndElement event.");
	}
	
	private void parseElement() throws XMLStreamException, IOException {
		while (xsr.hasNext()) {
			int type = xsr.next();
			switch(type) {
			  case XMLStreamConstants.ATTRIBUTE:
				throw new XMLStreamException("Invalid event type: ATTRIBUTE");
			  case XMLStreamConstants.END_ELEMENT:
					throw new XMLStreamException("Invalid event type: END_ELEMENT");
		      case XMLStreamConstants.DTD:
				handleDTD();
				break;
		      case XMLStreamConstants.START_ELEMENT:
		    	handleStartElement((StartElementEvent) xsr.createEvent());
		    	break;
			  case XMLStreamConstants.CDATA:
			  case XMLStreamConstants.CHARACTERS:
			  case XMLStreamConstants.SPACE:
				handleCharacters(type, xsr.getText());
				break;
			  case XMLStreamConstants.COMMENT:
				handleComment();
				break;
		      case XMLStreamConstants.START_DOCUMENT:
				handleStartDocument();
				break;
		      case XMLStreamConstants.END_DOCUMENT:
				handleEndDocument();
				break;
		      case XMLStreamConstants.ENTITY_REFERENCE:
				handleEntityReference();
				break;
			  case XMLStreamConstants.PROCESSING_INSTRUCTION:
				handleProcessingInstruction();
				break;
			}
		}
	}

	private void handleDTD() throws IOException {
		write("xsw.writeDTD(");
		writeEscaped(xsr.getText());
		write(")");
	}
	
	private void handleStartDocument() {
		//write("xsw.writeStartDocument(null, null)\n");
	}

	private void handleEndDocument() throws IOException {
		//write("xsw.writeEndDocument()\n");
		write("xsw.close()\n");
	}

	private void handleEntityReference() throws IOException {
		write("xsw.writeEntityRef(");
		writeEscaped(xsr.getText());
		write(")\n");
	}

	private void handleCharacters(int pEventType, String pText) throws IOException {
		write("buffer = ");
		writeEscaped(pText);
		write("\n");
		switch (pEventType) {
		  case XMLStreamConstants.CHARACTERS:
		  case XMLStreamConstants.SPACE:
			write("xsw.writeCharacters(buffer)\n");
			break;
		  case XMLStreamConstants.CDATA:
			write("xsw.writeCData(buffer)\n");
			break;
	      default:
	    	throw new IllegalStateException("Invalid event type: " + pEventType);
		}
	}

	private void handleComment() throws IOException {
		write("buffer = ");
		writeEscaped(xsr.getText());
		write("\n");
		write("xsw.writeComment(buffer)\n");
	}

	private void handleProcessingInstruction() throws IOException {
		final String data = xsr.getPIData();
		if (data == null) {
			write("xsw.writeProcessingInstruction(");
			writeEscaped(xsr.getPITarget());
			write(")\n");
		} else {
			write("xsw.writeProcessingInstruction(");
			writeEscaped(xsr.getPITarget());
			write(", ");
			writeEscaped(data);
			write(")\n");
		}
	}

	/**
	 * @param pEvent The scriptlet element with its attributes.
	 */
	private void handleScriptletInstruction(StartElementEvent pEvent) throws XMLStreamException, IOException {
		final String text = getElementText();
		write(text);
		write("\n");
	}

	private void handleElementInstruction(StartElementEvent pEvent) throws XMLStreamException, IOException {
		final String value = getAttributeValue(pEvent, ATTR_NAME);
		if (value == null) {
			throw new XMLStreamException("Missing attribute 'gsp:name' for '" + pEvent.getName() + "'.", pEvent.getLocation());
		}
		writeNamespaceDeclarations(pEvent);
		write("qName = xsw.getQName(");
		writeEscaped(value);
		write(")\n");
		write("xsw.writeStartElement(qName.getPrefix(), qName.getLocalPart(), qName.getNamespaceURI())\n");
		writeNamespaces(pEvent);
		writeElementAttributes(pEvent);
		handleElementContent();
		write("xsw.writeEndElement()\n");
	}
	
	private void handleStartElement(StartElementEvent pEvent) throws XMLStreamException, IOException {
		final QName name = pEvent.getName();
		final String uri = name.getNamespaceURI();
		final String localName = name.getLocalPart();
		if (XmlTemplateEngine.NS_URI.equals(uri)) {
			if ("scriptlet".equals(localName)) {
				handleScriptletInstruction(pEvent);
			} else if ("element".equals(localName)) {
				handleElementInstruction(pEvent);
			} else {
				throw new XMLStreamException("Unknown instruction: " + localName
						+ ", expected scriptlet", pEvent.getLocation());
			}
		} else {
			writeNamespaceDeclarations(pEvent);
			write("xsw.writeStartElement(");
			writeEscaped(name.getPrefix());
			write(", ");
			writeEscaped(localName);
			write(", ");
			writeEscaped(uri);
			write(")\n");
			writeNamespaces(pEvent);
			writeElementAttributes(pEvent);
			handleElementContent();
			write("xsw.writeEndElement()\n");
		}
	}

	private void handleElementContent() throws XMLStreamException, IOException {
		while (xsr.hasNext()) {
			final int type = xsr.next();
			switch(type) {
		      default:
				throw new XMLStreamException("Invalid event type: " + type);
			  case XMLStreamConstants.ENTITY_REFERENCE:
				handleEntityReference();
				break;
			  case XMLStreamConstants.CDATA:
			  case XMLStreamConstants.CHARACTERS:
			  case XMLStreamConstants.SPACE:
				handleCharacters(type, xsr.getText());
				break;
			  case XMLStreamConstants.COMMENT:
				handleComment();
				break;
			  case XMLStreamConstants.PROCESSING_INSTRUCTION:
				handleProcessingInstruction();
				break;
			  case XMLStreamConstants.START_ELEMENT:
				handleStartElement((StartElementEvent) xsr.createEvent());
				break;
			  case XMLStreamConstants.END_ELEMENT:
				return;
			}
		}
		throw new XMLStreamException("Unexpected end of stream, missing EndElement event.");
	}

	private String getAttributeValue(StartElementEvent pEvent, QName pQName) {
		final int numAttrs = pEvent.getAttributeCount();
		for (int i = 0;  i < numAttrs;  i++) {
			if (pQName.equals(pEvent.getAttributeName(i))) {
				return pEvent.getAttributeValue(i);
			}
		}
		return null;
	}

	private void writeAttribute(String pPrefix, String pUri, String pLocalName, String pValue) throws IOException {
		write("xsw.writeAttribute(");
		writeEscaped(pPrefix);
		write(",");
		writeEscaped(pUri);
		write(",");
		writeEscaped(pLocalName);
		write(",");
		writeEscaped(pValue);
		write(")\n");
	}
	
	private void writeElementAttributes(StartElementEvent pEvent) throws XMLStreamException, IOException {
		final int numAttrs = pEvent.getAttributeCount();
		for (int i = 0;  i < numAttrs;  i++) {
			final QName qName = pEvent.getAttributeName(i);
			final String uri = qName.getNamespaceURI();
			if (!XmlTemplateEngine.NS_URI.equals(uri)) {
				writeAttribute(qName.getPrefix(), uri, qName.getLocalPart(), pEvent.getAttributeValue(i));
			}
		}
		final List<Event> attrEvents = new ArrayList<Event>();
		while (xsr.hasNext()) {
			xsr.next();
			final Event event = xsr.createEvent();
			if (isAttributeEvent(event)) {
				attrEvents.add(event);
				continue;
			}
			if (event.getEventType() == XMLStreamConstants.START_ELEMENT) {
				final StartElementEvent startElement = (StartElementEvent) event;
				final QName qName = startElement.getName();
				if (qName.getNamespaceURI().equals(XmlTemplateEngine.NS_URI)
						&&  qName.getLocalPart().equals("attribute")) {
					for (Event attrEvent : attrEvents) {
						applyAttributeEvent(attrEvent);
					}
					attrEvents.clear();
					final String name = getAttributeValue(startElement, ATTR_SIMPLE_NAME);
					if (name == null) {
						throw new XMLStreamException("Missing attribute: "
								+ startElement.getName() + "/@name", startElement.getLocation());
					}
					final String value = getAttributeValue(startElement, ATTR_SIMPLE_VALUE);
					if (value == null) {
						throw new XMLStreamException("Missing attribute: "
								+ startElement.getName() + "/@value", startElement.getLocation());
					}
					write("qName = xsr.getQName(");
					writeEscaped(name);
					write("xsw.writeAttribute(qName.getPrefix(), qName.getNamespaceURI(), qName.getLocalPart(), ");
					writeEscaped(value);
					write(")\n");
					continue;
				}
			}
			xsr.pushBack(event);
			for (int i = attrEvents.size()-1;  i >= 0;  i--) {
				xsr.pushBack(attrEvents.get(i));
			}
			return;
		}
		throw new XMLStreamException("Unexpected end of stream, missing EndElement event.");
	}

	private void writeNamespaceDeclarations(StartElementEvent pEvent) throws IOException {
		write("xsw.startElement()\n");
		final int numNamespaces = pEvent.getNamespaceCount();
		for (int i = 0;  i < numNamespaces;  i++) {
			final String uri = pEvent.getNamespaceURI(i);
			if (!XmlTemplateEngine.NS_URI.equals(uri)) {
				String prefix = pEvent.getNamespacePrefix(i);
				if (prefix == null) {
					prefix = "";
				}
				write("xsw.declarePrefix(");
				writeEscaped(prefix);
				write(",");
				writeEscaped(uri);
				write(")\n");
			}
		}
	}

	private void writeNamespaces(StartElementEvent pEvent) throws IOException {
		final int numNamespaces = pEvent.getNamespaceCount();
		for (int i = 0;  i < numNamespaces;  i++) {
			final String uri = pEvent.getNamespaceURI(i);
			if (!XmlTemplateEngine.NS_URI.equals(uri)) {
				String prefix = pEvent.getNamespacePrefix(i);
				if (prefix == null) {
					prefix = "";
				}
				write("xsw.writeNamespace(");
				writeEscaped(prefix);
				write(",");
				writeEscaped(uri);
				write(")\n");
			}
		}
	}
}
