package net.sf.csutils.groovy.xml;

import java.io.InputStream;
import java.io.Reader;
import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;
import javax.xml.stream.Location;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamConstants;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;

public class XmlTemplateEngineStreamReader {
	public static class Event {
		private final int type;
		private final Location location; 

		Event(Location pLocation, int pType) {
			location = pLocation;
			type = pType;
		}

		public int getEventType() {
			return type;
		}

		public Location getLocation() {
			return location;
		}
	}

	public static class TextEvent extends Event {
		private final String text;

		TextEvent(Location pLocation, int pType, String pText) {
			super(pLocation, pType);
			text = pText;
		}

		public String getText() {
			return text;
		}
	}
	
	private static class Attribute {
		final QName name;
		final String value;

		Attribute(QName pName, String pValue) {
			name = pName;
			value = pValue;
		}
	}

	public static class Namespace {
		private final String prefix;
		private final String uri;

		Namespace(String pPrefix, String pUri) {
			if (pUri == null) {
				throw new IllegalArgumentException("The namespace URI must not be null.");
			}
			prefix = pPrefix;
			uri = pUri;
		}

		public String getPrefix() {
			return prefix;
		}

		public String getNamespaceURI() {
			return uri;
		}
	}

	public static class StartElementEvent extends Event {
		private final QName name;
		private final Attribute[] attrs;
		private final Namespace[] namespaces;

		StartElementEvent(Location pLocation, QName pQName, Attribute[] pAttrs, Namespace[] pNamespaces) {
			super(pLocation, XMLStreamConstants.START_ELEMENT);
			attrs = pAttrs;
			name = pQName;
			namespaces = pNamespaces;
		}

		public QName getName() {
			return name;
		}

		public int getAttributeCount() {
			return attrs.length;
		}

		public QName getAttributeName(int pIndex) {
			return attrs[pIndex].name;
		}

		public String getAttributeValue(int pIndex) {
			return attrs[pIndex].value;
		}

		public int getNamespaceCount() {
			return namespaces.length;
		}

		public String getNamespacePrefix(int pIndex) {
			return namespaces[pIndex].getPrefix();
		}

		public String getNamespaceURI(int pIndex) {
			return namespaces[pIndex].getNamespaceURI();
		}
	}
	
	private final XMLStreamReader xsr;
	private Event currentEvent;
	private final List<Event> events = new ArrayList<Event>();

	public XmlTemplateEngineStreamReader(Reader pReader) throws XMLStreamException {
		this(new StreamSource(pReader));
	}

	public XmlTemplateEngineStreamReader(InputStream pStream) throws XMLStreamException {
		this(new StreamSource(pStream));
	}

	public XmlTemplateEngineStreamReader(Source pSource) throws XMLStreamException {
		xsr = XMLInputFactory.newInstance().createXMLStreamReader(pSource);
	}

	public boolean hasNext() throws XMLStreamException {
		return !events.isEmpty() || xsr.hasNext();
	}

	public int next() throws XMLStreamException {
		final int result;
		if (events.isEmpty()) {
			currentEvent = null;
			result = xsr.next();
		} else {
			currentEvent = events.remove(0);
			result = currentEvent.getEventType();
		}
		return result;
	}

	public int getEventType() {
		if (currentEvent == null) {
			return xsr.getEventType();
		}
		return currentEvent.getEventType();
	}

	private TextEvent assertTextEvent(Event pEvent) {
		if (pEvent instanceof TextEvent) {
			return (TextEvent) pEvent;
		}
		throw new IllegalStateException("Current event is an instance of "
				+ currentEvent.getClass().getName()
				+ ", not an instance of " + TextEvent.class.getName());
	}

	private StartElementEvent assertStartElementEvent(Event pEvent) {
		if (pEvent instanceof StartElementEvent) {
			return (StartElementEvent) pEvent;
		}
		throw new IllegalStateException("Current event is an instance of "
				+ currentEvent.getClass().getName()
				+ ", not an instance of " + StartElementEvent.class.getName());
	}

	public String getText() {
		if (currentEvent == null) {
			return xsr.getText();
		}
		return assertTextEvent(currentEvent).getText();
	}

	public String getPIData() {
		if (currentEvent == null) {
			return xsr.getPIData();
		}
		throw new IllegalStateException("Unable to cache processing instructions.");
	}

	public String getPITarget() {
		if (currentEvent == null) {
			return xsr.getPITarget();
		}
		throw new IllegalStateException("Unable to cache processing instructions.");
	}

	public QName getName() {
		if (currentEvent == null) {
			return xsr.getName();
		}
		return assertStartElementEvent(currentEvent).getName();
	}

	public Location getLocation() {
		if (currentEvent == null) {
			return xsr.getLocation();
		}
		return currentEvent.getLocation();
	}

	public int getAttributeCount() {
		if (currentEvent == null) {
			return xsr.getAttributeCount();
		}
		return assertStartElementEvent(currentEvent).getAttributeCount();
	}

	public QName getAttributeName(int i) {
		if (currentEvent == null) {
			return xsr.getAttributeName(i);
		}
		return assertStartElementEvent(currentEvent).getAttributeName(i);
	}

	public String getAttributeValue(int i) {
		if (currentEvent == null) {
			return xsr.getAttributeValue(i);
		}
		return assertStartElementEvent(currentEvent).getAttributeValue(i);
	}

	public int getNamespaceCount() {
		if (currentEvent == null) {
			return xsr.getNamespaceCount();
		}
		return assertStartElementEvent(currentEvent).getNamespaceCount();
	}

	public String getNamespacePrefix(int pIndex) {
		if (currentEvent == null) {
			return xsr.getNamespacePrefix(pIndex);
		}
		return assertStartElementEvent(currentEvent).getNamespacePrefix(pIndex);
	}

	public String getNamespaceURI(int pIndex) {
		if (currentEvent == null) {
			return xsr.getNamespaceURI(pIndex);
		}
		return assertStartElementEvent(currentEvent).getNamespaceURI(pIndex);
	}

	public Event createEvent() {
		if (currentEvent == null) {
			final int type = getEventType();
			switch (type) {
				case XMLStreamConstants.SPACE:
				case XMLStreamConstants.CDATA:
				case XMLStreamConstants.CHARACTERS:
				case XMLStreamConstants.COMMENT:
					return new TextEvent(getLocation(), type, getText());
				case XMLStreamConstants.START_ELEMENT:
					final Attribute[] attrs = new Attribute[getAttributeCount()];
					for (int i = 0;  i < attrs.length;  i++) {
						attrs[i] = new Attribute(getAttributeName(i), getAttributeValue(i));
					}
					final Namespace[] namespaces = new Namespace[getNamespaceCount()];
					for (int i = 0;  i < namespaces.length;  i++) {
						namespaces[i] = new Namespace(getNamespacePrefix(i), getNamespaceURI(i));
					}
					return new StartElementEvent(getLocation(), getName(), attrs, namespaces);
				default:
					throw new IllegalStateException("Invalid event type: " + type);
			}
		}
		return currentEvent;
	}

	public void pushBack(Event event) {
		events.add(0, event);
		currentEvent = null;
	}
}
