package net.sf.csutils.impexp;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import javax.xml.registry.JAXRException;
import javax.xml.stream.Location;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.transform.Source;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.sax.SAXSource;
import javax.xml.transform.stax.StAXSource;
import javax.xml.transform.stream.StreamSource;

import net.sf.csutils.core.registry.ConnectionProvider;
import net.sf.csutils.impexp.impimpl.ContextProvidingParser;
import net.sf.csutils.impexp.impimpl.ContextProvidingParser.Context;
import net.sf.csutils.impexp.impimpl.ContextProvidingParser.ParseException;
import net.sf.csutils.importer.vo.Model;

import org.w3c.dom.Node;
import org.xml.sax.InputSource;


/**
 * The importer is used to read an XML file and create, or update matching
 * registry objects.
 */
public class Importer {
    public static class Error {
        private final int lineNumber, columnNumber;
        private final String systemId, publicId;
        private final String msg;
    
        public Error(String pMessage) {
            this(null, null, -1, -1, pMessage);
        }
        
        public Error(String pPublicId, String pSystemId,
                int pLineNumber, int pColumnNumber, String pMessage) {
            publicId = pPublicId;
            systemId = pSystemId;
            msg = pMessage;
            lineNumber = pLineNumber;
            columnNumber = pColumnNumber;
        }
    
        public String getMessage() {
            return msg;
        }

        public int getLineNumber() {
            return lineNumber;
        }

        public int getColumnNumber() {
            return columnNumber;
        }

        public String getPublicId() {
            return publicId;
        }

        public String getSystemId() {
            return systemId;
        }
    }

    private Unmarshaller newUnmarshaller() throws JAXBException {
        return JAXBContext.newInstance(Model.class).createUnmarshaller();
    }

    public List<Error> read(ConnectionProvider pProvider, URL pURL) throws JAXRException {
        InputStream stream = null;
        try {
            stream = pURL.openStream();
            final InputSource inputSource = new InputSource(stream);
            inputSource.setSystemId(pURL.toExternalForm());
            final List<Error> errors = read(pProvider, inputSource);
            stream.close();
            stream = null;
            return errors;
        } catch (IOException e) {
            throw new JAXRException(e);
        } finally {
            if (stream != null) { try { stream.close(); } catch (Throwable t) { /* Ignore me */ } }
        }
    }

    public List<Error> read(ConnectionProvider pProvider, File pFile) throws JAXRException {
        return read(pProvider, new StreamSource(pFile));
    }

    public List<Error> read(ConnectionProvider pProvider, InputSource pSource) throws JAXRException {
        return read(pProvider, new SAXSource(pSource));
    }

    public List<Error> read(ConnectionProvider pProvider, InputStream pStream) throws JAXRException {
        return read(pProvider, new StreamSource(pStream));
    }

    public List<Error> read(ConnectionProvider pProvider, Reader pReader) throws JAXRException {
        return read(pProvider, new StreamSource(pReader));
    }

    public List<Error> read(ConnectionProvider pProvider, Source pSource) throws JAXRException {
        try {
            final XMLInputFactory factory = XMLInputFactory.newFactory();
            final XMLStreamReader reader = factory.createXMLStreamReader(pSource);
            return read(pProvider, reader);
        } catch (XMLStreamException e) {
            throw new JAXRException(e);
        }
    }

    public List<Error> read(ConnectionProvider pProvider, XMLEventReader pReader) throws JAXRException {
        try {
            return read(pProvider, new StAXSource(pReader));
        } catch (XMLStreamException e) {
            throw new JAXRException(e);
        }
    }

    public List<Error> read(ConnectionProvider pProvider, Node pNode) throws JAXRException {
        return read(pProvider, new DOMSource(pNode));
    }

    public List<Error> read(ConnectionProvider pProvider, XMLStreamReader pReader) throws JAXRException {
        final List<Error> errors = new ArrayList<Error>();
        final net.sf.csutils.impexp.impimpl.Importer importer = new net.sf.csutils.impexp.impimpl.Importer(pProvider){
            @Override
            protected void logError(String pMessage) {
                super.logError(pMessage);
                final Context context = ContextProvidingParser.getContext();
                if (context != null) {
                    final Location location = context.getLocation();
                    if (location != null) {
                        errors.add(new Error(location.getPublicId(),
                                location.getSystemId(), location.getLineNumber(),
                                location.getColumnNumber(), pMessage));
                    }
                }
                errors.add(new Error(pMessage));
            }
        };
        try {
            new ContextProvidingParser() {
                @Override
                protected void doParse(XMLStreamReader pRdr) throws ParseException {
                    try {
                        final Unmarshaller unmarshaller = newUnmarshaller();
                        final Model model = (Model) unmarshaller.unmarshal(pRdr);
                        importer.read(model);
                    } catch (JAXBException e) {
                        throw new ParseException(e);
                    } catch (JAXRException e) {
                        throw new ParseException(e);
                    }
                }
            }.parse(pReader);
        } catch (ParseException e) {
            final Throwable t = e.getCause();
            if (t != null) {
                if (t instanceof JAXRException) {
                    throw (JAXRException) t;
                }
                if (t instanceof RuntimeException) {
                    throw (RuntimeException) t;
                }
                if (t instanceof java.lang.Error) {
                    throw (java.lang.Error) t;
                }
                throw new JAXRException(t);
            }
            throw new JAXRException(e);
        }
        return errors;
    }
}
