package net.sf.csutils.impexp;

import java.io.File;
import java.net.URL;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.log4j.BasicConfigurator;
import org.apache.log4j.PropertyConfigurator;
import org.apache.log4j.xml.DOMConfigurator;

import net.sf.csutils.impexp.jdbc.JdbcDbExporter;


/**
 * Command line interface of the {@link JdbcDbExporter}.
 */
public class JdbcDbExporterMain {
	private String jdbcDriver, jdbcUser, jdbcUrl, jdbcPassword;
	private String dbCatalog, dbSchema, mode;
	private File jdbcExportFile, outputFile;

	public File getOutputFile() {
		return outputFile;
	}

	public void setOutputFile(File outputFile) {
		this.outputFile = outputFile;
	}

	public File getJdbcExportFile() {
		return jdbcExportFile;
	}

	public void setJdbcExportFile(File jdbcExportFile) {
		this.jdbcExportFile = jdbcExportFile;
	}

	public String getMode() {
		return mode;
	}

	public void setMode(String mode) {
		this.mode = mode;
	}

	public String getDbCatalog() {
		return dbCatalog;
	}

	public void setDbCatalog(String dbCatalog) {
		this.dbCatalog = dbCatalog;
	}

	public String getDbSchema() {
		return dbSchema;
	}

	public void setDbSchema(String dbSchema) {
		this.dbSchema = dbSchema;
	}

	public String getJdbcDriver() {
		return jdbcDriver;
	}

	public void setJdbcDriver(String jdbcDriver) {
		this.jdbcDriver = jdbcDriver;
	}

	public String getJdbcUser() {
		return jdbcUser;
	}

	public void setJdbcUser(String jdbcUser) {
		this.jdbcUser = jdbcUser;
	}

	public String getJdbcUrl() {
		return jdbcUrl;
	}

	public void setJdbcUrl(String jdbcUrl) {
		this.jdbcUrl = jdbcUrl;
	}

	public String getJdbcPassword() {
		return jdbcPassword;
	}

	public void setJdbcPassword(String jdbcPassword) {
		this.jdbcPassword = jdbcPassword;
	}

	public void run() throws Exception {
		final JdbcDbExporter exporter = new JdbcDbExporter();
		Class.forName(getJdbcDriver());
		Connection conn = DriverManager.getConnection(getJdbcUrl(), getJdbcUser(), getJdbcPassword());
		try {
			final DatabaseMetaData metaData = conn.getMetaData();
			if (getDbCatalog() == null  &&  getDbSchema() == null) {
				final String productName = metaData.getDatabaseProductName();
				if ("Oracle".equals(productName)) {
					exporter.setSchema(getJdbcUser().toUpperCase());
				}
			} else {
				exporter.setCatalog(getDbCatalog());
				exporter.setSchema(getDbSchema());
			}
			if ("JdbcExport".equalsIgnoreCase(getMode())) {
				exporter.createJdbcExport(metaData, getOutputFile());
			} else if ("MetaModel".equalsIgnoreCase(getMode())) {
				exporter.createMetaModel(getJdbcExportFile(), metaData, getOutputFile());
			} else if ("Model".equalsIgnoreCase(getMode())) {
				exporter.createModel(getJdbcExportFile(), metaData, getOutputFile());
			} else {
				throw new IllegalStateException("Invalid mode: " + getMode());
			}
			conn.close();
			conn = null;
		} finally {
			if (conn != null) { try { conn.close(); } catch (Throwable t) { /* Ignore me */ } }
		}
	}
	
	private static RuntimeException Usage(String pMessage) {
		if (pMessage != null) {
			System.err.println(pMessage);
			System.err.println();
		}
		System.err.println("Usage: java " + JdbcDbExporterMain.class.getName()
				+ " <options>");
		System.err.println();
		System.err.println("Required options are:");
		System.err.println("  -jdbcDriver <class>     Sets the JDBC driver class.");
		System.err.println("  -jdbcUrl <url>          Sets the JDBC url.");
		System.err.println("  -jdbcUser <user>        Sets the JDBC user name.");
		System.err.println("  -jdbcPassword <pwd>     Sets the JDBC password.");
		System.err.println("  -mode <mode>            Either of JdbcExport|MetaModel|Model.");
		System.err.println("  -outputFile <file>      Sets the output file.");
		System.err.println();
		System.err.println("Possible options are:");
		System.err.println("  -dbCatalog <catalog>    Sets the database catalog.");
		System.err.println("  -dbSchema <schema>      Sets the database catalog.");
		System.err.println("  -jdbcExportFile <file>  Sets the JDBC export description file.");
		System.exit(1);
		return null;
	}
	
	public static void main(String[] pArgs) throws Exception {
		final JdbcDbExporterMain main = new JdbcDbExporterMain();
		final List<String> args = new ArrayList<String>(Arrays.asList(pArgs));
		while (!args.isEmpty()) {
			final String arg = args.remove(0);
			final String opt;
			if (arg.startsWith("--")) {
				opt = arg.substring(2);
			} else if (arg.startsWith("-")) {
				opt = arg.substring(1);
			} else {
				throw Usage("Invalid argument: " + arg);
			}
			if ("dbCatalog".equals(opt)) {
				if (args.isEmpty()) {
					throw Usage("Option " + arg + " requires an argument (database catalog)");
				}
				main.setDbCatalog(args.remove(0));
			} else if ("dbSchema".equals(opt)) {
				if (args.isEmpty()) {
					throw Usage("Option " + arg + " requires an argument (database schema)");
				}
				main.setJdbcDriver(args.remove(0));
			} else if ("jdbcDriver".equals(opt)) {
				if (args.isEmpty()) {
					throw Usage("Option " + arg + " requires an argument (JDBC driver class)");
				}
				main.setJdbcDriver(args.remove(0));
			} else if ("jdbcExportFile".equals(opt)) {
				if (args.isEmpty()) {
					throw Usage("Option " + arg + " requires an argument (JDBC export file)");
				}
				final File file = new File(args.remove(0));
				if (!file.canRead()) {
					throw Usage("Invalid value for option " + arg
							+ ", file does not exist, or is unreadable: " + file.getPath());
				}
				main.setJdbcExportFile(file);
			} else if ("jdbcUrl".equals(opt)) {
				if (args.isEmpty()) {
					throw Usage("Option " + arg + " requires an argument (JDBC URL)");
				}
				main.setJdbcUrl(args.remove(0));
			} else if ("jdbcUser".equals(opt)) {
				if (args.isEmpty()) {
					throw Usage("Option " + arg + " requires an argument (JDBC User)");
				}
				main.setJdbcUser(args.remove(0));
			} else if ("jdbcPassword".equals(opt)) {
				if (args.isEmpty()) {
					throw Usage("Option " + arg + " requires an argument (JDBC Password)");
				}
				main.setJdbcPassword(args.remove(0));
			} else if ("mode".equals(opt)) {
				if (args.isEmpty()) {
					throw Usage("Option " + arg + " requires an argument (JdbcExport|MetaModel|Model)");
				}
				final String mode = args.remove(0);
				if (!"JdbcExport".equalsIgnoreCase(mode)  &&
						!"MetaModel".equalsIgnoreCase(mode)  &&
						!"Model".equalsIgnoreCase(mode)) {
					throw Usage("Invalid value for option " + arg
							+ ", expected JdbcExport|MetaModel|Model: "
							+ mode);
				}
				main.setMode(mode);
			} else if ("outputFile".equals(opt)) {
				if (args.isEmpty()) {
					throw Usage("Option " + arg + " requires an argument (Output file name)");
				}
				final File file = new File(args.remove(0));
				main.setOutputFile(file);
			} else {
				throw Usage("Unknown option: " + arg);
			}
		}
		if (main.getJdbcDriver() == null) {
			throw Usage("Missing option: -jdbcDriver (JDBC driver class)");
		}
		if (main.getJdbcUrl() == null) {
			throw Usage("Missing option: -jdbcUrl (JDBC URL)");
		}
		if (main.getJdbcUser() == null) {
			throw Usage("Missing option: -jdbcUser (JDBC User)");
		}
		if (main.getJdbcPassword() == null) {
			throw Usage("Missing option: -jdbcPassword (JDBC Password)");
		}
		if (main.getOutputFile() == null) {
			throw Usage("Missing option: -outputFile (Output file name)");
		}
		final String mode = main.getMode();
		if (mode == null) {
			throw Usage("Missing option: -mode (JdbcExport|MetaModel|Model)");
		}
		if ("JdbcExport".equalsIgnoreCase(mode)) {
			
		} else if ("MetaModel".equalsIgnoreCase(mode)) {
			if (main.getJdbcExportFile() == null) {
				throw Usage("Option -jdbcExportFile is required for operation mode " + mode);
			}
		} else if ("Model".equalsIgnoreCase(mode)) {
			if (main.getJdbcExportFile() == null) {
				throw Usage("Option -jdbcExportFile is required for operation mode " + mode);
			}
		} else {
			throw new IllegalStateException("Invalid operation mode: " + mode);
		}

		final URL url1 = Thread.currentThread().getContextClassLoader().getResource("log4j.properties");
		if (url1 != null) {
			PropertyConfigurator.configure(url1);
		} else {
			final URL url2 = Thread.currentThread().getContextClassLoader().getResource("log4j.xml");
			if (url2 != null) {
				DOMConfigurator.configure(url2);
			} else {
				BasicConfigurator.configure();
			}
		}
		
		main.run();
	}
}
