package net.sf.csutils.impexp.impimpl;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import javax.xml.bind.JAXBElement;
import javax.xml.registry.BusinessLifeCycleManager;
import javax.xml.registry.Connection;
import javax.xml.registry.JAXRException;
import javax.xml.registry.LifeCycleManager;
import javax.xml.registry.infomodel.Association;
import javax.xml.registry.infomodel.InternationalString;
import javax.xml.registry.infomodel.LocalizedString;
import javax.xml.registry.infomodel.Organization;
import javax.xml.registry.infomodel.RegistryEntry;
import javax.xml.registry.infomodel.RegistryObject;
import javax.xml.registry.infomodel.Slot;

import net.sf.csutils.core.model.QName;
import net.sf.csutils.core.model.ROAttribute;
import net.sf.csutils.core.model.ROAttribute.Type;
import net.sf.csutils.core.model.ROMetaModel;
import net.sf.csutils.core.model.RORelation;
import net.sf.csutils.core.model.ROSlot;
import net.sf.csutils.core.model.ROType;
import net.sf.csutils.core.registry.ConnectionProvider;
import net.sf.csutils.core.registry.ModelDrivenRegistryFacade;
import net.sf.csutils.core.registry.ROModelAccessor;
import net.sf.csutils.core.registry.SimpleModelDrivenRegistryFacade;
import net.sf.csutils.core.utils.Generics;
import net.sf.csutils.core.utils.RegistryObjects;
import net.sf.csutils.impexp.impimpl.ImporterData.CachedAsset;
import net.sf.csutils.impexp.impimpl.ImporterData.CachedAsset.Operation;
import net.sf.csutils.impexp.impimpl.ImporterData.CachedAssetType;
import net.sf.csutils.importer.vo.AbstractAttribute;
import net.sf.csutils.importer.vo.AbstractIdentification;
import net.sf.csutils.importer.vo.AbstractInternationalString;
import net.sf.csutils.importer.vo.AbstractReference;
import net.sf.csutils.importer.vo.AbstractRegistryObject;
import net.sf.csutils.importer.vo.AbstractRegistryObject.Attributes;
import net.sf.csutils.importer.vo.AbstractRelation;
import net.sf.csutils.importer.vo.AbstractSlot;
import net.sf.csutils.importer.vo.AbstractTypedReference;
import net.sf.csutils.importer.vo.Model;
import net.sf.csutils.importer.vo.Model.Assets;
import net.sf.csutils.importer.vo.Model.Identifications;

import org.apache.ws.commons.util.XsDateFormat;
import org.apache.ws.commons.util.XsDateTimeFormat;
import org.apache.ws.commons.util.XsTimeFormat;

import com.centrasite.jaxr.infomodel.CentraSiteRegistryObject;
import com.centrasite.jaxr.type.CentraSiteSlotType;

public abstract class Importer {
    private static final QName QNAME_ORGANIZATION = new QName(LifeCycleManager.ORGANIZATION);
    private static final XsDateFormat dateFormat = new XsDateFormat();
    private static final XsDateTimeFormat dateTimeFormat = new XsDateTimeFormat();
    private static final XsTimeFormat timeFormat = new XsTimeFormat();

    private final ConnectionProvider provider;
    private ImporterData data;
    private ModelDrivenRegistryFacade facade;
    private ROModelAccessor mAcc;
    private ROMetaModel metaModel;
    private boolean hasErrors;

    public Importer(ConnectionProvider pProvider) {
        provider = pProvider;
    }

    /**
     * Called to indicate that an error has occurred. Subclasses
     * should overwrite this, so that the message is recorded,
     * for example by logging it.
     * @param pMessage The message to log.
     */
    protected void logError(String pMessage) {
        hasErrors = true;
    }

    protected QName asQName(String pValue) {
        return QName.valueOf(pValue);
    }
    
    private void readIdentifications(Model pModel) throws JAXRException {
        final Identifications identifications = pModel.getIdentifications();
        if (identifications != null) {
            for (JAXBElement<? extends AbstractIdentification> identificationElement : identifications.getUniqueKeyOrUniqueNameOrUniqueSlot()) {
                final AbstractIdentification identification = identificationElement.getValue();
                final net.sf.csutils.core.model.QName qName = asQName(identification.getAssetType());
                final ROType type = metaModel.getROType(qName);
                if (type == null) {
                    logError("Unknown type " + identification.getAssetType()
                            + " in identification.");
                    continue;
                }
                final String localPart = identificationElement.getName().getLocalPart();
                final AssetType assetType;
                if ("uniqueKey".equals(localPart)) {
                    assetType = new AssetType(AssetType.Type.key, qName);
                } else if ("uniqueName".equals(localPart)) {
                    assetType = new AssetType(AssetType.Type.name, qName);
                } else if ("uniqueSlot".equals(localPart)) {
                    final Identifications.UniqueSlot uniqueSlot = (Identifications.UniqueSlot) identification;
                    final String slotName = uniqueSlot.getAttributeName();
                    final ROAttribute attr = type.getAttribute(slotName);
                    if (attr == null) {
                        logError("Unknown slot " + slotName + " for asset type " + identification.getAssetType());
                        continue;
                    }
                    final Type slotType = attr.getType();
                    if (ROAttribute.Type.relation == slotType  ||  ROAttribute.Type.classification == slotType) {
                        logError("Invalid slot type " + slotType.name()
                                + " for unique slot " + slotName
                                + " of asset type " + identification.getAssetType());
                        continue;
                    }
                    assetType = new SlotValueAssetType(qName, uniqueSlot.getAttributeName(), slotType);
                } else {
                    logError("Invalid identification: " + localPart);
                    continue;
                }
                data.addAssetType(assetType);
            }
        }
    }

    private void setNames(CachedAsset pAsset) throws JAXRException {
    	if (!pAsset.getBean().isDelete()) {
	    	final InternationalString is = pAsset.getRegistryObject().getName();
	        final List<AbstractInternationalString> names = pAsset.getBean().getNames().getName();
	        final InternationalString update = setInternationalString(names, is);
	        if (update != null) {
	            pAsset.getRegistryObject().setName(update);
	            pAsset.setModified();
	        }
    	}
    }

    private static final class ISKey {
        final String value;
        final Locale locale;

        ISKey(String pValue, Locale pLocale) {
            if (pValue == null) {
                throw new IllegalArgumentException("The value must not be null.");
            }
            if (pLocale == null) {
                throw new IllegalArgumentException("The locale must not be null.");
            }
            value = pValue;
            locale = pLocale;
        }

        @Override
        public int hashCode() {
            return 31 * (31 + locale.hashCode()) + value.hashCode();
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (getClass() != obj.getClass())
                return false;
            ISKey other = (ISKey) obj;
            return value.equals(other.value)  &&  locale.equals(other.locale);
        }
    }

    private Set<ISKey> isSame(List<AbstractInternationalString> pValues, InternationalString pIs)
            throws JAXRException {
        final Set<ISKey> keys = new HashSet<ISKey>();
        for (AbstractInternationalString value : pValues) {
            keys.add(new ISKey(value.getValue(), RegistryObjects.asLocale(value.getLocale())));
        }
        final Collection<LocalizedString> localizedStrings = Generics.cast(pIs.getLocalizedStrings());
        if (localizedStrings.size() != keys.size()) {
            return keys;
        }
        for (LocalizedString ls : localizedStrings) {
            final ISKey key = new ISKey(ls.getValue(), ls.getLocale());
            if (!keys.contains(key)) {
                return keys;
            }
        }
        return null;
    }

    private InternationalString setInternationalString(List<AbstractInternationalString> pValues, InternationalString pIs) throws JAXRException {
        final Set<ISKey> values = isSame(pValues, pIs);
        if (values == null) {
            return null;
        }
        final InternationalString is = facade.getBusinessLifeCycleManager().createInternationalString();
        for (ISKey value : values) {
            is.setValue(value.locale, value.value);
        }
        return is;
        
    }

    private void setDescriptions(CachedAsset pAsset) throws JAXRException {
        final InternationalString is = pAsset.getRegistryObject().getDescription();
        final List<AbstractInternationalString> descriptions;
        if (pAsset.getBean().getDescriptions() != null) {
            descriptions = pAsset.getBean().getDescriptions().getDescription();
        } else {
            descriptions = new ArrayList<AbstractInternationalString>();
        }
        final InternationalString update = setInternationalString(descriptions, is);
        if (update != null) {
            pAsset.getRegistryObject().setDescription(update);
            pAsset.setModified();
        }
    }

    private List<Object> isSame(ROAttribute.Type pType, List<String> pValues, Slot pSlot) throws JAXRException {
        final List<Object> valueObjects = new ArrayList<Object>(pValues.size());
        for (int i = 0;  i < pValues.size();  i++) {
            valueObjects.add(asSlotObject(pType, pValues.get(i)));
        }
        if (pSlot == null) {
            return valueObjects;
        }
        final Collection<String> slotValues = Generics.cast(pSlot.getValues());
        if (slotValues.size() != pValues.size()) {
            return valueObjects;
        }
        final Iterator<String> iter = slotValues.iterator();
        for (int i = 0;  i < pValues.size();  i++) {
            final Object o = asSlotObject(pType, iter.next());
            if (!o.equals(valueObjects.get(i))) {
                return valueObjects;
            }
        }
        return null;
    }

    private String getSlotType(ROAttribute.Type pType) {
        switch (pType) {
            case bool:
                return CentraSiteSlotType.BOOLEAN;
            case date:
                return CentraSiteSlotType.XS_DATE;
            case dateTime:
                return CentraSiteSlotType.XS_DATETIME;
            case time:
                return CentraSiteSlotType.XS_TIME;
            case duration:
                return CentraSiteSlotType.XS_DURATION;
            case emailAddress:
                return CentraSiteSlotType.EMAILADDRESS;
            case ipv4Address:
                return CentraSiteSlotType.IPV4;
            case ipv6Address:
                return CentraSiteSlotType.IPV6;
            case integer:
                return CentraSiteSlotType.XS_INTEGER;
            case longInteger:
                return CentraSiteSlotType.XS_INTEGER;
            case number:
                return CentraSiteSlotType.XS_DOUBLE;
            case string:
                return CentraSiteSlotType.XS_STRING;
            case uri:
                return CentraSiteSlotType.XS_ANYURI;
            case classification:
            case relation:
            default:
                throw new IllegalStateException("Invalid slot type: " + pType.name());
        }
    }

    private Map<String,ROSlot> getSlots(ROType pType) throws JAXRException {
        final Map<String,ROAttribute> attrs = pType.getAttributes();
        final Map<String,ROSlot> slots = new HashMap<String,ROSlot>();
        for (Map.Entry<String,ROAttribute> entry : attrs.entrySet()) {
            final ROAttribute attr = entry.getValue();
            if (attr instanceof ROSlot) {
                slots.put(entry.getKey(), (ROSlot) attr);
            }
        }
        return slots;
    }

    private Map<String,RORelation> getRelations(ROType pType) throws JAXRException {
        final Map<String,ROAttribute> attrs = pType.getAttributes();
        final Map<String,RORelation> relations = new HashMap<String,RORelation>();
        for (Map.Entry<String,ROAttribute> entry : attrs.entrySet()) {
            final ROAttribute attr = entry.getValue();
            if (attr instanceof RORelation) {
                relations.put(entry.getKey(), (RORelation) attr);
            }
        }
        return relations;
    }

    private void setSlots(CachedAsset pAsset) throws JAXRException {
        final RegistryObject ro = pAsset.getRegistryObject();
        final Map<String,ROSlot> attrs = getSlots(pAsset.getType().getROType());
        final Attributes beanAttributes = pAsset.getBean().getAttributes();
        final QName qName = pAsset.getType().getROType().getQueryName();
        if (beanAttributes != null) {
            final List<AbstractAttribute> beanAttrs = beanAttributes.getSlotOrRelation();
            for (AbstractAttribute beanAttr : beanAttrs) {
                if (!(beanAttr instanceof AbstractSlot)) {
                    continue;
                }
                final AbstractSlot beanSlot = (AbstractSlot) beanAttr;
                final List<String> values = beanSlot.getValue();
                final ROSlot roAttribute = attrs.remove(beanSlot.getName());
                final ROAttribute.Type slotType;
                if (roAttribute == null) {
                    if (pAsset.getType().getROType().getAttribute(beanSlot.getName()) == null) {
                        slotType = ROAttribute.Type.string;
                    } else {
                        logError("Attribute " + beanSlot.getName()
                                + " of type " + pAsset.getType().getROType().getQName()
                                + " is no slot.");
                        continue;
                    }
                } else {
                    slotType = roAttribute.getType();
                }
                final String qualifiedSlotName = "".equals(qName.getNamespaceURI()) ? beanSlot.getName() : "{" + qName.getNamespaceURI() + "}" + beanSlot.getName();
                final Slot slot = ro.getSlot(qualifiedSlotName);
                if (values.isEmpty()) {
                    if (slot != null) {
                        ro.removeSlot(qualifiedSlotName);
                        pAsset.setModified();
                    }
                } else {
                    final List<Object> newValues = isSame(slotType, values, slot);
                    if (newValues != null) {
                        final List<String> slotValues = new ArrayList<String>();
                        for (Object newValue : newValues) {
                            slotValues.add(asSlotString(slotType, newValue));
                        }
                        ro.addSlot(facade.getBusinessLifeCycleManager().createSlot(qualifiedSlotName, slotValues, getSlotType(slotType)));
                        pAsset.setModified();
                    }
                }
            }
        }
        for (ROAttribute attr : attrs.values()) {
            final String qualifiedSlotName = "".equals(qName.getNamespaceURI()) ? attr.getName() : "{" + qName.getNamespaceURI() + "}" + attr.getName();
            final Slot slot = ro.getSlot(qualifiedSlotName);
            if (slot != null) {
                ro.removeSlot(qualifiedSlotName);
                pAsset.setModified();
            }
        }
    }

    private List<CachedAsset> getTargets(AssetType pType, RORelation pRelation, List<AbstractReference> pTargetRefs) throws JAXRException {
        final List<CachedAsset> cachedAssets = new ArrayList<CachedAsset>();
        for (AbstractReference ref : pTargetRefs) {
            final String typeName = ref.getType();
            final QName typeQName;
            if (typeName == null) {
                if (pRelation.getTargetTypes().size() > 1) {
                    logError("A target referenced by the relation "
                            + pRelation.getName()
                            + " of " + pType.getQName()
                            + " must have its type attribute set.");
                    return null;
                }
                typeQName = pRelation.getTargetTypes().get(0).getQName();
            } else {
                typeQName = asQName(typeName);
            }
            final CachedAssetType cachedAssetType = data.getAssetType(typeQName);
            final CachedAsset cachedAsset = getReferencedObject(cachedAssetType, ref);
            if (cachedAsset == null) {
                return null;
            }
            if (cachedAsset.getBean().isDelete()) {
            	logError("A deleted bean must not be referenced by a relation.");
            	return null;
            }
            cachedAssets.add(cachedAsset);
        }
        return cachedAssets;
    }

    private List<Association> getAssociations(RORelation pRelation, RegistryObject pRegistryObject) throws JAXRException {
        final String attributeKey = pRelation.getAttributeKey();
        if (attributeKey == null) {
            throw new IllegalStateException("The relations attributeKey must not be null.");
        }
        final List<Association> result = new ArrayList<Association>();
        final Collection<Association> assocs = Generics.cast(pRegistryObject.getAssociations());
        for (Association assoc : assocs) {
            final Slot slot = assoc.getSlot("{http://namespaces.CentraSite.com/Schema}attributeKey");
            if (slot != null) {
                final Iterator<?> iter = slot.getValues().iterator();
                if (iter.hasNext()  &&  attributeKey.equals(iter.next())) {
                    result.add(assoc);
                }
            }
        }
        return result;
    }

    private boolean isSame(Collection<CachedAsset> pAssets, Collection<RegistryObject> pTargets) throws JAXRException {
        if (pAssets.size() != pTargets.size()) {
            return false;
        }
        for (CachedAsset asset : pAssets) {
        	boolean found = false;
        	for (RegistryObject target : pTargets) {
        		if (target.getKey().equals(asset.getRegistryObject().getKey())) {
        			found = true;
        			break;
        		}
        	}
        	if (!found) {
        		return false;
        	}
        }
        return true;
    }

    private void setRelations(CachedAsset pAsset) throws JAXRException {
        final CachedAssetType cachedAssetType = pAsset.getType();
        final ROType assetType = cachedAssetType.getROType();
        final Map<String,RORelation> attrs = getRelations(assetType);
        final Attributes beanAttributes = pAsset.getBean().getAttributes();
        if (beanAttributes != null) {
            final List<AbstractAttribute> beanAttrs = beanAttributes.getSlotOrRelation();
            for (AbstractAttribute beanAttr : beanAttrs) {
                if (!(beanAttr instanceof AbstractRelation)) {
                    continue;
                }
                final AbstractRelation beanRelation = (AbstractRelation) beanAttr;
                final List<AbstractReference> targetRefs = beanRelation.getTarget();
                final RORelation roRelation = attrs.remove(beanRelation.getName());
                if (roRelation == null) {
                    if (assetType.getAttribute(beanRelation.getName()) == null) {
                        logError("Unknown relation " + beanRelation.getName()
                                + " of type " + assetType.getQName());
                    } else {
                        logError("Attribute " + beanRelation.getName()
                                + " of type " + assetType.getQName()
                                + " is no relation.");
                    }
                    continue;
                }
                final Collection<RegistryObject> targets = mAcc.getRelationValues(roRelation, pAsset.getRegistryObject());
                final List<CachedAsset> targetAssets = getTargets(cachedAssetType.getAssetType(), roRelation, targetRefs);
                if (targetAssets == null  ||  isSame(targetAssets, targets)) {
                    continue;
                }
                final List<RegistryObject> ros = new ArrayList<RegistryObject>(targetAssets.size());
                for (CachedAsset target : targetAssets) {
                	ros.add(target.getRegistryObject());
                }
                mAcc.setRelationValues(roRelation, pAsset.getRegistryObject(), ros);
                pAsset.setModified();
            }
        }
        for (RORelation attr : attrs.values()) {
            final List<Association> associations = getAssociations(attr, pAsset.getRegistryObject());
            if (associations.size() > 0) {
                pAsset.getRegistryObject().removeAssociations(associations);
                pAsset.setModified();
            }
        }
    }

    private CachedAsset getReferencedObject(CachedAssetType pType, AbstractTypedReference pRef) throws JAXRException {
        switch (pType.getAssetType().getType()) {
            case key:
                return getReferencedObjectByKey(pType, pRef);
            case name:
                return getReferencedObjectByName(pType, pRef);
            case slotValue:
                return getReferencedObjectBySlotValue(pType, pRef);
            default:
                throw new IllegalStateException("Invalid type: " + pType.getAssetType().getType());
        }
    }

    private CachedAsset getReferencedObjectBySlotValue(CachedAssetType pType, AbstractTypedReference pRef) throws JAXRException {
        final SlotValueAssetType assetType = (SlotValueAssetType) pType.getAssetType();
        final String slotValue = pRef.getSlotValue();
        if (slotValue == null) {
            logError("Identification for " + pType.getAssetType().getQName()
                + " is name, but slot value of referenced object is missing.");
            return null;
        }
        final Object slotValueObject = asSlotObject(assetType.getSlotType(), slotValue);
        final SlotValueKey key = new SlotValueKey(assetType, slotValueObject); 
        CachedAsset cachedAsset = data.getAsset(key);
        if (cachedAsset == null) {
            cachedAsset = new CachedAsset(pType, key, null);
            data.addAsset(key, cachedAsset);
            final RegistryObject ro = findRegistryObjectBySlotValue(cachedAsset, Collections.singletonList(slotValueObject), false);
            if (ro == null) {
                return null;
            }
            cachedAsset.setRegistryObject(ro);
        }
        return cachedAsset;
    }

    private CachedAsset getReferencedObjectByName(CachedAssetType pType, AbstractTypedReference pRef) throws JAXRException {
        final String name = pRef.getName();
        if (name == null) {
            logError("Identification for " + pType.getAssetType().getQName()
                + " is name, but name of referenced object is missing.");
            return null;
        }
        final String localeName = pRef.getLocale();
        final Locale locale = localeName == null ? null : RegistryObjects.asLocale(localeName);
        final NameKey key = new NameKey(pType.getAssetType(), name, locale);
        CachedAsset cachedAsset = data.getAsset(key);
        if (cachedAsset == null) {
            cachedAsset = new CachedAsset(pType, key, null);
            data.addAsset(key, cachedAsset);
            final RegistryObject ro = findRegistryObjectByName(cachedAsset, Collections.singletonList(name), false);
            if (ro == null) {
                return null;
            }
            cachedAsset.setRegistryObject(ro);
        }
        return cachedAsset;
    }
    
    private CachedAsset getReferencedObjectByKey(CachedAssetType pType,
            AbstractTypedReference pRef) throws JAXRException {
        final String id = pRef.getKey();
        if (id == null) {
            logError("Identification for " + pType.getAssetType().getQName()
                + " is key, but key of referenced object is missing.");
            return null;
        }
        final KeyKey key = new KeyKey(pType.getAssetType(), id);
        CachedAsset cachedAsset = data.getAsset(key);
        if (cachedAsset == null) {
            cachedAsset = new CachedAsset(pType, key, null);
            data.addAsset(key, cachedAsset);
            final RegistryObject ro = findRegistryObjectByKey(cachedAsset);
            if (ro == null) {
                return null;
            }
            cachedAsset.setRegistryObject(ro);
        }
        return cachedAsset;
    }
    
    private void setOrganization(CachedAsset pAsset) throws JAXRException {
        final AbstractTypedReference orgRef = pAsset.getBean().getSubmittingOrganization();
        if (orgRef == null) {
            return;
        }
        final CachedAssetType orgType = data.getAssetType(QNAME_ORGANIZATION);
        final CachedAsset org = getReferencedObject(orgType, orgRef);
        if (org != null) {
            final Organization o = (Organization) org.getRegistryObject();
            if (!o.getKey().equals(pAsset.getRegistryObject().getKey())) {
                ((CentraSiteRegistryObject) pAsset.getRegistryObject()).setSubmittingOrganization(o);
                pAsset.setModified();
            }
        }
    }
    
    private void setValues(List<CachedAsset> pAssets) throws JAXRException {
        for (CachedAsset asset : pAssets) {
            setNames(asset);
            setDescriptions(asset);
            setSlots(asset);
            setOrganization(asset);
            setRelations(asset);
        }
    }

    /**
     * Called to create a registry object, which matches the descriptions given by
     * the parameters. This method is invoked, if a new registry object must be
     * inserted.
     * @param pAssetType The beans asset type.
     * @param pBean The bean matching the registry object.
     * @return The created registry object.
     * @throws JAXRException An error occurred while creating the registry object.
     */
    protected RegistryObject create(CachedAssetType pAssetType, AbstractRegistryObject pBean) throws JAXRException {
        final String qName = pAssetType.getROType().getQName().toString();
        final BusinessLifeCycleManager blcm = facade.getBusinessLifeCycleManager();
        final InternationalString is = facade.getBusinessLifeCycleManager().createInternationalString();
        final RegistryObject ro;
        if (LifeCycleManager.ORGANIZATION.equals(qName)) {
            ro = blcm.createOrganization(is);
        } else if (LifeCycleManager.REGISTRY_ENTRY.equals(qName)) {
            ro = (RegistryEntry) blcm.createObject(LifeCycleManager.REGISTRY_ENTRY);
            ro.setName(is);
        } else if (LifeCycleManager.USER.equals(qName)) {
            ro = blcm.createUser();
            ro.setName(is);
        } else if (LifeCycleManager.SERVICE.equals(qName)) {
            ro = blcm.createService(is);
        } else {
            ro = (RegistryEntry) blcm.createObject(LifeCycleManager.REGISTRY_ENTRY);
            ro.addClassification(blcm.createClassification(pAssetType.getObjectType()));
            ro.setName(is);
        }
        return ro;
    }

    private RegistryObject findRegistryObjectByKey(CachedAsset pAsset) throws JAXRException {
        final KeyKey key = (KeyKey) pAsset.getKey();
        final RegistryObject ro = facade.getBusinessQueryManager().getRegistryObject(key.getKey());
        if (ro == null) {
            logError("No registry object found for key " + key.getKey()
                    + " and asset type " + key.getType().getQName());
            return null;
        }
        return ro;
    }

    private RegistryObject findRegistryObjectByName(CachedAsset pAsset) throws JAXRException {
        final List<String> names = new ArrayList<String>();
        for (AbstractInternationalString is : pAsset.getBean().getNames().getName()) {
            names.add(is.getValue());
        }
        return findRegistryObjectByName(pAsset, names, true);
        
    }

    private RegistryObject findRegistryObjectByName(CachedAsset pAsset, List<String> pNames, boolean pCreate) throws JAXRException {
        final AssetType assetType = pAsset.getType().getAssetType();
        final QName qName = pAsset.getType().getAssetType().getQName();
        final String query;
        {
	        final StringBuilder sb = new StringBuilder();
	        sb.append("DECLARE NAMESPACE ns='");
	        sb.append(qName.getNamespaceURI());
	        sb.append("'\nFROM ns:");
	        sb.append(qName.getLocalPart());
	        sb.append(" ro WHERE ro.name() IN (");
	        for (int i = 0;  i < pNames.size();  i++) {
	        	if (i > 0) {
	        		sb.append(",");
	        	}
	        	sb.append("?");
	        }
	        sb.append(")");
	        query = sb.toString();
        }
        final List<RegistryObject> ros = mAcc.executeQuery(query, (Object[]) pNames.toArray());
        switch (ros.size()) {
            case 0:
                if (!pCreate) {
                    final StringBuilder sb = new StringBuilder();
                    sb.append("No registry object found for name ");
                    for (int i = 0;  i < pNames.size();  i++) {
                        if (i > 0) {
                            sb.append('|');
                        }
                        sb.append(pNames.get(i));
                    }
                    sb.append(" and asset type ");
                    sb.append(assetType.getQName());
                    logError(sb.toString());
                    return null;
                }
                pAsset.setOperation(CachedAsset.Operation.insert);
                return create(pAsset.getType(), pAsset.getBean());
            case 1:
                pAsset.setOperation(CachedAsset.Operation.update);
                return ros.get(0);
            default:
                final StringBuilder sb = new StringBuilder();
                if (pNames.size() > 1) {
                    sb.append("The names ");
                    for (int i = 0;  i < pNames.size();  i++) {
                        if (i > 0) {
                            sb.append("|");
                        }
                        sb.append(pNames.get(i));
                    }
                    sb.append(" are ");
                } else {
                    sb.append("The name ");
                    sb.append(pNames.get(0));
                    sb.append(" is ");
                }
                sb.append("matching multiple objects of type ");
                sb.append(assetType.getQName());
                sb.append(": ");
                for (int i = 0;  i < ros.size();  i++) {
                    if (i > 0) {
                        sb.append(", ");
                    }
                    sb.append(ros.get(i).getKey().getId());
                }
                logError(sb.toString());
                return null;
        }
    }

    private RegistryObject findRegistryObjectBySlotValue(CachedAsset pAsset, boolean pCreate) throws JAXRException {
        final SlotValueAssetType slotValueAssetType = (SlotValueAssetType) pAsset.getType().getAssetType();
        final AbstractSlot slot = findSlot(pAsset.getBean(), slotValueAssetType.getSlotName());
        final List<Object> slotValues = new ArrayList<Object>();
        for (String s : slot.getValue()) {
            slotValues.add(asSlotObject(slotValueAssetType.getSlotType(), s));
        }
        return findRegistryObjectBySlotValue(pAsset, slotValues, pCreate);
    }

    private RegistryObject findRegistryObjectBySlotValue(CachedAsset pAsset, List<Object> pSlotValues, boolean pCreate) throws JAXRException {
        final SlotValueAssetType slotValueAssetType = (SlotValueAssetType) pAsset.getType().getAssetType();
        final AbstractSlot slot = findSlot(pAsset.getBean(), slotValueAssetType.getSlotName());
        final QName qName = pAsset.getType().getROType().getQName();
        final StringBuilder qsb = new StringBuilder();
        if (!"".equals(qName.getNamespaceURI())) {
            qsb.append("declare namespace ns='");
            qsb.append(qName.getNamespaceURI());
            qsb.append("'\n");
        }
        qsb.append("FROM ");
        qsb.append(qName.getLocalPart());
        qsb.append(" AS o WHERE ");
        final List<Object> params = new ArrayList<Object>();
        for (int i = 0;  i < slot.getValue().size();  i++) {
            if (i > 0) {
                qsb.append(" OR ");
            }
            qsb.append("o.");
            qsb.append(slotValueAssetType.getSlotName());
            qsb.append("=?");
            params.add(asSlotObject(slotValueAssetType.getSlotType(), slot.getValue().get(i)));
        }
        final List<RegistryObject> ros = mAcc.executeQuery(qsb.toString(), params.toArray());
        switch (ros.size()) {
            case 0:
                if (!pCreate) {
                    final StringBuilder sb = new StringBuilder();
                    sb.append("No registry object found for slot name ");
                    sb.append(slotValueAssetType.getSlotName());
                    sb.append(" and slot value ");
                    for (int i = 0;  i < pSlotValues.size();  i++) {
                        if (i > 0) {
                            sb.append('|');
                        }
                        sb.append(asSlotString(slotValueAssetType.getSlotType(), pSlotValues.get(i)));
                    }
                    sb.append(" and asset type ");
                    sb.append(slotValueAssetType.getQName());
                    logError(sb.toString());
                    return null;
                }
                pAsset.setOperation(CachedAsset.Operation.insert);
                return create(pAsset.getType(), pAsset.getBean());
            case 1:
                pAsset.setOperation(CachedAsset.Operation.update);
                return ros.get(0);
            default:
                final StringBuilder sb2 = new StringBuilder();
                if (pSlotValues.size() > 1) {
                    sb2.append("The slot values ");
                    for (int i = 0;  i < pSlotValues.size();  i++) {
                        if (i > 0) {
                            sb2.append("|");
                        }
                        sb2.append(pSlotValues.get(i));
                    }
                    sb2.append(" are ");
                } else {
                    sb2.append("The slot value ");
                    sb2.append(pSlotValues.get(0));
                    sb2.append(" is ");
                }
                sb2.append("matching multiple objects of type ");
                sb2.append(slotValueAssetType.getQName());
                sb2.append(": ");
                for (int i = 0;  i < ros.size();  i++) {
                    if (i > 0) {
                        sb2.append(", ");
                    }
                    sb2.append(ros.get(i).getKey().getId());
                }
                logError(sb2.toString());
                return null;
        }
    }
    
    protected void findRegistryObjects(List<CachedAsset> pAssets) throws JAXRException {
        for (CachedAsset asset : pAssets) {
            if (asset.getRegistryObject() != null) {
                continue;
            }
            final RegistryObject ro;
            final AssetType assetType = asset.getKey().getType();
            switch (assetType.getType()) {
                case key:
                    ro = findRegistryObjectByKey(asset);
                    break;
                case name:
                    ro = findRegistryObjectByName(asset);
                    break;
                case slotValue:
                    ro = findRegistryObjectBySlotValue(asset, true);
                    break;
                default:
                    throw new IllegalStateException("Invalid type: "
                            + assetType.getType());
            }
            asset.setRegistryObject(ro);
            if (asset.getBean().isDelete()) {
            	switch (asset.getOperation()) {
            		case insert:
            			logError("Unable to delete non-existing asset "
            					+ asset.getKey());
            			break;
            		case update:
            			asset.setOperation(Operation.delete);
            			break;
            		default:
                        throw new IllegalStateException("Invalid type: "
                                + assetType.getType());
            	}
            }
        }
    }
    
    private CachedAsset getAssetByKey(CachedAssetType pAssetType, AbstractRegistryObject pAsset) throws JAXRException {
        final String key = pAsset.getKey();
        if (key == null) {
            /* Special case: We need an object to create an instance of KeyKey, so we
             * create it now.
             */
            final RegistryObject ro = create(pAssetType, pAsset);
            final KeyKey keyKey = new KeyKey(pAssetType.getAssetType(), ro.getKey().getId());
            final CachedAsset asset = new CachedAsset(pAssetType, keyKey, pAsset);
            asset.setOperation(CachedAsset.Operation.insert);
            data.addAsset(keyKey, asset);
            return asset;
        }
        final KeyKey keyKey = new KeyKey(pAssetType.getAssetType(), key);
        CachedAsset asset = new CachedAsset(pAssetType, keyKey, pAsset);
        asset.setOperation(CachedAsset.Operation.update);
        data.addAsset(keyKey, asset);
        return asset;
    }

    private CachedAsset getAssetByName(CachedAssetType pAssetType, AbstractRegistryObject pAsset) {
        CachedAsset result = null;
        final List<AbstractInternationalString> names = pAsset.getNames().getName();
        for (AbstractInternationalString name : names) {
            final NameKey nameKey = new NameKey(pAssetType.getAssetType(), name.getValue(), RegistryObjects.asLocale(name.getLocale()));
            if (result == null) {
                result = new CachedAsset(pAssetType, nameKey, pAsset);
            }
            data.addAsset(nameKey, result);
        }
        return result;
    }

    private String asSlotString(ROAttribute.Type pType, Object pValue) {
        switch (pType) {
            case bool:
            case ipv4Address:
            case ipv6Address:
            case emailAddress:
            case string:
            case uri:
            case integer:
            case longInteger:
            case number:
                return pValue.toString();
            case date:
                return dateFormat.format(pValue);
            case dateTime:
                return dateTimeFormat.format(pValue);
            case time:
                return timeFormat.format(pValue);
            case duration:
            case classification:
            case relation:
            default:
                throw new IllegalStateException("Invalid slot type: " + pType);
        }
    }

    private Object asSlotObject(ROAttribute.Type pType, String pValue) {
        switch (pType) {
            case bool:
                return Boolean.valueOf(pValue);
            case date:
                try {
                    return dateFormat.parseObject(pValue);
                } catch (ParseException e) {
                    return null;
                }
            case dateTime:
                try {
                    return dateTimeFormat.parseObject(pValue);
                } catch (ParseException e) {
                    return null;
                }
            case time:
                try {
                    return timeFormat.parseObject(pValue);
                } catch (ParseException e) {
                    return null;
                }
            case ipv4Address:
            case ipv6Address:
            case emailAddress:
            case string:
            case uri:
                return pValue;
            case integer:
                try {
                    return Integer.valueOf(pValue);
                } catch (Exception e) {
                    return null;
                }
            case longInteger:
                try {
                    return Long.valueOf(pValue);
                } catch (Exception e) {
                    return null;
                }
            case number:
                try {
                    return Double.valueOf(pValue);
                } catch (Exception e) {
                    return null;
                }
            case duration:
            case classification:
            case relation:
            default:
                throw new IllegalStateException("Invalid slot type: " + pType.name());
        }
    }

    private AbstractSlot findSlot(AbstractRegistryObject pAsset, String pSlotName) {
        Attributes attributes = pAsset.getAttributes();
        if (attributes != null) {
            for (AbstractAttribute attribute : attributes.getSlotOrRelation()) {
                if (attribute instanceof AbstractSlot) {
                    final AbstractSlot slot = (AbstractSlot) attribute;
                    if (pSlotName.equals(slot.getName())) {
                        return slot;
                    }
                }
            }
        }
        return null;
    }
    
    private CachedAsset getAssetBySlotValue(CachedAssetType pAssetType, AbstractRegistryObject pAsset) {
        final SlotValueAssetType slotValueAssetType = (SlotValueAssetType) pAssetType.getAssetType();
        CachedAsset result = null;
        final AbstractSlot slot = findSlot(pAsset, slotValueAssetType.getSlotName());
        if (slot != null) {
            final List<String> values = slot.getValue();
            if (values.isEmpty()) {
                logError("No slot values present for unique slot "
                        + slot.getName() + " " + pAssetType.getAssetType().getQName());
                return null;
            }
            for (String value : values) {
                Object o = asSlotObject(slotValueAssetType.getSlotType(), value);
                if (o == null) {
                    logError("Invalid slot value " + value
                            + " for unique slot " + slot.getName()
                            + " of slot type " + slotValueAssetType.getSlotType().name()
                            + " and asset type " + pAssetType.getAssetType().getQName());
                    return null;
                }
                final SlotValueKey key = new SlotValueKey(slotValueAssetType, o);
                if (result == null) {
                    result = new CachedAsset(pAssetType, key, pAsset);
                }
                data.addAsset(key, result);
            }
        }
        if (result == null) {
            logError("No value given for unique slot " + slotValueAssetType.getSlotName()
                    + " of " + pAssetType.getAssetType().getQName());
        }
        return result;
    }

    private List<CachedAsset> readAssets(Model pModel) throws JAXRException {
        final List<CachedAsset> cacheEntries = new ArrayList<CachedAsset>();
        final Assets assets = pModel.getAssets();
        if (assets != null) {
            for (AbstractRegistryObject assetDefinition : assets.getRegistryObjectOrAsset()) {
                final QName qName = asQName(assetDefinition.getType());
                final CachedAssetType assetType = data.getAssetType(qName);
                final CachedAsset asset;
                switch (assetType.getAssetType().getType()) {
                    case key:
                        asset = getAssetByKey(assetType, assetDefinition);
                        break;
                    case name:
                        asset = getAssetByName(assetType, assetDefinition);
                        break;
                    case slotValue:
                        asset = getAssetBySlotValue(assetType, assetDefinition);
                        break;
                    default:
                        throw new IllegalStateException("Invalid key type: " + assetType.getAssetType().getType());
                }
                cacheEntries.add(asset);
            }
        }
        return cacheEntries;
    }

    protected void saveAssets(List<CachedAsset> pAssets) throws JAXRException {
    	final List<RegistryObject> ros = new ArrayList<RegistryObject>();
    	final Set<javax.xml.registry.infomodel.Key> keys = new HashSet<javax.xml.registry.infomodel.Key>();
    	final Set<javax.xml.registry.infomodel.Key> deletedRos = new HashSet<javax.xml.registry.infomodel.Key>();
    	final List<Association> assocs = new ArrayList<Association>();
    	for (CachedAsset asset : pAssets) {
            final RegistryObject ro = asset.getRegistryObject();
            if (ro == null) {
                throw new IllegalStateException("The registry object must not be null!");
            }
            switch (asset.getOperation()) {
                case delete:
                	deletedRos.add(ro.getKey());
                    assocs.addAll(facade.getReferencingAssociations(ro));
                    break;
                case insert:
                	if (!keys.contains(ro.getKey())) {
                		ros.add(ro);
                		keys.add(ro.getKey());
                	}
                    break;
                case update:
                	if (!keys.contains(ro.getKey())) {
                		ros.add(ro);
                		keys.add(ro.getKey());
                	}
                    break;
                default:
                    throw new IllegalStateException("Invalid operation: " + asset.getOperation());
            }
        }
    	for (Association assoc : assocs) {
    		final RegistryObject source = assoc.getSourceObject();
    		source.removeAssociation(assoc);
        	if (!keys.contains(source.getKey())) {
        		ros.add(source);
        		keys.add(source.getKey());
        	}
    	}
    	final BusinessLifeCycleManager blcm = facade.getBusinessLifeCycleManager();
    	blcm.saveObjects(ros);
        blcm.deleteObjects(deletedRos);
    }
    
    public void read(Model pModel) throws JAXRException {
        Connection conn = provider.getConnection();
        try {
            facade = new SimpleModelDrivenRegistryFacade(conn);
            mAcc = facade.getModelAccessor();
            metaModel = facade.getMetaModel();
            data = new ImporterData(facade);
            readIdentifications(pModel);
            if (!hasErrors) {
                final List<CachedAsset> entries = readAssets(pModel);
                if (!hasErrors) {
                    findRegistryObjects(entries);
                    if (!hasErrors) {
                        setValues(entries);
                        if (!hasErrors) {
                            saveAssets(entries);
                        }
                    }
                }
            }
            conn.close();
            conn = null;
        } finally {
            if (conn != null) { try { conn.close(); } catch (Throwable t) { /* Ignore me */ } }
        }
    }
}
