package net.sf.csutils.impexp.impimpl;

import java.util.HashMap;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;

import javax.xml.registry.JAXRException;
import javax.xml.registry.infomodel.Concept;
import javax.xml.registry.infomodel.RegistryObject;

import net.sf.csutils.core.model.QName;
import net.sf.csutils.core.model.ROMetaModel;
import net.sf.csutils.core.model.ROType;
import net.sf.csutils.core.registry.ModelDrivenRegistryFacade;
import net.sf.csutils.importer.vo.AbstractRegistryObject;


/**
 * This object is used as the importers data container.
 */
public class ImporterData {
    public static class CachedAssetType {
        private final AssetType assetType;
        private final Concept objectType;
        private final ROType roType;

        /**
         * Creates a new instance.
         */
        CachedAssetType(AssetType pAssetType, ROType pROType, Concept pObjectType) {
            assetType = pAssetType;
            objectType = pObjectType;
            roType = pROType;
        }

        /**
         * Returns the object type concept.
         * @return The object type concept.
         */
        public Concept getObjectType() {
            return objectType;
        }

        /**
         * Returns the object types meta data.
         * @return The object types meta data
         */
        public ROType getROType() {
            return roType;
        }

        /**
         * Returns the asset type.
         * @return The asset type
         */
        public AssetType getAssetType() {
            return assetType;
        }
    }

    public static class CachedAsset {
        public enum Operation {
            insert,
            update,
            delete
        }
        private final CachedAssetType assetType;
        private final Key key;
        private RegistryObject ro;
        private final AbstractRegistryObject bean;
        private Operation operation;
        private boolean modified;

        public CachedAsset(CachedAssetType pAssetType, Key pKey, AbstractRegistryObject pBean) {
            assetType = pAssetType;
            key = pKey;
            bean = pBean;
        }

        public Key getKey() {
            return key;
        }

        public void setRegistryObject(RegistryObject pRegistryObject) {
            ro = pRegistryObject;
        }
        
        public RegistryObject getRegistryObject() {
            return ro;
        }

        public AbstractRegistryObject getBean() {
            return bean;
        }

        public CachedAssetType getType() {
            return assetType;
        }

        /**
         * Returns the operation.
         * @return The operation
         */
        public Operation getOperation() {
            return operation;
        }

        /**
         * Sets the operation.
         * @param pOperation The operation
         */
        public void setOperation(Operation pOperation) {
            operation = pOperation;
        }

        /**
         * Returns, whether the asset is modified.
         * @return True, if the asset is modified.
         */
        public boolean isModified() {
            return modified;
        }

        /**
         * Marks the asset as modified.
         */
        public void setModified() {
            modified = true;
        }
    }

    private final ModelDrivenRegistryFacade facade;
    private final ROMetaModel metaModel;
    private final Map<QName,CachedAssetType> types = new HashMap<QName,CachedAssetType>();
    private final NavigableMap<Key,CachedAsset> cache = new TreeMap<Key,CachedAsset>();

    public ImporterData(ModelDrivenRegistryFacade pFacade) {
        facade = pFacade;
        metaModel = pFacade.getMetaModel();
    }
    
    public CachedAssetType addAssetType(AssetType pType) throws JAXRException {
        final ROType roType = metaModel.getROType(pType.getQName());
        if (roType == null) {
            throw new IllegalArgumentException("Invalid, or unknown asset type: " + pType.getQName());
        }
        final Concept concept = facade.findObjectType(pType.getQName());
        if (concept == null) {
            throw new IllegalArgumentException("Invalid, or unknown asset type: " + pType.getQName());
        }
        final CachedAssetType entry = new CachedAssetType(pType, roType, concept);
        types.put(pType.getQName(), entry);
        return entry;
    }

    public void addAsset(Key pKey, CachedAsset pEntry) {
        cache.put(pKey, pEntry); 
    }

    private boolean isMatching(NameKey pKey1, Key pKey2) {
    	return pKey2 != null  &&  pKey2 instanceof NameKey
    		&&  pKey1.getName().equals(((NameKey) pKey2).getName());
    }
    
    /**
     * Returns the asset with the given key.
     * @param pKey Asset key to search for
     * @return Asset with the given key, if any, or null.
     * @throws IllegalStateException Multiple matching assets have been found.
     */
    public CachedAsset getAsset(Key pKey) {
    	final CachedAsset asset = cache.get(pKey);
    	if (asset == null  &&  pKey instanceof NameKey) {
    		final NameKey nameKey = (NameKey) pKey;
    		final NavigableMap<Key,CachedAsset> map = cache.tailMap(pKey, false);
    		if (!map.isEmpty()) {
    			final Map.Entry<Key,CachedAsset> firstEntry = map.firstEntry();
    			final Key firstKey = firstEntry.getKey();
    			if (isMatching(nameKey, firstKey)) {
    				final Key nextKey = map.higherKey(firstKey);
    				if (isMatching(nameKey, nextKey)) {
    					throw new IllegalStateException("Multiple matching assets: "
    							+ firstKey + ", " + nextKey);
    				}
    				return firstEntry.getValue();
    			}
    		}
    	}
    	return asset;
    }

    public CachedAssetType getAssetType(QName pQName) throws JAXRException {
        CachedAssetType type = types.get(pQName);
        if (type == null) {
            addAssetType(new AssetType(AssetType.Type.key, pQName));
            type = types.get(pQName);
            if (type == null) {
                throw new IllegalStateException("No asset type created for " +pQName);
            }
        }
        return type;
    }
}
