package net.sf.csutils.impexp.jdbc;

import java.io.IOException;
import java.math.BigDecimal;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.registry.JAXRException;

import net.sf.csutils.core.model.QName;
import net.sf.csutils.core.model.ROAttribute;
import net.sf.csutils.core.model.ROAttribute.Type;
import net.sf.csutils.core.model.ROMetaModel;
import net.sf.csutils.core.model.RORelation;
import net.sf.csutils.core.model.ROType;
import net.sf.csutils.core.model.impl.ROMetaModelWriter;
import net.sf.csutils.core.model.impl.StaticROMetaModel;
import net.sf.csutils.core.model.impl.StaticRORelation;
import net.sf.csutils.core.model.impl.StaticROSlot;
import net.sf.csutils.core.model.impl.StaticROType;
import net.sf.csutils.core.utils.JaxbReader;
import net.sf.csutils.core.utils.JaxbWriter;
import net.sf.csutils.importer.vo.AbstractIdentification;
import net.sf.csutils.importer.vo.AbstractInternationalString;
import net.sf.csutils.importer.vo.AbstractRegistryEntry;
import net.sf.csutils.importer.vo.AbstractRegistryObject;
import net.sf.csutils.importer.vo.AbstractRegistryObject.Attributes;
import net.sf.csutils.importer.vo.AbstractRegistryObject.Descriptions;
import net.sf.csutils.importer.vo.AbstractRegistryObject.Names;
import net.sf.csutils.importer.vo.Model.Identifications;
import net.sf.csutils.importer.vo.Model.Identifications.UniqueSlot;
import net.sf.csutils.importer.vo.AbstractSlot;
import net.sf.csutils.importer.vo.Model;
import net.sf.csutils.jdbcExporter.vo.JdbcExport;
import net.sf.csutils.jdbcExporter.vo.ObjectFactory;

import org.apache.log4j.Logger;


/**
 * The {@link JdbcDbExporter} is used to export the contents of an SQL database,
 * which is accessible via JDBC, to files, which are suitable for the importer.
 */
public class JdbcDbExporter {
	private static final JaxbReader<JdbcExport> jdbcExportReader = new JaxbReader<JdbcExport>(JdbcExport.class);
	private static final JaxbWriter<JdbcExport> jdbcExportWriter = new JaxbWriter<JdbcExport>(JdbcExport.class,
			new javax.xml.namespace.QName("http://namespaces.csutils.sf.net/importer/jdbcExporter/1.0.0", "jdbcExport"));
	private static final String NS_IMPORTER = "http://namespaces.csutils.sf.net/importer/model/1.0.0";
	private static final javax.xml.namespace.QName QNAME_UNIQUE_SLOT = new javax.xml.namespace.QName(NS_IMPORTER, "uniqueSlot");
	private static final javax.xml.namespace.QName QNAME_UNIQUE_NAME = new javax.xml.namespace.QName(NS_IMPORTER, "uniqueName");
	private static final javax.xml.namespace.QName QNAME_UNIQUE_KEY = new javax.xml.namespace.QName(NS_IMPORTER, "uniqueKey");
	private static final JaxbWriter<Model> modelWriter = new JaxbWriter<Model>(Model.class,
			new javax.xml.namespace.QName(NS_IMPORTER, "model"));
	private static final ObjectFactory OF = new ObjectFactory();
	private static final net.sf.csutils.importer.vo.ObjectFactory OFM = new net.sf.csutils.importer.vo.ObjectFactory();
	private static final Logger log = Logger.getLogger(JdbcDbExporter.class);

	private String catalog, schema;
	private JdbcExport jdbcExport;
	private JdbcDbReader reader;

	public String getCatalog() {
		return catalog;
	}

	public void setCatalog(String catalog) {
		this.catalog = catalog;
	}

	public String getSchema() {
		return schema;
	}

	public void setSchema(String schema) {
		this.schema = schema;
	}

	protected JdbcDbForeignKey checkRelationshipColumn(JdbcDbColumn pColumn) {
		final JdbcDbTable table = pColumn.getTable();
		for (final JdbcDbForeignKey foreignKey : table.getForeignKeys()) {
			final List<JdbcDbColumn> localColumns = foreignKey.getLocalColumns();
			final List<JdbcDbColumn> foreignColumns = foreignKey.getForeignColumns();
			if (localColumns == null  ||  foreignColumns == null) {
				continue;
			}
			if (localColumns.isEmpty()  ||  foreignColumns.isEmpty()) {
				continue;
			}
			if (localColumns.size() != foreignColumns.size()) {
				continue;
			}
			if (localColumns.size() > 1) {
				continue;
			}
			final JdbcDbColumn col = localColumns.get(0);
			if (col.equals(pColumn)) {
				return foreignKey;
			}
		}
		return null;
	}

	protected RORelation write(JdbcDbForeignKey pColumn) {
		final List<JdbcDbColumn> localColumns = pColumn.getLocalColumns();
		final List<JdbcDbColumn> foreignColumns = pColumn.getForeignColumns();
		if (localColumns == null  ||  foreignColumns == null) {
			log.error("Neither of the column sets of a foreign key may be null. ("
					+ JdbcDbReader.asString(pColumn) + ")");
			return null;
		}
		if (localColumns.isEmpty()  ||  foreignColumns.isEmpty()) {
			log.error("Neither of the column sets of a foreign key may be empty. ("
					+ JdbcDbReader.asString(pColumn) + ")");
			return null;
		}
		if (localColumns.size() != foreignColumns.size()) {
			log.error("The column sets of a forein key must have the same number of columns. ("
					+ JdbcDbReader.asString(pColumn) + ", "
					+ localColumns.size() + " != " + foreignColumns.size() + ")");
			return null;
		}
		if (localColumns.size() > 1) {
			log.warn("Can't create a relationship for a foreign key with multiple columns. ("
					+ JdbcDbReader.asString(pColumn) + ")");
			return null;
		}
		StaticRORelation relation = new StaticRORelation();
		relation.setMinOccurs(localColumns.get(0).isNullable() ? 0 : 1);
		relation.setMaxOccurs(1);
		final String associationType = pColumn.getAssociationType();
		relation.setAssociationType(associationType);
		relation.setName(associationType);
		return relation;
	}

	protected ROAttribute write(JdbcDbColumn pColumn) {
		final int sqlType = pColumn.getType();
		final Type type;
		switch (sqlType) {
			case Types.VARCHAR:
				type = Type.string;
				break;
			case Types.INTEGER:
				type = Type.integer;
				break;
			default:
				log.error("Invalid SQL type " + sqlType
						+ " for column " + pColumn.getName()
						+ " of table " + JdbcDbReader.asString(pColumn.getTable()));
				return null;
		}
		final JdbcExport.Table.Column col = getJdbcExportColumn(pColumn);
		final StaticROSlot attr = new StaticROSlot();
		String name = col.getAttributeName();
		if (name == null  ||  name.length() == 0) {
			name = col.getName();
		}
		attr.setName(name);
		attr.setMinOccurs(pColumn.isNullable() ? 0 : 1);
		attr.setMaxOccurs(1);
		attr.setType(type);
		return attr;
	}

	protected JdbcExport.Table getJdbcExportTable(JdbcDbTable pTable) {
		for (JdbcExport.Table table : jdbcExport.getTable()) {
			if (table.getName().equalsIgnoreCase(pTable.getName())) {
				return table;
			}
		}
		throw new IllegalStateException("No such export table: " + pTable.getName());
	}

	protected JdbcExport.Table.Column getJdbcExportColumn(JdbcDbColumn pColumn) {
		final JdbcExport.Table table = getJdbcExportTable(pColumn.getTable());
		for (JdbcExport.Table.Column col : table.getColumn()) {
			if (col.getName().equalsIgnoreCase(pColumn.getName())) {
				return col;
			}
		}
		throw new IllegalStateException("No such export column: " + table.getName()
				+ "." + pColumn.getName());
	}
	
	protected ROType write(JdbcDbTable pTable) {
		final StaticROType type = new StaticROType();
		type.setQName(pTable.getQName());
		final Map<String,ROAttribute> attrs = new HashMap<String,ROAttribute>();
		for (JdbcDbColumn col : pTable.getColumns()) {
			if (pTable.getNameColumn() != null  &&
					pTable.getNameColumn().equals(col)) {
				continue;
			}
			if (pTable.getDescriptionColumn() != null  &&
					pTable.getDescriptionColumn().equals(col)) {
				continue;
			}
			if (checkRelationshipColumn(col) != null) {
				continue;
			}
			final ROAttribute attr = write(col);
			if (attr != null) {
				attrs.put(attr.getName(), attr);
			}
		}
		for (JdbcDbForeignKey key : pTable.getForeignKeys()) {
			final RORelation relation = write(key);
			if (relation != null) {
				attrs.put(relation.getName(), relation);
			}
		}
		type.setAttributes(attrs);
		return type;
	}

	protected ROMetaModel write(JdbcDbMetaData pMetaData) throws JAXRException {
		final StaticROMetaModel model = new StaticROMetaModel();
		final Map<QName,ROType> types = new HashMap<QName,ROType>();
		for (JdbcDbTable table : pMetaData.getTables()) {
			final ROType type = write(table);
			types.put(table.getQName(), type);
		}
		for (JdbcDbTable table : pMetaData.getTables()) {
			final ROType type = types.get(table.getQName());
			for (JdbcDbForeignKey foreignKey : table.getForeignKeys()) {
				final RORelation relation = (RORelation) type.getAttribute(foreignKey.getAssociationType());
				if (relation != null) {
					final JdbcDbColumn foreignColumn = foreignKey.getForeignColumns().get(0);
					final QName foreignQName = foreignColumn.getTable().getQName();
					final ROType foreignType = types.get(foreignQName);
					if (foreignType == null) {
						throw new IllegalStateException("Invalid type: " + foreignType);
					}
					final List<ROType> targetTypes = new ArrayList<ROType>(1);
					targetTypes.add(foreignType);
					((StaticRORelation) relation).setTargetTypes(targetTypes);
				}
			}
		}
		model.setROTypes(types);
		return model;
	}

	public ROMetaModel createMetaModel(Object pDatabaseDescription, DatabaseMetaData pMetaData)
			throws JAXRException, JAXBException, IOException, SQLException {
		jdbcExport = jdbcExportReader.read(pDatabaseDescription);
		final JdbcExportTableFilter filter = new JdbcExportTableFilter(jdbcExport, getCatalog(), getSchema());
		reader = new JdbcDbReader(pMetaData);
		reader.setTableFilter(filter);
		final JdbcDbMetaData metaData = reader.getMetaData();
		for (JdbcDbTable table : metaData.getTables()) {
			JdbcExport.Table jdbcExportTable = filter.getTable(jdbcExport,
					table.getCatalog(), table.getSchema(), table.getName());
			final String name = jdbcExportTable.getRegistryObjectType();
			if (name != null) {
				table.setQName(QName.valueOf(name));
			}
			for (JdbcDbForeignKey foreignKey : table.getForeignKeys()) {
				final List<JdbcDbColumn> localColumns = foreignKey.getLocalColumns();
				if (localColumns == null
						||  localColumns.isEmpty()) {
					continue;
				}
				final JdbcDbColumn localColumn = localColumns.get(0);
				final JdbcExport.Table.Column col = filter.getColumn(jdbcExportTable, localColumn.getName());
				String associationType = null;
				if (col != null) {
					associationType = col.getAssociationType();
				}
				if (associationType == null) {
					associationType = col.getName();
				}
				foreignKey.setAssociationType(associationType);
			}
		}
		return write(metaData);
	}

	public void createMetaModel(Object pDatabaseDescription, DatabaseMetaData pMetaData, Object pOutput)
			throws JAXRException {
		try {
			new ROMetaModelWriter().write(createMetaModel(pDatabaseDescription, pMetaData), pOutput);
		} catch (SQLException e) {
			throw new JAXRException(e);
		} catch (IOException e) {
			throw new JAXRException(e);
		} catch (JAXBException e) {
			throw new JAXRException(e);
		}
	}
	
	public JdbcExport createJdbcExport(DatabaseMetaData pMetaData) throws SQLException {
		reader = new JdbcDbReader(pMetaData);
		reader.setSchema(getCatalog());
		reader.setSchema(getSchema());
		reader.setTableFilter(new DefaultTableFilter());
		final JdbcDbMetaData myMetaData = reader.getMetaData();
		jdbcExport = OF.createJdbcExport();
		jdbcExport.setLocale(Locale.getDefault().toString());
		for (JdbcDbTable table : myMetaData.getTables()) {
			final JdbcExport.Table jdbcExportTable = OF.createJdbcExportTable();
			jdbcExportTable.setName(table.getName());
			jdbcExportTable.setRegistryObjectType("{http://namespaces.example.com/model}"
					+ table.getName());
			final List<JdbcDbColumn> primaryKeyColumns = table.getPrimaryKeyColumns();
			if (primaryKeyColumns != null  &&  primaryKeyColumns.size() == 1) {
				jdbcExportTable.setIdentification(primaryKeyColumns.get(0).getName());
			}
			for (JdbcDbColumn col : table.getColumns()) {
				JdbcExport.Table.Column jdbcExportColumn = OF.createJdbcExportTableColumn();
				jdbcExportColumn.setName(col.getName());
				final JdbcDbForeignKey foreignKey = checkRelationshipColumn(col);
				if (foreignKey != null) {
					jdbcExportColumn.setAssociationType(col.getName());
					jdbcExportColumn.setTargetType(foreignKey.getForeignTable().getName());
				}
				jdbcExportTable.getColumn().add(jdbcExportColumn);
			}
			jdbcExport.getTable().add(jdbcExportTable);
		}
		return jdbcExport;
	}
	
	public void createJdbcExport(DatabaseMetaData pMetaData, Object pOutput) throws JAXRException {
		try {
			final JdbcExport jdbcExport = createJdbcExport(pMetaData);
			jdbcExportWriter.write(jdbcExport, pOutput);
		} catch (SQLException e) {
			throw new JAXRException(e);
		} catch (JAXBException e) {
			throw new JAXRException(e);
		}
	}

	private ROAttribute getAttribute(Map<String,ROAttribute> pAttrs, JdbcDbColumn pColumn) {
		ROAttribute attr = pAttrs.get(pColumn.getName());
		if (attr == null) {
			attr = write(pColumn);
			pAttrs.put(pColumn.getName(), attr);
		}
		return attr;
	}
	
	protected void readAssets(JdbcDbTable pTable, List<AbstractRegistryObject> pList,
			ResultSet pResultSet) throws SQLException {
		final Map<String,ROAttribute> attrs = new HashMap<String,ROAttribute>();
		while (pResultSet.next()) {
			final AbstractRegistryEntry re = new AbstractRegistryEntry();
			Attributes reAttributes = null;
			for (JdbcDbColumn col : pTable.getColumns()) {
				if (pTable.getNameColumn() != null  &&
						pTable.getNameColumn().equals(col)) {
					final Names names = OFM.createAbstractRegistryObjectNames();
					names.getName().add(createInternationalString(pResultSet, col));
					re.setNames(names);
					continue;
				}
				if (pTable.getDescriptionColumn() != null  &&
						pTable.getDescriptionColumn().equals(col)) {
					final Descriptions descriptions = OFM.createAbstractRegistryObjectDescriptions();
					descriptions.getDescription().add(createInternationalString(pResultSet, col));
					re.setDescriptions(descriptions);
					continue;
				}
				if (checkRelationshipColumn(col) != null) {
					continue;
				}
				ROAttribute attr = getAttribute(attrs, col);
				if (reAttributes == null) {
					reAttributes = OFM.createAbstractRegistryObjectAttributes();
					re.setAttributes(reAttributes);
				}
				final AbstractSlot slot = OFM.createAbstractSlot();
				reAttributes.getSlotOrRelation().add(slot);
				slot.setName(attr.getName());
				slot.getValue().add(getColumnValue(pResultSet, col));
			}
			pList.add(re);
		}
	}

	protected String getColumnValue(ResultSet pResultSet, JdbcDbColumn pColumn)
			throws SQLException {
		switch (pColumn.getType()) {
			case Types.VARCHAR:
			case Types.CHAR:
			{
				return pResultSet.getString(pColumn.getName());
			}
			case Types.TINYINT:
			{
				final byte val = pResultSet.getByte(pColumn.getName());
				return pResultSet.wasNull() ? null : String.valueOf(val);
			}
			case Types.SMALLINT:
			{
				final short val = pResultSet.getShort(pColumn.getName());
				return pResultSet.wasNull() ? null : String.valueOf(val);
			}
			case Types.INTEGER:
			{
				final int val = pResultSet.getInt(pColumn.getName());
				return pResultSet.wasNull() ? null : String.valueOf(val);
			}
			case Types.BIGINT:
			{
				final long val = pResultSet.getLong(pColumn.getName());
				return pResultSet.wasNull() ? null : String.valueOf(val);
			}
			case Types.FLOAT:
			{
				final float val = pResultSet.getFloat(pColumn.getName());
				return pResultSet.wasNull() ? null : String.valueOf(val);
			}
			case Types.DOUBLE:
			{
				final double val = pResultSet.getDouble(pColumn.getName());
				return pResultSet.wasNull() ? null : String.valueOf(val);
			}
			case Types.DECIMAL:
			{
				final BigDecimal val = pResultSet.getBigDecimal(pColumn.getName());
				return val == null ? null : val.toString();
			}
			case Types.BOOLEAN:
			{
				final boolean val = pResultSet.getBoolean(pColumn.getName());
				return pResultSet.wasNull() ? null : String.valueOf(val);
			}
			default:
			{
				throw new IllegalStateException("Invalid column type: " + pColumn.getType());
			}
		}
	}
	
	private AbstractInternationalString createInternationalString(ResultSet pResultSet,
			JdbcDbColumn pColumn) throws SQLException {
		final AbstractInternationalString is = OFM.createAbstractInternationalString();
		is.setLocale(jdbcExport.getLocale());
		is.setValue(getColumnValue(pResultSet, pColumn));
		return is;
	}

	protected void createIdentification(Model pModel, JdbcDbTable pTable) {
		final JdbcExport.Table table = getJdbcExportTable(pTable);
		final String identification = table.getIdentification();
		Identifications identifications = pModel.getIdentifications();
		if (identifications == null) {
			identifications = OFM.createModelIdentifications();
			pModel.setIdentifications(identifications);
		}
		final AbstractIdentification abstractIdentification;
		final javax.xml.namespace.QName qName;
		if (identification == null  ||  identification.length() == 0) {
			abstractIdentification = OFM.createAbstractIdentification();
			qName = QNAME_UNIQUE_KEY;
		} else if ("BY_NAME".equals(identification)) {
			abstractIdentification = OFM.createAbstractIdentification();
			qName = QNAME_UNIQUE_NAME;
		} else {
			final UniqueSlot uniqueSlot = OFM.createModelIdentificationsUniqueSlot();
			uniqueSlot.setAssetType(table.getRegistryObjectType());
			final JdbcDbColumn col = reader.findColumn(pTable, identification);
			if (col == null) {
				throw new IllegalStateException("Invalid identification column for table "
						+ JdbcDbReader.asString(pTable) + ": " + identification);
			}
			final JdbcExport.Table.Column exCol = getJdbcExportColumn(col);
			String name = exCol.getAttributeName();
			if (name == null  ||  name.length() == 0) {
				name = exCol.getName();
			}
			uniqueSlot.setAttributeName(name);
			abstractIdentification = uniqueSlot;
			qName = QNAME_UNIQUE_SLOT;
		}
		final JAXBElement<AbstractIdentification> e = new JAXBElement<AbstractIdentification>(qName,
				(Class<AbstractIdentification>) abstractIdentification.getClass(),
				abstractIdentification);
		abstractIdentification.setAssetType(table.getRegistryObjectType());
		identifications.getUniqueKeyOrUniqueNameOrUniqueSlot().add(e);
	}
	
	public Model createModel(Object pDatabaseDescription, DatabaseMetaData pMetaData)
			throws JAXRException, SQLException, JAXBException, IOException {
		jdbcExport = getJdbcExport(pDatabaseDescription);
		final JdbcExportTableFilter filter = new JdbcExportTableFilter(jdbcExport, getCatalog(), getSchema());
		reader = new JdbcDbReader(pMetaData);
		reader.setTableFilter(filter);
		final JdbcDbMetaData metaData = reader.getMetaData();

		final Model model = OFM.createModel();
		final Model.Assets assets = OFM.createModelAssets();
		model.setAssets(assets);
		final List<AbstractRegistryObject> ros = assets.getRegistryObjectOrAsset();
		for (JdbcDbTable table : metaData.getTables()) {
			final String query = "SELECT * FROM " + JdbcDbReader.asString(table);
			log.debug(query);
			createIdentification(model, table);
			PreparedStatement stmt = null;
			ResultSet rs = null;
			try {
				stmt = pMetaData.getConnection().prepareStatement(query);
				rs = stmt.executeQuery();
				readAssets(table, ros, rs);
				rs.close();
				rs = null;
				stmt.close();
				stmt = null;
			} finally {
				if (stmt != null) { try { stmt.close(); } catch (Throwable ignore) { /* Ignore me */ } }
				if (rs != null) { try { rs.close(); } catch (Throwable ignore) { /* Ignore me */ } }
			}
		}
		
		return model;
	}

	private JdbcExport getJdbcExport(Object pJdbcExportFile)
			throws JAXBException, IOException {
		final JdbcExport exp = jdbcExportReader.read(pJdbcExportFile);
		if (exp.getLocale() == null) {
			exp.setLocale(Locale.getDefault().toString());
		}
		return exp;
	}
	
	public void createModel(Object pJdbcExportFile, DatabaseMetaData pMetaData, Object pOutput)
			throws JAXRException {
		try {
			final Model model = createModel(pJdbcExportFile, pMetaData);
			modelWriter.write(model, pOutput);
		} catch (SQLException e) {
			throw new JAXRException(e);
		} catch (JAXBException e) {
			throw new JAXRException(e);
		} catch (IOException e) {
			throw new JAXRException(e);
		}
	}
}