package net.sf.csutils.impexp.jdbc;

import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;


/**
 * The {@link JdbcDbReader} is used to read a databases metadata and create
 * an instance of {@link JdbcDbMetaData}.
 */
public class JdbcDbReader {
	private static class KeyPart {
		final String columnName;
		final int keySeq;
		KeyPart(String pColumnName, int pKeySeq) {
			columnName = pColumnName;
			keySeq = pKeySeq;
		}
	}
	private static final Comparator<KeyPart> keyPartComparator = new Comparator<KeyPart>(){
		@Override
		public int compare(KeyPart pKeyPart1, KeyPart pKeyPart2) {
			return pKeyPart1.keySeq - pKeyPart2.keySeq;
		}
	};
	
	private static final Logger log = Logger.getLogger(JdbcDbReader.class);
	private final DatabaseMetaData metaData;
	private final JdbcDbMetaData myMetaData;
	private JdbcDbTableFilter tableFilter;
	private String catalog;
	private String schema;

	public JdbcDbReader(DatabaseMetaData pMetaData) {
		metaData = pMetaData;
		myMetaData = new JdbcDbMetaData();
	}
	
	public String getCatalog() {
		return catalog;
	}

	public void setCatalog(String catalog) {
		this.catalog = catalog;
	}

	public String getSchema() {
		return schema;
	}

	public void setSchema(String schema) {
		this.schema = schema;
	}
	
	public JdbcDbTableFilter getTableFilter() {
		return tableFilter;
	}


	public void setTableFilter(JdbcDbTableFilter tableFilter) {
		this.tableFilter = tableFilter;
	}

	protected JdbcDbColumn getColumnMetaData(JdbcDbTable pTable,
			String pColumnName, int pColumnType, boolean pNullable) throws SQLException {
		if (tableFilter != null  &&  !tableFilter.isExported(pTable.getCatalog(),
				pTable.getSchema(), pTable.getName(), pColumnName, pColumnType)) {
			return null;
		}
		final JdbcDbColumn column = new JdbcDbColumn(pTable, pColumnName, pColumnType, pNullable);
		return column;
	}

	static String asString(JdbcDbTable pTable) {
		final String cat = pTable.getCatalog();
		final String schem = pTable.getSchema();
		final String tableName = pTable.getName();
		return asString(cat, schem, tableName);
	}

	static String asString(JdbcDbColumn pColumn) {
		return asString(pColumn.getTable()) + "." + pColumn.getName();
	}

	static String asString(JdbcDbForeignKey pKey) {
		return asString(pKey.getLocalTable()) + " => " + asString(pKey.getForeignTable());
	}

	static String asString(final String pCatalog, final String pSchema,
			final String pTableName, final String pColumnName) {
		return asString(pCatalog, pSchema, pTableName) + "." + pColumnName;
	}

	static String asString(final String pCatalog, final String pSchema,
			final String pTableName) {
		final StringBuilder sb = new StringBuilder();
		if (pCatalog != null) {
			sb.append(pCatalog);
		}
		if (pSchema != null) {
			if (sb.length() > 0) {
				sb.append('.');
			}
			sb.append(pSchema);
		}
		if (sb.length() > 0) {
			sb.append('.');
		}
		sb.append(pTableName);
		return sb.toString();
	}

	protected JdbcDbTable findTable(JdbcDbMetaData pMetaData, String pTableName) {
		for (JdbcDbTable table : pMetaData.getTables()) {
			if (table.getName().equalsIgnoreCase(pTableName)) {
				return table;
			}
		}
		return null;
	}

	protected JdbcDbColumn findColumn(JdbcDbTable pTable, String pColumnName) {
		for (JdbcDbColumn col : pTable.getColumns()) {
			if (col.getName().equalsIgnoreCase(pColumnName)) {
				return col;
			}
		}
		return null;
	}
	
	protected void readPrimaryKeyColumns(JdbcDbTable pTable) throws SQLException {
		final ResultSet rs = metaData.getPrimaryKeys(pTable.getCatalog(), pTable.getSchema(), pTable.getName());
		final List<KeyPart> pkColumnNames = new ArrayList<KeyPart>();
		while (rs.next()) {
			final String pkColumn = rs.getString("COLUMN_NAME");
			final int keySeq = rs.getInt("KEY_SEQ");
			for (KeyPart kp : pkColumnNames) {
				if (kp.keySeq == keySeq) {
					log.error("Multiple primary keys found for table " +
							asString(pTable) + ", ignoring primary keys.");
					return;
				}
			}
			final KeyPart keyPart = new KeyPart(pkColumn, keySeq);
			pkColumnNames.add(keyPart);
		}
		rs.close();
		Collections.sort(pkColumnNames, keyPartComparator);
		final List<JdbcDbColumn> pkColumns = new ArrayList<JdbcDbColumn>();
		for (int i = 0;  i < pkColumnNames.size();  i++) {
			final JdbcDbColumn col = findColumn(pTable, pkColumnNames.get(i).columnName);
			if (col == null) {
				log.error("Primary key column " + pkColumnNames.get(i)
						+ " not found for table " + asString(pTable)
						+ ", ignoring primary key.");
				return;
			}
			pkColumns.add(col);
		}
		pTable.getPrimaryKeyColumns().addAll(pkColumns);
	}
	
	protected JdbcDbTable getTableMetaData(String pTableCat, String pTableSchem, String pTableName) throws SQLException {
		if (tableFilter != null  &&  !tableFilter.isExported(pTableCat, pTableSchem, pTableName)) {
			return null;
		}
		final JdbcDbTable myTable = new JdbcDbTable(myMetaData, pTableCat, pTableSchem, pTableName);
		readColumns(myTable);
		readPrimaryKeyColumns(myTable);
		return myTable;
	}

	private boolean isSameTable(String pCat1, String pSchema1, String pTable1,
			String pCat2, String pSchema2, String pTable2) {
		if (pCat1 == null) {
			if (pCat2 != null) {
				return false;
			}
		} else {
			if (!pCat1.equals(pCat2)) {
				return false;
			}
		}
		if (pSchema1 == null) {
			if (pSchema2 != null) {
				return false;
			}
		} else {
			if (!pSchema1.equals(pSchema2)) {
				return false;
			}
		}
		return pTable1.equals(pTable2);
	}
	
	private void readForeignKeys(final JdbcDbTable pTable) throws SQLException {
		if (!pTable.getName().equalsIgnoreCase("TESTDBORD")
				&&  !pTable.getName().equalsIgnoreCase("TESTDBUSER")) {
			return;
		}
		final Map<String,List<KeyPart>> foreignKeyColumns = new HashMap<String,List<KeyPart>>();
		final Map<String,List<KeyPart>> primaryKeyColumns = new HashMap<String,List<KeyPart>>();
		final Map<String,JdbcDbTable> primaryKeyTables = new HashMap<String,JdbcDbTable>();
		final ResultSet rs = metaData.getImportedKeys(pTable.getCatalog(),
				pTable.getSchema(), pTable.getName());
		while(rs.next()) {
			final String pkTableCat = rs.getString("PKTABLE_CAT");
			final String pkTableSchem = rs.getString("PKTABLE_SCHEM");
			final String pkTableName = rs.getString("PKTABLE_NAME");
			final boolean isExported = tableFilter.isExported(pkTableCat, pkTableSchem, pkTableName);
			log.debug("readForeignKeys: pkTableCat=" + pkTableCat + ", pkTableSchem="
					+ pkTableSchem + ", pkTableName=" + pkTableName
					+ ", isExported=" + isExported);
			if (!isExported) {
				continue;
			}
			final String pkColumnName = rs.getString("PKCOLUMN_NAME");
			final String fkColumnName = rs.getString("FKCOLUMN_NAME");
			final int keySeq = rs.getInt("KEY_SEQ");
			final String fkName = rs.getString("FK_NAME");
			log.debug("readForeignKeys: pkColumnName=" + pkColumnName
					+ ", fkColumnName=" + fkColumnName
					+ ", keySeq=" + keySeq + ", fkName=" + fkName);
			List<KeyPart> fkKeyParts = foreignKeyColumns.get(fkName);
			List<KeyPart> pkKeyParts = primaryKeyColumns.get(fkName);
			if (fkKeyParts == null) {
				fkKeyParts = new ArrayList<KeyPart>();
				foreignKeyColumns.put(fkName, fkKeyParts);
				pkKeyParts = new ArrayList<KeyPart>();
				primaryKeyColumns.put(fkName, pkKeyParts);
				boolean found = false;
				for (JdbcDbTable table : myMetaData.getTables()) {
					if (isSameTable(pkTableCat, pkTableSchem, pkTableName,
							table.getCatalog(), table.getSchema(), table.getName())) {
						primaryKeyTables.put(fkName, table);
						found = true;
						break;
					}
				}
				if (!found) {
					log.error("No such table found for foreign key "
							+ fkName + ": " + asString(pkTableCat, pkTableSchem, pkTableName)
							+ ", ignoring foreign key.");
					continue;
				}
			} else {
				JdbcDbTable table = primaryKeyTables.get(fkName);
				if (!isSameTable(pkTableCat, pkTableSchem, pkTableName,
						table.getCatalog(), table.getSchema(), table.getName())) {
					log.error("Mismatching primary key tables found for primary key "
							+ fkName + ": " + asString(table) + " <-> "
							+ asString(pkTableCat, pkTableSchem, pkTableName));
					continue;
				}
			}
			final KeyPart fkKeyPart = new KeyPart(fkColumnName, keySeq);
			fkKeyParts.add(fkKeyPart);
			final KeyPart pkKeyPart = new KeyPart(pkColumnName, keySeq);
			pkKeyParts.add(pkKeyPart);
		}
		OUTER: for (String fkName : foreignKeyColumns.keySet()) {
			final List<KeyPart> fkKeyParts = foreignKeyColumns.get(fkName);
			final List<KeyPart> pkKeyParts = primaryKeyColumns.get(fkName);
			final JdbcDbTable pkTable = primaryKeyTables.get(fkName);
			Collections.sort(fkKeyParts, keyPartComparator);
			Collections.sort(pkKeyParts, keyPartComparator);
			final JdbcDbForeignKey myForeignKey = new JdbcDbForeignKey(pTable, pkTable);
			for (KeyPart keyPart : fkKeyParts) {
				final JdbcDbColumn fkCol = findColumn(pTable, keyPart.columnName);
				if (fkCol == null) {
					log.error("Column " + keyPart.columnName + " not found in table "
							+ asString(pTable) + " for foreign key " + fkName
							+ ", ignoring foreign key.");
					continue OUTER;
				}
				myForeignKey.getLocalColumns().add(fkCol);
			}
			for (KeyPart keyPart : pkKeyParts) {
				final JdbcDbColumn pkCol = findColumn(pkTable, keyPart.columnName);
				if (pkCol == null) {
					log.error("Column " + keyPart.columnName + " not found in table "
							+ asString(pkTable) + " for foreign key " + fkName
							+ ", ignoring foreign key.");
					continue OUTER;
				}
				myForeignKey.getForeignColumns().add(pkCol);
			}
			pTable.getForeignKeys().add(myForeignKey);
		}
	}
	
	private void readColumns(final JdbcDbTable pTable) throws SQLException {
		final ResultSet rs = metaData.getColumns(pTable.getCatalog(),
				pTable.getSchema(), pTable.getName(), null);
		while (rs.next()) {
			final String columnName = rs.getString("COLUMN_NAME");
			final int columnType = rs.getInt("DATA_TYPE");
			final boolean nullable = !"NO".equalsIgnoreCase(rs.getString("IS_NULLABLE"));
			final JdbcDbColumn myColumn = getColumnMetaData(pTable,
					columnName, columnType, nullable);
			if (myColumn != null) {
				pTable.getColumns().add(myColumn);
				if (tableFilter != null  &&  tableFilter.isNameColumn(pTable.getCatalog(),
						pTable.getSchema(), pTable.getName(), columnName)) {
					pTable.setNameColumn(myColumn);
				}
				if (tableFilter != null  &&  tableFilter.isDescriptionColumn(pTable.getCatalog(),
						pTable.getSchema(), pTable.getName(), columnName)) {
					pTable.setDescriptionColumn(myColumn);
				}
			}
		}
		rs.close();
	}

	public JdbcDbMetaData getMetaData() throws SQLException {
		log.debug("getMetaData: ->");
		final ResultSet rs = metaData.getTables(getCatalog(), getSchema(), null, new String[]{"TABLE"});
		while (rs.next()) {
			final String tableCat = rs.getString("TABLE_CAT");
			final String tableSchem = rs.getString("TABLE_SCHEM");
			final String tableName = rs.getString("TABLE_NAME");
			log.debug("getMetaData: cat=" + tableCat + ", schema=" + tableSchem + ", name=" + tableName);
			final JdbcDbTable myTable = getTableMetaData(tableCat, tableSchem, tableName);
			if (myTable != null) {
				myMetaData.getTables().add(myTable);
			}
		}
		rs.close();
		for (JdbcDbTable table : myMetaData.getTables()) {
			readForeignKeys(table);
		}
		log.debug("getMetaData: <-");
		return myMetaData;
	}
}
