/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry;

import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Locale;
import java.util.Properties;

import javax.xml.registry.Connection;
import javax.xml.registry.ConnectionFactory;
import javax.xml.registry.FederatedConnection;
import javax.xml.registry.JAXRException;

import org.apache.labs.jaxmas.registry.infomodel.ConnectionImpl;
import org.apache.labs.jaxmas.registry.infomodel.NLSStrings;
import org.apache.labs.jaxmas.registry.util.Locales;
import org.apache.labs.jaxmas.registry.util.Strings;


/**
 * Implementation of the {@link ConnectionFactory}.
 */
public class ConnectionFactoryImpl extends ConnectionFactory {
	/**
	 * The property for setting a locale.
	 */
	public static final String PROPERTY_LOCALE = ConnectionFactoryImpl.class.getName() + ".LOCALE"; //$NON-NLS-1$
	/**
	 * The property for specifying a database driver.
	 */
	public static final String PROPERTY_DRIVER = ConnectionFactoryImpl.class.getName() + ".DRIVER"; //$NON-NLS-1$
	/**
	 * The property for specifying a JDBC URL.
	 */
	public static final String PROPERTY_URL = ConnectionFactoryImpl.class.getName() + ".URL"; //$NON-NLS-1$
	/**
	 * The property for specifying a database user.
	 */
	public static final String PROPERTY_USER = ConnectionFactoryImpl.class.getName() + ".USER"; //$NON-NLS-1$
	/**
	 * The property for specifying a database password.
	 */
	public static final String PROPERTY_PASSWORD = ConnectionFactoryImpl.class.getName() + ".PASSWORD"; //$NON-NLS-1$

	private Properties properties;

	private Locale getLocale() {
		final String localeProperty = getProperty(PROPERTY_LOCALE);
		if (localeProperty != null) {
			return Locales.getLocale(localeProperty);
		}
		return Locale.getDefault();
	}

	private String getProperty(String pProperty) {
		return properties == null ? null : (String) properties.get(pProperty);
	}

	private String requireProperty(String pProperty, NLSStrings pBundle) throws JAXRException {
		final String s = getProperty(pProperty);
		if (s == null) {
			throw new JAXRException(pBundle.format(NLSStrings.PROPERTY_NOT_SET, pProperty));
		}
		return s;
	}

	@Override
	public Connection createConnection() throws JAXRException {
		final NLSStrings bundle = getBundle();
		final String driver = getProperty(PROPERTY_DRIVER);
		if (!Strings.isEmpty(driver)) {
			try {
				Class.forName(driver);
			} catch (ClassNotFoundException e) {
				throw new JAXRException(e);
			}
		}
		final String url = requireProperty(PROPERTY_URL, bundle);
		final String user = getProperty(PROPERTY_USER);
		final java.sql.Connection conn;
        try {
            if (user == null) {
				conn = DriverManager.getConnection(url);
            } else {
                final String password = requireProperty(PROPERTY_PASSWORD, bundle);
                conn = DriverManager.getConnection(url, user, password);
            }
            return new ConnectionImpl(conn, bundle);
        } catch (SQLException e) {
			throw new JAXRException(e);
		}
	}

	private NLSStrings getBundle() {
		final Locale locale = getLocale();
		final NLSStrings bundle = new NLSStrings(locale);
		return bundle;
	}

	@Override
	public FederatedConnection createFederatedConnection(@SuppressWarnings("rawtypes") Collection pConnections) throws JAXRException {
		final NLSStrings bundle = getBundle();
		throw new JAXRException(bundle.get(NLSStrings.OPERATION_NOT_IMPLEMENTED));
	}

	@Override
	public Properties getProperties() throws JAXRException {
		return properties;
	}

	@Override
	public void setProperties(Properties pProperties) throws JAXRException {
		properties = pProperties;
	}
}
