/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.accessor;

import java.util.Map;

import javax.xml.registry.JAXRException;
import javax.xml.registry.RegistryService;
import javax.xml.registry.infomodel.Key;
import javax.xml.registry.infomodel.RegistryObject;
import javax.xml.registry.infomodel.Slot;

import org.apache.labs.jaxmas.registry.infomodel.ROState;
import org.apache.labs.jaxmas.registry.infomodel.RegistryObjectImpl;
import org.apache.labs.jaxmas.registry.infomodel.RegistryServiceImpl;
import org.apache.labs.jaxmas.registry.sql.DbDriver;



/**
 * Base class for registry loaders.
 */
public abstract class AbstractROAccessor<RO extends RegistryObject> implements ROAccessor<RO> {
    /**
     * Creates a new registry object instance. The new instance has state
     * {@link ROState#created}, if the key is null. Otherwise, the instance
     * has state {@link ROState#referenced}.
     */
    protected abstract RO newInstance(RegistryServiceImpl pRegistryService, Key pKey) throws JAXRException;

	/**
	 * Returns the database driver.
	 */
	protected DbDriver getDbDriver(RegistryService pRegistryService) {
		return ((RegistryServiceImpl) pRegistryService).getDbDriver();
	}

	@Override
    public Key newKey(RegistryService pRegistryService) throws JAXRException {
		final long id = getDbDriver(pRegistryService).newId("keys"); //$NON-NLS-1$
		final String s = String.valueOf(id);
		final String key = "00000000000000000000000000000000".substring(s.length()) + s; //$NON-NLS-1$
		return pRegistryService.getBusinessLifeCycleManager().createKey(key);
	}

	/**
	 * Creates a new instance.
	 */
	@Override
    public RO create(RegistryService pRegistryService) throws JAXRException {
	    final RO ro = newInstance((RegistryServiceImpl) pRegistryService, null);
	    ro.setKey(newKey(pRegistryService));
	    return ro;
	}

    @Override
    public RO create(RegistryService pRegistryService, Key pKey) throws JAXRException {
        return newInstance((RegistryServiceImpl) pRegistryService, pKey);
    }

	@Override
	public Map<String, Slot> getSlots(RegistryService pRegistryService, RO pRegistryObject) throws JAXRException {
		return getDbDriver(pRegistryService).getSlots(pRegistryObject);
	}
	
	@Override
	public void save(RegistryService pRegistryService, RegistryObject pObject) throws JAXRException {
		final RegistryObjectImpl<?> ro = (RegistryObjectImpl<?>) pObject;
		switch (ro.getState()) {
		    case created:
	            getDbDriver(pRegistryService).insert(pObject, getRegistryObjectType());
	            break;
		    case loaded:
		    case referenced:
		        getDbDriver(pRegistryService).update(pObject);
		        break;
		    case deleted:
		        throw new IllegalStateException("Unable to save a deleted object."); //$NON-NLS-1$
		}
        ro.setState(ROState.loaded);
	}

    @Override
    public void remove(RegistryService pRegistryService, RO pRegistryObject)
            throws JAXRException {
        /* We try to delete the registry object first, so that we get exceptions
         * caused by an ON DELETE RESTRICT as soon as possible.
         */
        final DbDriver dbDriver = getDbDriver(pRegistryService);
        dbDriver.deleteRegistryObject(pRegistryObject.getKey());

        /*
         * Finally, lets delete the registry objects slots.
         */
        dbDriver.deleteSlots(pRegistryObject);
    }

    @Override
    public void load(RegistryServiceImpl pRegistryService, RegistryObject pRegistryObject) throws JAXRException {
        getRegistryObjectLoader().load(pRegistryService, pRegistryObject);
    }
}
