/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.accessor;

import java.lang.reflect.UndeclaredThrowableException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.xml.registry.BusinessQueryManager;
import javax.xml.registry.FindQualifier;
import javax.xml.registry.JAXRException;
import javax.xml.registry.RegistryService;
import javax.xml.registry.infomodel.ClassificationScheme;
import javax.xml.registry.infomodel.Key;

import org.apache.labs.jaxmas.registry.infomodel.ClassificationSchemeImpl;
import org.apache.labs.jaxmas.registry.infomodel.NLSStrings;
import org.apache.labs.jaxmas.registry.infomodel.RegistryServiceImpl;
import org.apache.labs.jaxmas.registry.sql.FilterPredicate;
import org.apache.labs.jaxmas.registry.sql.NamePredicate;
import org.apache.labs.jaxmas.registry.sql.RegistryObjectLoader;


/**
 * Implementation of {@link ROAccessor} for classification schemes.
 */
public class ClassificationSchemeAccessor extends AbstractROAccessor<ClassificationScheme> {
	private static final ClassificationSchemeAccessor theInstance = new ClassificationSchemeAccessor();
    private static final RegistryObjectLoader rol;
    static {
        try {
            rol = new RegistryObjectLoader(ROAccessors.ObjectTypes.CLASSIFICATION_SCHEME, null, null);
        } catch (JAXRException e) {
            throw new UndeclaredThrowableException(e);
        } catch (SQLException e) {
            throw new UndeclaredThrowableException(e);
        }
    }

	/**
	 * Private constructor, to ensure singleton pattern.
	 */
	private ClassificationSchemeAccessor() {
		// Does nothing
	}

	/**
	 * Returns the singleton instance.
	 */
	public static ClassificationSchemeAccessor getInstance() {
		return theInstance;
	}

	@Override
	public int getRegistryObjectType() {
		return ROAccessors.ObjectTypes.CLASSIFICATION_SCHEME.ordinal();
	}

	@SuppressWarnings("unchecked")
    private static Collection<String> asStringCollection(Collection<?> pCollection) {
	    return (Collection<String>) pCollection;
	}

	@SuppressWarnings("unchecked")
	private static List<ClassificationScheme> asClassificationSchemeList(List<?> pList) {
        return (List<ClassificationScheme>) pList;
    }

	/**
	 * Implementation of {@link BusinessQueryManager#findClassificationSchemes(Collection, Collection, Collection, Collection)}.
	 */
	public List<ClassificationScheme> findClassificationSchemes(
			final RegistryService pRegistryService, Collection<?> pFindQualifiers,
			Collection<?> pNamePatterns, Collection<?> pClassifications,
			Collection<?> pExternalLinks) throws JAXRException {
	    final List<FilterPredicate> predicates = new ArrayList<FilterPredicate>();
	    if (pFindQualifiers != null  &&  !pFindQualifiers.isEmpty()) {
	    	for (Object o : pFindQualifiers) {
	    		if (!FindQualifier.EXACT_NAME_MATCH.equals(o)) {
	    			throw new JAXRException(RegistryServiceImpl.getNLSStrings(pRegistryService).format(NLSStrings.UNSUPPORTED_PARAMETER, "pFindQualifiers")); //$NON-NLS-1$
	    		}
	    	}
		}
		if (pNamePatterns != null  &&  !pNamePatterns.isEmpty()) {
		    predicates.add(new NamePredicate(asStringCollection(pNamePatterns)));
		}
		if (pClassifications != null  &&  !pClassifications.isEmpty()) {
			throw new JAXRException(RegistryServiceImpl.getNLSStrings(pRegistryService).format(NLSStrings.UNSUPPORTED_PARAMETER, "pClassifications")); //$NON-NLS-1$
		}
		if (pExternalLinks != null  &&  !pExternalLinks.isEmpty()) {
			throw new JAXRException(RegistryServiceImpl.getNLSStrings(pRegistryService).format(NLSStrings.UNSUPPORTED_PARAMETER, "pExternalLinks")); //$NON-NLS-1$
		}
		return asClassificationSchemeList(rol.getResultList(pRegistryService, predicates, null));
	}

    @Override
    public void remove(RegistryService pRegistryService,
            ClassificationScheme pRegistryObject) throws JAXRException {
        super.remove(pRegistryService, pRegistryObject);
    }

    @Override
    protected ClassificationScheme newInstance(RegistryServiceImpl pRegistryService, Key pKey)
            throws JAXRException {
        return new ClassificationSchemeImpl(pRegistryService, pKey);
    }

    @Override
    public RegistryObjectLoader getRegistryObjectLoader() {
        return rol;
    }
}
