/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.accessor;

import java.lang.reflect.UndeclaredThrowableException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.xml.registry.BusinessQueryManager;
import javax.xml.registry.FindQualifier;
import javax.xml.registry.JAXRException;
import javax.xml.registry.RegistryService;
import javax.xml.registry.infomodel.Key;
import javax.xml.registry.infomodel.Organization;

import org.apache.labs.jaxmas.registry.accessor.ROAccessors.ObjectTypes;
import org.apache.labs.jaxmas.registry.infomodel.NLSStrings;
import org.apache.labs.jaxmas.registry.infomodel.OrganizationImpl;
import org.apache.labs.jaxmas.registry.infomodel.RegistryServiceImpl;
import org.apache.labs.jaxmas.registry.sql.FilterPredicate;
import org.apache.labs.jaxmas.registry.sql.NamePredicate;
import org.apache.labs.jaxmas.registry.sql.RegistryObjectLoader;
import org.apache.labs.jaxmas.registry.sql.RegistryObjectTypePredicate;


/**
 * Implementation of {@link ROAccessor} for instances of {@link Organization}.
 */
public class OrganizationAccessor extends AbstractROAccessor<Organization> {
    private static final RegistryObjectLoader rol;
    static {
        try {
            rol = new RegistryObjectLoader(ROAccessors.ObjectTypes.ORGANIZATION, null, null);
        } catch (JAXRException e) {
            throw new UndeclaredThrowableException(e);
        } catch (SQLException e) {
            throw new UndeclaredThrowableException(e);
        }
    }

    private static final OrganizationAccessor theInstance = new OrganizationAccessor();

    /**
     * Private constructor, to ensure singleton pattern.
     */
    private OrganizationAccessor() {
        // Nothing to do.
    }

    /**
     * Returns the singleton instance.
     */
    public static OrganizationAccessor getInstance() {
        return theInstance;
    }

    @Override
    protected Organization newInstance(RegistryServiceImpl pRegistryService, Key pKey) throws JAXRException {
        return new OrganizationImpl(pRegistryService, pKey);
    }

    @Override
    public RegistryObjectLoader getRegistryObjectLoader() {
        return rol;
    }

    @Override
    public int getRegistryObjectType() {
        return ROAccessors.ObjectTypes.ORGANIZATION.ordinal();
    }

	@SuppressWarnings("unchecked")
    private static Collection<String> asStringCollection(Collection<?> pCollection) {
	    return (Collection<String>) pCollection;
	}

	@SuppressWarnings("unchecked")
	private static List<Organization> asOrganizationList(List<?> pList) {
		return (List<Organization>) pList;
	}

	/**
	 * Implementation of {@link BusinessQueryManager#findOrganizations(Collection, Collection, Collection, Collection, Collection, Collection)}.
	 */
	public List<Organization> findOrganizations(
			final RegistryService pRegistryService, Collection<?> pFindQualifiers,
			Collection<?> pNamePatterns, Collection<?> pClassifications,
			Collection<?> pSpecifications, Collection<?> pExternalIdentifiers,
			Collection<?> pExternalLinks) throws JAXRException {
	    final List<FilterPredicate> predicates = new ArrayList<FilterPredicate>();
	    predicates.add(new RegistryObjectTypePredicate(ObjectTypes.ORGANIZATION));
	    if (pFindQualifiers != null  &&  !pFindQualifiers.isEmpty()) {
	    	for (Object o : pFindQualifiers) {
	    		if (!FindQualifier.EXACT_NAME_MATCH.equals(o)) {
	    			throw new JAXRException(RegistryServiceImpl.getNLSStrings(pRegistryService).format(NLSStrings.UNSUPPORTED_PARAMETER, "pFindQualifiers")); //$NON-NLS-1$
	    		}
	    	}
		}
		if (pNamePatterns != null  &&  !pNamePatterns.isEmpty()) {
		    predicates.add(new NamePredicate(asStringCollection(pNamePatterns)));
		}
		if (pClassifications != null  &&  !pClassifications.isEmpty()) {
			throw new JAXRException(RegistryServiceImpl.getNLSStrings(pRegistryService).format(NLSStrings.UNSUPPORTED_PARAMETER, "pClassifications")); //$NON-NLS-1$
		}
		if (pSpecifications != null  &&  !pSpecifications.isEmpty()) {
			throw new JAXRException(RegistryServiceImpl.getNLSStrings(pRegistryService).format(NLSStrings.UNSUPPORTED_PARAMETER, "pSpecifications")); //$NON-NLS-1$
		}
		if (pExternalIdentifiers != null  &&  !pExternalIdentifiers.isEmpty()) {
			throw new JAXRException(RegistryServiceImpl.getNLSStrings(pRegistryService).format(NLSStrings.UNSUPPORTED_PARAMETER, "pExternalIdentifiers")); //$NON-NLS-1$
		}
		if (pExternalLinks != null  &&  !pExternalLinks.isEmpty()) {
			throw new JAXRException(RegistryServiceImpl.getNLSStrings(pRegistryService).format(NLSStrings.UNSUPPORTED_PARAMETER, "pExternalLinks")); //$NON-NLS-1$
		}
		return asOrganizationList(rol.getResultList(pRegistryService, predicates, null));
	}
}
