/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.accessor;

import java.util.Map;

import javax.xml.registry.JAXRException;
import javax.xml.registry.RegistryService;
import javax.xml.registry.infomodel.Key;
import javax.xml.registry.infomodel.RegistryObject;
import javax.xml.registry.infomodel.Slot;

import org.apache.labs.jaxmas.registry.infomodel.ROState;
import org.apache.labs.jaxmas.registry.infomodel.RegistryServiceImpl;
import org.apache.labs.jaxmas.registry.sql.RegistryObjectLoader;


/**
 * A loader for registry objects is used to load a particular
 * registry objects data from the registry.
 */
public interface ROAccessor<RO extends RegistryObject> {
    /**
	 * Returns the registry objects type.
	 */
	int getRegistryObjectType();

	/**
	 * Creates a new instance with the given key. The instance has the state
	 * {@link ROState#loaded}.
	 */
	RO create(RegistryService pRegistryService, Key pKey) throws JAXRException;
	
	/**
	 * Creates a new instance with a new, unique key. The instance has the state
     * {@link ROState#created}.
	 */
	RO create(RegistryService pRegistryService) throws JAXRException;

	/**
	 * Loads the given objects slots.
	 */
	Map<String, Slot> getSlots(RegistryService pRegistryService, RO pRegistryObject) throws JAXRException;

	/**
	 * Saves the given object.
	 */
	void save(RegistryService pRegistryService, RegistryObject pObject) throws JAXRException;

	/**
	 * Creates a new registry object key.
	 */
	Key newKey(RegistryService pRegistryService) throws JAXRException;

	/**
	 * Called to remove an object from the registry.
	 * @param pRegistryService The registry service.
	 * @param pRO The registry object to remove.
	 */
	void remove(RegistryService pRegistryService, RO pRegistryObject) throws JAXRException;

	/**
	 * Loads the given objects data from the registry.
	 */
    void load(RegistryServiceImpl pRegistryService, RegistryObject pRegistryObject) throws JAXRException;

    /**
     * Returns a prepared instance of {@link RegistryObjectLoader} for loading
     * registry objects.
     */
    RegistryObjectLoader getRegistryObjectLoader();
}
