/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.accessor;

import java.util.HashMap;
import java.util.Map;

import javax.xml.registry.LifeCycleManager;


/**
 * A utility class, which provides access to instances of
 * {@link ROAccessor}.
 */
public class ROAccessors {
	/**
	 * The enumeration of builtin taxonomies.
	 */
	public enum ClassificationSchemes {
		/**
		 * A classification scheme: ObjectType
		 */
		OBJECT_TYPE("00000000000000000000000000000001", "ObjectType"),  //$NON-NLS-1$//$NON-NLS-2$
		/**
		 * A classification scheme: ObjectType
		 */
		ASSOCIATION_TYPE("00000000000000000000000000000002", "AssociationType"); //$NON-NLS-1$ //$NON-NLS-2$

		private final String id;
		private final String name;

		ClassificationSchemes(String pId, String pName) {
			id = pId;
			name = pName;
		}

		/**
		 * Returns the classification schemes id.
		 */
		public String getId() { return id; }
		/**
		 * Returns the classification schemes name.
		 */
		public String getName() { return name; }
	}

	/**
     * The enumeration of available object types.
     */
    public enum ObjectTypes {
        /**
         * An object type: Association.
         */
        ASSOCIATION("00000000000000000000000000000003", LifeCycleManager.ASSOCIATION), //$NON-NLS-1$
        /**
         * An object type: AuditableEvent
         */
        AUDITABLE_EVENT("00000000000000000000000000000004", LifeCycleManager.AUDITABLE_EVENT), //$NON-NLS-1$
        /**
         * An object type: Classification
         */
        CLASSIFICATION("00000000000000000000000000000005", LifeCycleManager.CLASSIFICATION), //$NON-NLS-1$
        /**
         * An object type: ExternalIdentifier
         */
        EXTERNAL_IDENTIFIER("00000000000000000000000000000006", LifeCycleManager.EXTERNAL_IDENTIFIER), //$NON-NLS-1$
        /**
         * An object type: ExternalLink
         */
        EXTERNAL_LINK("00000000000000000000000000000007", LifeCycleManager.EXTERNAL_LINK), //$NON-NLS-1$
        /**
         * An object type: RegistryEntry
         */
        REGISTRY_ENTRY("00000000000000000000000000000009", LifeCycleManager.REGISTRY_ENTRY), //$NON-NLS-1$
        /**
         * An object type: Service
         */
        SERVICE("000000000000000000000000000000010", LifeCycleManager.SERVICE), //$NON-NLS-1$
        /**
         * An object type: Classification scheme.
         */
        CLASSIFICATION_SCHEME("00000000000000000000000000000011", LifeCycleManager.CLASSIFICATION_SCHEME), //$NON-NLS-1$
        /**
         * An object type: User
         */
        USER("00000000000000000000000000000012", LifeCycleManager.USER), //$NON-NLS-1$
        /**
         * An object type: Organization
         */
        ORGANIZATION("00000000000000000000000000000015", LifeCycleManager.ORGANIZATION), //$NON-NLS-1$
        /**
         * An object type: Concept.
         */
        CONCEPT("00000000000000000000000000000016", LifeCycleManager.CONCEPT), //$NON-NLS-1$
        /**
         * An object type: ServiceBinding
         */
        SERVICE_BINDING("00000000000000000000000000000017", LifeCycleManager.SERVICE_BINDING), //$NON-NLS-1$
        /**
         * An object type SpecificationLink
         */
        SPECIFICATION_LINK("00000000000000000000000000000018", LifeCycleManager.SPECIFICATION_LINK), //$NON-NLS-1$
        /**
         * An object type: RegistryPackage
         */
        REGISTRY_PACKAGE("00000000000000000000000000000019", LifeCycleManager.REGISTRY_PACKAGE), //$NON-NLS-1$
        /**
         * An object type: ExtrinsicObject
         */
        EXTRINSIC_OBJECT("000000000000000000000000000000020", LifeCycleManager.EXTRINSIC_OBJECT);//$NON-NLS-1$

        private final String id;
        private final String ifaceName;

        ObjectTypes(String pId, String pInterfaceName) {
            id = pId;
            ifaceName = pInterfaceName;
        }

        /**
         * Returns the categories UDDI key.
         */
        public String getId() {
            return id;
        }

        /**
         * Returns the lifecycle managers interface name for this object type.
         */
        public String getInterfaceName() {
            return ifaceName;
        }
    }

    private static final ROAccessor<?>[] roAccessors = new ROAccessor[]{
        AssociationAccessor.getInstance(),
        AuditableEventAccessor.getInstance(),
        ClassificationAccessor.getInstance(),
        ExternalIdentifierAccessor.getInstance(),
        ExternalLinkAccessor.getInstance(),
        RegistryEntryAccessor.getInstance(),
        ServiceAccessor.getInstance(),
        ClassificationSchemeAccessor.getInstance(),
        UserAccessor.getInstance(),
        OrganizationAccessor.getInstance(),
        ConceptAccessor.getInstance(),
        ServiceBindingAccessor.getInstance(),
        SpecificationLinkAccessor.getInstance(),
        ExtrinsicObjectAccessor.getInstance(),
        RegistryPackageAccessor.getInstance()
    };

    private static final Map<String,ROAccessor<?>> roAccessorsByInterfaceName = new HashMap<String,ROAccessor<?>>();
    static {
        for (ObjectTypes type : ROAccessors.ObjectTypes.values()) {
            final ROAccessor<?> roAcc = roAccessors[type.ordinal()];
            roAccessorsByInterfaceName.put(type.getInterfaceName(), roAcc);
        }
    }
    
    /**
     * Returns a registry object accessor for the given object type.
     */
    public static ROAccessor<?> getROAccessor(ObjectTypes pRoType) {
        return roAccessors[pRoType.ordinal()];
    }

    /**
     * Returns a registry object accessor for the interface name.
     * @see LifeCycleManager
     * @return Registry object accessor, if available, or null.
     */
    public static ROAccessor<?> getROAccessor(String pInterfaceName) {
        return roAccessorsByInterfaceName.get(pInterfaceName);
    }

    /**
     * Returns a registry object accessor for the given object type.
     */
    public static ROAccessor<?> getROAccessor(int pRoType) {
        return roAccessors[pRoType];
    }

    /**
     * Returns the registry object accessor for concepts.
     */
    public static ConceptAccessor getConceptAccessor() {
        return (ConceptAccessor) getROAccessor(ObjectTypes.CONCEPT);
    }

    /**
     * Returns the registry object accessor for classification schemes.
     */
    public static ClassificationSchemeAccessor getClassificationSchemeAccessor() {
        return (ClassificationSchemeAccessor) getROAccessor(ObjectTypes.CLASSIFICATION_SCHEME);
    }
}
