/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.accessor;

import java.lang.reflect.UndeclaredThrowableException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.xml.registry.FindQualifier;
import javax.xml.registry.JAXRException;
import javax.xml.registry.RegistryService;
import javax.xml.registry.infomodel.Key;
import javax.xml.registry.infomodel.RegistryEntry;

import org.apache.labs.jaxmas.registry.accessor.ROAccessors.ObjectTypes;
import org.apache.labs.jaxmas.registry.infomodel.NLSStrings;
import org.apache.labs.jaxmas.registry.infomodel.RegistryEntryImpl;
import org.apache.labs.jaxmas.registry.infomodel.RegistryServiceImpl;
import org.apache.labs.jaxmas.registry.sql.CustomObjectTypePredicate;
import org.apache.labs.jaxmas.registry.sql.FilterPredicate;
import org.apache.labs.jaxmas.registry.sql.NamePredicate;
import org.apache.labs.jaxmas.registry.sql.RegistryObjectLoader;
import org.apache.labs.jaxmas.registry.sql.RegistryObjectTypePredicate;


/**
 * Implementation of {@link ROAccessor} for registry entries.
 */
public class RegistryEntryAccessor extends AbstractROAccessor<RegistryEntry> {
    private static final RegistryObjectLoader rol;
    static {
        try {
            rol = new RegistryObjectLoader(ROAccessors.ObjectTypes.REGISTRY_ENTRY, null, null);
        } catch (JAXRException e) {
            throw new UndeclaredThrowableException(e);
        } catch (SQLException e) {
            throw new UndeclaredThrowableException(e);
        }
    }

    private static final RegistryEntryAccessor theInstance = new RegistryEntryAccessor();

    /**
     * Private constructor, to ensure singleton pattern.
     */
    private RegistryEntryAccessor() {
        // Does nothing.
    }

    /**
     * Returns the singleton instance.
     */
    public static RegistryEntryAccessor getInstance() {
        return theInstance;
    }

    @Override
    public int getRegistryObjectType() {
        return ROAccessors.ObjectTypes.REGISTRY_ENTRY.ordinal();
    }

    @Override
    protected RegistryEntry newInstance(RegistryServiceImpl pRegistryService, Key pKey) throws JAXRException {
        return new RegistryEntryImpl<RegistryEntry>(pRegistryService, pKey);
    }

    @Override
    public RegistryObjectLoader getRegistryObjectLoader() {
        return rol;
    }

    @SuppressWarnings("unchecked")
	private static List<RegistryEntry> asRegistryEntryList(List<?> pList) {
    	return (List<RegistryEntry>) pList;
    }
    
	public List<RegistryEntry> findRegistryEntries(RegistryService pRegistryService,
			Collection<String> pFindQualifiers,
			Collection<String> pObjectTypes, Collection<String> pNamePatterns,
			Collection<String> pDescriptionPatterns,
			Collection<?> pClassifications, Collection<?> pExternalIdentifiers,
			Collection<?> pExternalLinks) throws JAXRException {
	    final List<FilterPredicate> predicates = new ArrayList<FilterPredicate>();
	    if (pObjectTypes == null  ||  pObjectTypes.isEmpty()) {
	        predicates.add(new RegistryObjectTypePredicate(ObjectTypes.REGISTRY_ENTRY));
	    } else {
            predicates.add(new CustomObjectTypePredicate(pObjectTypes));
	    }
	    if (pFindQualifiers != null  &&  !pFindQualifiers.isEmpty()) {
	    	for (Object o : pFindQualifiers) {
	    		if (!FindQualifier.EXACT_NAME_MATCH.equals(o)) {
	    			throw new JAXRException(RegistryServiceImpl.getNLSStrings(pRegistryService).format(NLSStrings.UNSUPPORTED_PARAMETER, "pFindQualifiers")); //$NON-NLS-1$
	    		}
	    	}
		}
		if (pNamePatterns != null  &&  !pNamePatterns.isEmpty()) {
		    predicates.add(new NamePredicate(pNamePatterns));
		}
		if (pDescriptionPatterns != null  &&  !pDescriptionPatterns.isEmpty()) {
			throw new JAXRException(RegistryServiceImpl.getNLSStrings(pRegistryService).format(NLSStrings.UNSUPPORTED_PARAMETER, "pDescriptionPatterns")); //$NON-NLS-1$
		}
		if (pClassifications != null  &&  !pClassifications.isEmpty()) {
			throw new JAXRException(RegistryServiceImpl.getNLSStrings(pRegistryService).format(NLSStrings.UNSUPPORTED_PARAMETER, "pClassifications")); //$NON-NLS-1$
		}
		if (pExternalIdentifiers != null  &&  !pExternalIdentifiers.isEmpty()) {
			throw new JAXRException(RegistryServiceImpl.getNLSStrings(pRegistryService).format(NLSStrings.UNSUPPORTED_PARAMETER, "pExternalIdentifiers")); //$NON-NLS-1$
		}
		if (pExternalLinks != null  &&  !pExternalLinks.isEmpty()) {
			throw new JAXRException(RegistryServiceImpl.getNLSStrings(pRegistryService).format(NLSStrings.UNSUPPORTED_PARAMETER, "pExternalLinks")); //$NON-NLS-1$
		}
		return asRegistryEntryList(rol.getResultList(pRegistryService, predicates, null));
	}
}
