/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.infomodel;

import javax.xml.registry.JAXRException;
import javax.xml.registry.infomodel.Association;
import javax.xml.registry.infomodel.Concept;
import javax.xml.registry.infomodel.Key;
import javax.xml.registry.infomodel.RegistryObject;

import org.apache.labs.jaxmas.registry.accessor.AssociationAccessor;
import org.apache.labs.jaxmas.registry.accessor.ROAccessor;


/**
 * Default implementation of {@link Association}.
 */
public class AssociationImpl extends OwnedRegistryObject<Association> implements Association {
    private Key associationTypeKey;
    private Concept associationType;
    private Key targetObjectKey;
    private RegistryObject targetObject;
	private boolean confirmed, confirmedBySourceOwner, confirmedByTargetOwner, extramural;
	private int num;

	/**
	 * Creates a new instance.
	 */
	public AssociationImpl(RegistryServiceImpl pRegistryService, Key pKey) {
		super(pRegistryService, pKey);
	}

	/**
	 * Returns the associations number within its parents list of associations.
	 */
	public int getNum() {
		return num;
	}

	/**
	 * Sets the associations number within its parents list of associations.
	 */
	public void setNum(int pNum) {
		num = pNum;
	}

	@Override
	public Concept getAssociationType() throws JAXRException {
	    ensureLoaded();
	    if (associationType == null) {
	        if (associationTypeKey != null) {
	            associationType = (Concept) getRegistryService().getBusinessQueryManager().getRegistryObject(associationTypeKey.getId());
	        }
	    }
	    return associationType;
	}

	@Override
	public RegistryObject getSourceObject() throws JAXRException {
	    return getOwner();
	}

	@Override
	public RegistryObject getTargetObject() throws JAXRException {
	    ensureLoaded();
	    if (targetObject == null) {
	        if (targetObjectKey != null) {
	            targetObject = getRegistryService().getBusinessQueryManager().getRegistryObject(targetObjectKey.getId());
	        }
	    }
	    return targetObject;
	}

	@Override
	public boolean isConfirmed() throws JAXRException {
		return confirmed;
	}

	@Override
	public boolean isConfirmedBySourceOwner() throws JAXRException {
		return confirmedBySourceOwner;
	}

	@Override
	public boolean isConfirmedByTargetOwner() throws JAXRException {
		return confirmedByTargetOwner;
	}

	@Override
	public boolean isExtramural() throws JAXRException {
		return extramural;
	}

    /**
     * Sets the association types key.
     */
    public void setAssociationTypeKey(Key pKey) throws JAXRException {
        ensureLoaded();
        if (pKey == null) {
            associationType = null;
        } else if (associationType != null  &&  !associationType.getKey().equals(pKey)) {
            associationType = null;
        }
        associationTypeKey = pKey;
    }

    @Override
	public void setAssociationType(Concept pAssociationType) throws JAXRException {
	    ensureLoaded();
        associationType = pAssociationType;
	    if (pAssociationType == null) {
	        associationTypeKey = null;
	    } else {
	        associationTypeKey = pAssociationType.getKey();
	    }
	}

	@Override
	public void setSourceObject(RegistryObject pSource) throws JAXRException {
	    setOwner(pSource);
	}

	@Override
	public void setTargetObject(RegistryObject pTarget) throws JAXRException {
        ensureLoaded();
        targetObject = pTarget;
        if (pTarget == null) {
            targetObjectKey = null;
        } else {
            targetObjectKey = pTarget.getKey();
        }
	}

	/**
	 * Sets the target objects key.
	 */
	public void setTargetObjectKey(Key pTargetKey) throws JAXRException {
        ensureLoaded();
        if (pTargetKey == null) {
            targetObject = null;
        } else if (targetObject != null  &&  !targetObject.getKey().equals(pTargetKey)) {
            targetObject = null;
        }
        targetObjectKey = pTargetKey;
    }

    @Override
	public ROAccessor<Association> getROLoader() {
	    return AssociationAccessor.getInstance();
	}

	@Override
	public boolean isRestricting() {
		return false;
	}
}
