/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.infomodel;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.StringTokenizer;

import javax.xml.registry.BulkResponse;
import javax.xml.registry.BusinessQueryManager;
import javax.xml.registry.JAXRException;
import javax.xml.registry.infomodel.ClassificationScheme;
import javax.xml.registry.infomodel.Concept;
import javax.xml.registry.infomodel.Key;
import javax.xml.registry.infomodel.Organization;
import javax.xml.registry.infomodel.RegistryEntry;

import org.apache.labs.jaxmas.registry.accessor.AssociationAccessor;
import org.apache.labs.jaxmas.registry.accessor.ClassificationSchemeAccessor;
import org.apache.labs.jaxmas.registry.accessor.OrganizationAccessor;
import org.apache.labs.jaxmas.registry.accessor.RegistryEntryAccessor;
import org.apache.labs.jaxmas.registry.sql.AssociationSourcePredicate;
import org.apache.labs.jaxmas.registry.sql.AssociationTargetPredicate;
import org.apache.labs.jaxmas.registry.sql.AssociationTypePredicate;
import org.apache.labs.jaxmas.registry.sql.FilterPredicate;


/**
 * Default implementation of {@link BusinessQueryManager}.
 */
public class BusinessQueryManagerImpl extends QueryManagerImpl implements BusinessQueryManager, JaxMasQueryManager {
	/**
	 * Creates a new instance with the given registry service.
	 */
	public BusinessQueryManagerImpl(RegistryServiceImpl pRegistryService) {
		super(pRegistryService);
	}

	@Override
	public BulkResponse findAssociations(@SuppressWarnings("rawtypes") Collection pFindQualifiers,
			String pSourceObjectId, String pTargetObjectId,
			@SuppressWarnings("rawtypes") Collection pAssociationTypes) throws JAXRException {
	    final List<FilterPredicate> predicates = new ArrayList<FilterPredicate>();
	    if (pFindQualifiers != null  &&  pFindQualifiers.size() > 0) {
	        throw new JAXRException(NLSStrings.UNSUPPORTED_PARAMETER);
	    }
	    if (pSourceObjectId != null) {
	        predicates.add(new AssociationSourcePredicate(pSourceObjectId));
	    }
	    if (pTargetObjectId != null) {
	        predicates.add(new AssociationTargetPredicate(pTargetObjectId));
	    }
	    if (pAssociationTypes != null  &&  pAssociationTypes.size() > 0) {
	        predicates.add(new AssociationTypePredicate(pAssociationTypes));
	    }
	    final Collection<?> assocs = AssociationAccessor.getInstance().getRegistryObjectLoader().getResultList(getRegistryService(), predicates, null);
	    return new BulkResponseImpl(assocs);
	}

	@Override
	public BulkResponse findCallerAssociations(@SuppressWarnings("rawtypes") Collection pFindQualifiers,
			Boolean pConfirmedByCaller, Boolean pConfirmedByOtherParty,
			@SuppressWarnings("rawtypes") Collection pAssociationTypes) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public ClassificationScheme findClassificationSchemeByName(@SuppressWarnings("rawtypes") Collection pFindQualifiers, String pNamePatterns)
			throws JAXRException {
	    final List<ClassificationScheme> result = ClassificationSchemeAccessor.getInstance().findClassificationSchemes(getRegistryService(), pFindQualifiers, Collections.singleton(pNamePatterns), null, null);
	    if (result == null  ||  result.isEmpty()) {
	        return null;
	    }
	    if (result.size() > 1) {
	        throw new JAXRException(NLSStrings.MULTIPLE_INSTANCES_FOUND);
	    }
	    return result.get(0);
	}

	@Override
	public BulkResponse findClassificationSchemes(@SuppressWarnings("rawtypes") Collection pFindQualifiers,
	        @SuppressWarnings("rawtypes") Collection pNamePatterns, @SuppressWarnings("rawtypes") Collection pClassifications,
	        @SuppressWarnings("rawtypes") Collection pExternalLinks) throws JAXRException {
		final Collection<ClassificationScheme> list =
			ClassificationSchemeAccessor.getInstance().findClassificationSchemes(getRegistryService(),
					pFindQualifiers, pNamePatterns, pClassifications, pExternalLinks);
		return new BulkResponseImpl(list);
	}

    /**
     * Returns the concept with the given value, the given classification scheme, and no parent concept.
     */
	private Concept findConcept(Collection<?> pConcepts, String pValue) throws JAXRException {
	    if (pConcepts != null) {
    	    for (Object o : pConcepts) {
                final Concept concept = (Concept) o;
                if (pValue.equals(concept.getValue())) {
                    return concept;
                }
            }
	    }
        return null;
    }

	@Override
	public Concept findConceptByPath(String pPath) throws JAXRException {
        final StringTokenizer st = new StringTokenizer(pPath, "/"); //$NON-NLS-1$
        if (!st.hasMoreTokens()) {
            throw new JAXRException("Invalid concept path: " + pPath); //$NON-NLS-1$
        }
        final String classificationSchemeName = st.nextToken();
        if (!st.hasMoreTokens()) {
            throw new JAXRException("Invalid classification scheme " //$NON-NLS-1$
                    + classificationSchemeName + " in path " + pPath); //$NON-NLS-1$
        }
        final ClassificationScheme cs = findClassificationSchemeByName(null, classificationSchemeName);
        if (cs == null) {
            return null;
        }
        Concept concept = findConcept(cs.getChildrenConcepts(), st.nextToken());
        if (concept == null) {
            return null;
        }
        while (st.hasMoreTokens()) {
            concept = findConcept(concept.getChildrenConcepts(), st.nextToken());
            if (concept == null) {
                return null;
            }
        }
        return concept;
	}

	@Override
	public BulkResponse findConcepts(@SuppressWarnings("rawtypes") Collection pFindQualifiers,
	        @SuppressWarnings("rawtypes") Collection pNamePatterns, @SuppressWarnings("rawtypes") Collection pClassifications,
	        @SuppressWarnings("rawtypes") Collection pExternalIdentifiers, @SuppressWarnings("rawtypes") Collection pExternalLinks)
			throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public BulkResponse findOrganizations(@SuppressWarnings("rawtypes") Collection pFindQualifiers,
	        @SuppressWarnings("rawtypes") Collection pNamePatterns, @SuppressWarnings("rawtypes") Collection pClassifications,
	        @SuppressWarnings("rawtypes") Collection pSpecifications, @SuppressWarnings("rawtypes") Collection pExternalIdentifiers,
	        @SuppressWarnings("rawtypes") Collection pExternalLinks) throws JAXRException {
		final List<Organization> list = OrganizationAccessor.getInstance().findOrganizations(getRegistryService(),
				pFindQualifiers, pNamePatterns, pClassifications,
				pSpecifications, pExternalIdentifiers,
				pExternalLinks);
		return new BulkResponseImpl(list);
	}

	@Override
	public BulkResponse findRegistryPackages(@SuppressWarnings("rawtypes") Collection pFindQualifiers,
	        @SuppressWarnings("rawtypes") Collection pNamePatterns, @SuppressWarnings("rawtypes") Collection pClassifications,
	        @SuppressWarnings("rawtypes") Collection pExternalLinks) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public BulkResponse findServiceBindings(Key pServiceKey,
	        @SuppressWarnings("rawtypes") Collection pFindQualifiers, @SuppressWarnings("rawtypes") Collection pClassifications,
			@SuppressWarnings("rawtypes") Collection pSpecificationa) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public BulkResponse findServices(Key pOrgKey, @SuppressWarnings("rawtypes") Collection pFindQualifiers,
	        @SuppressWarnings("rawtypes") Collection pNamePattersn, @SuppressWarnings("rawtypes") Collection pClassifications,
	        @SuppressWarnings("rawtypes") Collection pSpecificationa) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public BulkResponse findObjects(Collection<String> pFindQualifiers,
			Collection<String> pObjectTypes, Collection<String> pNamePatterns,
			Collection<String> pDescriptionPatterns,
			Collection<?> pClassifications, Collection<?> pExternalIdentifiers,
			Collection<?> pExternalLinks) throws JAXRException {
		final List<RegistryEntry> list = RegistryEntryAccessor.getInstance().findRegistryEntries(getRegistryService(),
				pFindQualifiers,
				pObjectTypes, pNamePatterns, pDescriptionPatterns, pClassifications,
				pExternalIdentifiers, pExternalLinks);
		return new BulkResponseImpl(list);
	}
}
