/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.infomodel;

import javax.xml.registry.JAXRException;
import javax.xml.registry.infomodel.Classification;
import javax.xml.registry.infomodel.ClassificationScheme;
import javax.xml.registry.infomodel.Concept;
import javax.xml.registry.infomodel.Key;
import javax.xml.registry.infomodel.RegistryObject;

import org.apache.labs.jaxmas.registry.accessor.ClassificationAccessor;
import org.apache.labs.jaxmas.registry.accessor.ROAccessor;


/**
 * Default implementation of {@link Classification}.
 */
public class ClassificationImpl extends OwnedRegistryObject<Classification> implements Classification {
	private Key conceptKey;
	private Concept concept;
	private int num;

	/**
	 * Creates a new instance.
	 */
	public ClassificationImpl(RegistryServiceImpl pRegistryService, Key pKey) {
		super(pRegistryService, pKey);
	}

	/**
	 * Returns the classifications number within its parents list of classifications.
	 */
	public int getNum() {
		return num;
	}

	/**
	 * Sets the classifications number within its parents list of classifications.
	 */
	public void setNum(int pNum) {
		num = pNum;
	}

	@Override
	public ClassificationScheme getClassificationScheme() throws JAXRException {
		return getConcept().getClassificationScheme();
	}

	@Override
	public RegistryObject getClassifiedObject() throws JAXRException {
		return getOwner();
	}

	@Override
	public String getValue() throws JAXRException {
	    return getConcept().getValue();
	}

	@Override
	public boolean isExternal() throws JAXRException {
		return false;
	}

	@Override
	public void setClassificationScheme(ClassificationScheme pClassificationScheme) throws JAXRException {
	    throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public void setClassifiedObject(RegistryObject pClassifiedObject) throws JAXRException {
	    setOwner(pClassifiedObject);
	}

    @Override
    public Concept getConcept() throws JAXRException {
        ensureLoaded();
        if (concept == null) {
            if (conceptKey != null) {
                concept = (Concept) getRegistryService().getBusinessQueryManager().getRegistryObject(conceptKey.getId());
            }
        }
        return concept;
    }

	@Override
	public void setConcept(Concept pConcept) throws JAXRException {
	    ensureLoaded();
	    if (pConcept == null) {
	        conceptKey = null;
	        concept = null;
	    } else {
	        conceptKey = pConcept.getKey();
	        concept = pConcept;
	    }
	}

	/**
	 * Sets the concepts key.
	 */
	public void setConceptKey(Key pKey) throws JAXRException {
	    ensureLoaded();
	    conceptKey = pKey;
	    if (pKey == null  ||  (concept != null  &&  !concept.getKey().equals(pKey))) {
	        concept = null;
	    }
	}
	
	@Override
	public void setValue(String pValue) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

    @Override
    public ROAccessor<Classification> getROLoader() {
        return ClassificationAccessor.getInstance();
    }

	@Override
	public boolean isRestricting() {
		return false;
	}
}
