/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.infomodel;

import java.lang.reflect.UndeclaredThrowableException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.xml.registry.JAXRException;
import javax.xml.registry.RegistryService;
import javax.xml.registry.infomodel.Concept;

import org.apache.labs.jaxmas.registry.accessor.ConceptAccessor;
import org.apache.labs.jaxmas.registry.accessor.ROAccessors;
import org.apache.labs.jaxmas.registry.sql.FilterPredicate;
import org.apache.labs.jaxmas.registry.sql.OwnerPredicate;
import org.apache.labs.jaxmas.registry.sql.RegistryObjectLoader;


/**
 * An object of this class is used to maintain the child concept lists
 * of classification schemes and concepts.
 */
class ConceptChildrenController {
    private static final RegistryObjectLoader rol;
    static {
        try {
            rol = new RegistryObjectLoader(ROAccessors.ObjectTypes.CONCEPT, null, null);
        } catch (SQLException e) {
            throw new UndeclaredThrowableException(e);
        } catch (JAXRException e) {
            throw new UndeclaredThrowableException(e);
        }
    }

    private final RegistryObjectImpl<?> registryObject;
	private List<Concept> storedChildren = new ArrayList<Concept>();
	private Collection<Concept> children;

	ConceptChildrenController(RegistryObjectImpl<?> pRegistryObject) {
		registryObject = pRegistryObject;
	}

	private void initStoredChildren() {
		storedChildren.clear();
		if (children != null) {
			storedChildren.addAll(children);
		}
	}

	private Collection<Concept> getChildren() throws JAXRException {
		if (children == null) {
		    final FilterPredicate predicate = new OwnerPredicate(registryObject.getKey(), true);
		    children = asConceptCollection(rol.getResultList(registryObject.getRegistryService(), Collections.singleton(predicate), null));
			initStoredChildren();
		}
		return children;
	}
	
	void addChildConcept(Concept pConcept) throws JAXRException {
		if (pConcept == null) {
			throw new IllegalArgumentException(registryObject.getNLSStrings().format(NLSStrings.NULL_PARAMETER, "pConcept")); //$NON-NLS-1$
		}
        ((ConceptImpl) pConcept).setOwner(registryObject);
		getChildren().add(pConcept);
	}

	void addChildConcepts(Collection<Concept> pConcepts) throws JAXRException {
		if (pConcepts == null) {
			throw new IllegalArgumentException(registryObject.getNLSStrings().format(NLSStrings.NULL_PARAMETER, "pConcepts")); //$NON-NLS-1$
		}
		for (Concept c : pConcepts) {
			addChildConcept(c);
		}
	}

	int getChildConceptCount() throws JAXRException {
		return getChildren().size();
	}

	Collection<Concept> getChildrenConcepts() throws JAXRException {
		return Collections.unmodifiableCollection(getChildren());
	}

	void removeChildConcept(Concept pConcept) throws JAXRException {
		if (pConcept == null) {
			throw new IllegalArgumentException(registryObject.getNLSStrings().format(NLSStrings.NULL_PARAMETER, "pConcept")); //$NON-NLS-1$
		}
		getChildren().remove(pConcept);
	}

	void removeChildConcepts(Collection<Concept> pConcepts) throws JAXRException {
		if (pConcepts == null) {
			throw new IllegalArgumentException(registryObject.getNLSStrings().format(NLSStrings.NULL_PARAMETER, "pConcepts")); //$NON-NLS-1$
		}
		for (Object o : pConcepts) {
			removeChildConcept((Concept) o);
		}
	}

	@SuppressWarnings("unchecked")
	private static final Collection<Concept> asConceptCollection(Collection<?> pConcepts) {
		return (Collection<Concept>) pConcepts;
	}

	private void addDescendantConcepts(Collection<Concept> pTargetCollection, Collection<Concept> pChildrenCollection) throws JAXRException {
		if (pChildrenCollection != null) {
			for (Concept concept : pChildrenCollection) {
				pTargetCollection.add(concept);
				addDescendantConcepts(pTargetCollection, asConceptCollection(concept.getChildrenConcepts()));
			}
		}
	}

	Collection<Concept> getDescendantConcepts() throws JAXRException {
		final List<Concept> descendants = new ArrayList<Concept>();
		addDescendantConcepts(descendants, getChildrenConcepts());
		return Collections.unmodifiableCollection(descendants);
	}

	/**
	 * Called to save the children list.
	 */
	public void save() throws JAXRException {
		if (children == null) {
			return; // Nothing to do
		}
		final ConceptAccessor conceptAccessor = ROAccessors.getConceptAccessor();
		final RegistryService registryService = registryObject.getRegistryService();

		int i = 0;
		for (Concept concept : children) {
			for (Iterator<Concept> iter = storedChildren.iterator();  iter.hasNext();  ) {
				final Concept child = iter.next();
				if (child.getKey().equals(concept.getKey())) {
					iter.remove();
					break;
				}
			}
			conceptAccessor.save(registryService, concept);
			i++;
		}
		for (Concept concept : storedChildren) {
			conceptAccessor.remove(registryService, concept);
		}
		initStoredChildren();
	}
}
