/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.infomodel;

import java.util.Collection;

import javax.xml.registry.JAXRException;
import javax.xml.registry.infomodel.ClassificationScheme;
import javax.xml.registry.infomodel.Concept;
import javax.xml.registry.infomodel.Key;
import javax.xml.registry.infomodel.RegistryObject;

import org.apache.labs.jaxmas.registry.accessor.ConceptAccessor;
import org.apache.labs.jaxmas.registry.accessor.ROAccessor;


/**
 * Default implementation of {@link Concept}.
 */
public class ConceptImpl extends OwnedRegistryObject<Concept> implements Concept {
	private String value;
	private final ConceptChildrenController childrenConceptList = new ConceptChildrenController(this);

	/**
	 * Creates a new instance.
	 */
	public ConceptImpl(RegistryServiceImpl pRegistryService, Key pKey) {
		super(pRegistryService, pKey);
	}

	@Override
	public Concept getParentConcept() throws JAXRException {
		final RegistryObject p = getParentConcept();
		if (p instanceof Concept) {
			return (Concept) p;
		}
		return null;
	}

	@Override
	public String getPath() throws JAXRException {
		final Concept parentConcept = getParentConcept();
		if (parentConcept == null) {
			return "/" + getValue(); //$NON-NLS-1$
		}
		return parentConcept.getPath() + "/" + getValue(); //$NON-NLS-1$
	}

	@Override
	public String getValue() throws JAXRException {
	    ensureLoaded();
		return value;
	}

	@Override
	public void setValue(String pValue) throws JAXRException {
	    ensureLoaded();
		value = pValue;
	}

	@Override
	public ClassificationScheme getClassificationScheme() throws JAXRException {
        final RegistryObject owner = getOwner();
        return (owner instanceof ClassificationScheme) ? ((ClassificationScheme) owner) : ((Concept) owner).getClassificationScheme();
	}

	@Override
	public RegistryObject getParent() throws JAXRException {
		return getOwner();
	}

	@Override
	public void addChildConcept(Concept pConcept) throws JAXRException {
		childrenConceptList.addChildConcept(pConcept);
	}

	@SuppressWarnings("unchecked")
	private static final Collection<Concept> asConceptCollection(Collection<?> pConcepts) {
		return (Collection<Concept>) pConcepts;
	}

	@Override
	public void addChildConcepts(@SuppressWarnings("rawtypes") Collection pConcepts) throws JAXRException {
		childrenConceptList.addChildConcepts(asConceptCollection(pConcepts));
	}

	@Override
	public int getChildConceptCount() throws JAXRException {
		return childrenConceptList.getChildConceptCount();
	}

	@SuppressWarnings("rawtypes")
	@Override
	public Collection getChildrenConcepts() throws JAXRException {
		return childrenConceptList.getChildrenConcepts();
	}

	@SuppressWarnings("rawtypes")
	@Override
	public Collection getDescendantConcepts() throws JAXRException {
		return childrenConceptList.getDescendantConcepts();
	}

	@Override
	public void removeChildConcept(Concept pConcept) throws JAXRException {
		childrenConceptList.removeChildConcept(pConcept);
	}

	@SuppressWarnings("rawtypes")
	@Override
	public void removeChildConcepts(Collection pConcepts) throws JAXRException {
		childrenConceptList.removeChildConcepts(asConceptCollection(pConcepts));
	}

	@Override
    public ROAccessor<Concept> getROLoader() {
		return ConceptAccessor.getInstance();
	}

	@Override
	public boolean isRestricting() {
		return true;
	}
}
