/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.infomodel;

import java.sql.SQLException;
import java.util.Locale;
import java.util.Set;

import javax.xml.registry.Connection;
import javax.xml.registry.JAXRException;
import javax.xml.registry.RegistryService;


/**
 * Implementation of {@link Connection}.
 */
public class ConnectionImpl implements Connection {
	private final NLSStrings nlsStrings;
	private final RegistryService registryService;
	private java.sql.Connection connection;
	private Set<?> credentials;
	private boolean synchronous;

	/**
	 * Creates a new instance, which uses the given
	 * {@link java.sql.Connection SQL connection}.
	 */
	public ConnectionImpl(java.sql.Connection pConnection, NLSStrings pNLSStrings) throws SQLException {
		connection = pConnection;
		nlsStrings = pNLSStrings;
		registryService = new RegistryServiceImpl(this);
	}

	@Override
	public void close() throws JAXRException {
		if (connection != null) {
			try {
				connection.close();
			} catch (SQLException e) {
				throw new JAXRException(e);
			}
			connection = null;
		}
	}

	@SuppressWarnings("rawtypes")
	@Override
	public Set getCredentials() throws JAXRException {
		return credentials;
	}

	@Override
	public RegistryService getRegistryService() throws JAXRException {
		return registryService;
	}

	@Override
	public boolean isClosed() throws JAXRException {
		return connection == null;
	}

	@Override
	public boolean isSynchronous() throws JAXRException {
		return synchronous;
	}

	@SuppressWarnings("rawtypes")
	@Override
	public void setCredentials(Set pCredentials) throws JAXRException {
		credentials = pCredentials;
	}

	@Override
	public void setSynchronous(boolean pSync) throws JAXRException {
		synchronous = pSync;
	}

	/**
	 * Returns the database connection.
	 */
	public java.sql.Connection getConnection() {
		return connection;
	}

	/**
	 * Returns the users resource bundle.
	 */
	public NLSStrings getNLSStrings() {
		return nlsStrings;
	}

	/**
	 * Returns the users locale.
	 */
	public Locale getLocale() {
		return nlsStrings.getLocale();
	}

	/**
	 * Returns the given connections {@link Locale}. Shortcut for
	 * <pre>
	 *   ((ConnectionImpl) pConnection).getLocale()
	 * </pre>
	 */
	public static Locale getLocale(Connection pConnection) {
		return ((ConnectionImpl) pConnection).getLocale();
	}

	/**
	 * Returns the given connections {@link NLSStrings}. Shortcut for
	 * <pre>
	 *   ((ConnectionImpl) pConnection).getNLSStrings()
	 * </pre>
	 */
	public static NLSStrings getNLSStrings(Connection pConnection) {
		return ((ConnectionImpl) pConnection).getNLSStrings();
	}
}
