/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.infomodel;

import java.util.Collection;
import java.util.Map;

import javax.xml.registry.JAXRException;
import javax.xml.registry.infomodel.ExtensibleObject;
import javax.xml.registry.infomodel.Slot;


/**
 * Default implementation of {@link ExtensibleObject}.
 */
public abstract class ExtensibleObjectImpl implements ExtensibleObject {
	private final RegistryServiceImpl registryService;
	private Map<String,Slot> slts;

	/**
	 * Called to load the extensible objects slots.
	 */
	protected abstract Map<String,Slot> loadSlots() throws JAXRException;

	public boolean hasSlotsLoaded() {
		return slts != null;
	}
	
	private Map<String,Slot> slots() throws JAXRException {
		if (slts == null) {
			slts = loadSlots();
		}
		return slts;
	}

	/**
	 * Creates a new instance with the given registry service.
	 */
	public ExtensibleObjectImpl(RegistryServiceImpl pRegistryService) {
		registryService = pRegistryService;
	}

	/**
	 * Returns the registry service.
	 */
	public RegistryServiceImpl getRegistryService() {
		return registryService;
	}

	@Override
	public void addSlot(Slot pSlot) throws JAXRException {
		slots().put(pSlot.getName(), pSlot);
	}

	@SuppressWarnings("rawtypes")
	@Override
	public void addSlots(Collection pSlots) throws JAXRException {
		for (Object o : pSlots) {
			addSlot((Slot) o);
		}
	}

	@Override
	public Slot getSlot(String pSlotName) throws JAXRException {
		return slots().get(pSlotName);
	}

	@Override
	public Collection<?> getSlots() throws JAXRException {
		return slots().values();
	}

	@Override
	public void removeSlot(String pSlotName) throws JAXRException {
		slots().remove(pSlotName);
	}

	@SuppressWarnings("rawtypes")
	@Override
	public void removeSlots(Collection pSlotNames) throws JAXRException {
		for (Object o : pSlotNames) {
			removeSlot((String) o);
		}
	}
}
