/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.infomodel;

import javax.xml.registry.JAXRException;
import javax.xml.registry.infomodel.InternationalString;
import org.apache.labs.jaxmas.registry.infomodel.InternationalStringImpl.Type;

import org.apache.labs.jaxmas.registry.sql.DbDriver;


/**
 * An object, which is used to provide access to an
 * {@link InternationalString}.
 */
public class InternationalStringController {
    private ROState state;
    private final Type type;
    private final RegistryObjectImpl<?> ro;
	private InternationalString string;

	/**
	 * Creates a new instance with the given registry object.
	 */
	public InternationalStringController(RegistryObjectImpl<?> pRo, Type pType, ROState pState) {
		ro = pRo;
		type = pType;
		state = pState;
	}

	/**
	 * Sets the objects state.
	 */
	public void setState(ROState pState) {
	    state = pState;
	}

	private void ensureLoaded() throws JAXRException {
	    switch (state) {
	        case created:
	        case loaded:
            case deleted:
            	if (string == null) {
            		string = new InternationalStringImpl();
            	}
            	break;
	        case referenced:
	            string = getDbDriver().loadInternationalString(ro.getKey(), type);
	            state = ROState.loaded;
	            break;
	    }
	}

	/**
	 * Returns the string value.
	 */
	public InternationalString getString() throws JAXRException {
	    ensureLoaded();
		return string;
	}

	/**
	 * Sets the string value.
	 */
	public void setString(InternationalString pString) throws JAXRException {
	    ensureLoaded();
	    string = pString;
	}

	private DbDriver getDbDriver() {
		return RegistryServiceImpl.getDbDriver(ro.getRegistryService());
	}

	/**
	 * Writes the instance to the registry.
	 */
	public void save() throws JAXRException {
	    switch (state) {
	        case created:
	            getDbDriver().insertInternationalString(ro.getKey(), type, string);
	            state = ROState.loaded;
	            break;
	        case referenced:
	            // Nothing to do.
	            break;
	        case deleted:
	            throw new IllegalStateException("Unable to save an object, which has already been deleted."); //$NON-NLS-1$
	        case loaded:
				getDbDriver().updateInternationalString(ro.getKey(), type, string);
		}
	}
}
