/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.infomodel;

import java.util.Collection;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import javax.xml.registry.JAXRException;
import javax.xml.registry.infomodel.InternationalString;
import javax.xml.registry.infomodel.LocalizedString;


/**
 * Default implementation of {@link InternationalString}.
 */
public class InternationalStringImpl implements InternationalString {
    /**
     * This enumeration is used to control, whether the {@link InternationalString}
     * is an objects description, or its name.
     */
    public enum Type {
        /**
         * Indicates, that the {@link InternationalString} is the objects name.
         */
        name,
        /**
         * Indicates, that the {@link InternationalString} is the objects type.
         */
        description
    }

    private final Map<String,LocalizedString> localizedStrings = new HashMap<String,LocalizedString>();

	private String makeKey(Locale pLocale, String pCharsetName) {
		final String charsetName = pCharsetName == null ? LocalizedString.DEFAULT_CHARSET_NAME : pCharsetName;
		final Locale locale = pLocale == null ? Locale.getDefault() : pLocale;
		return locale.toString() + ":" + charsetName; //$NON-NLS-1$
	}

	private String makeKey(LocalizedString pLocalizedString) throws JAXRException {
		return makeKey(pLocalizedString.getLocale(), pLocalizedString.getCharsetName());
	}


	@Override
	public void addLocalizedString(LocalizedString pLocalizedString) throws JAXRException {
		localizedStrings.put(makeKey(pLocalizedString), pLocalizedString);
	}

	@SuppressWarnings("rawtypes")
	@Override
	public void addLocalizedStrings(Collection pLocalizedStrings) throws JAXRException {
		for (Object o : pLocalizedStrings) {
			addLocalizedString((LocalizedString) o);
		}
	}

	@Override
	public LocalizedString getLocalizedString(Locale locale, String charsetName) throws JAXRException {
		return localizedStrings.get(makeKey(locale, charsetName));
	}

    @SuppressWarnings("rawtypes")
	@Override
	public Collection getLocalizedStrings() throws JAXRException {
		return localizedStrings.values();
	}

	@Override
	public String getValue() throws JAXRException {
		return getValue(Locale.getDefault());
	}

	@Override
	public String getValue(Locale locale) throws JAXRException {
		final LocalizedString ls = localizedStrings.get(makeKey(locale,LocalizedString.DEFAULT_CHARSET_NAME));
		return ls == null ? null : ls.getValue();
	}

	@Override
	public void removeLocalizedString(LocalizedString pLocalizedString) throws JAXRException {
		localizedStrings.remove(makeKey(pLocalizedString));
	}

    @SuppressWarnings("rawtypes")
	@Override
	public void removeLocalizedStrings(Collection pLocalizedStrings) throws JAXRException {
		for (Object o : pLocalizedStrings) {
			removeLocalizedString((LocalizedString) o);
		}
	}

	@Override
	public void setValue(String pValue) throws JAXRException {
		setValue(Locale.getDefault(), pValue);
	}

	@Override
	public void setValue(Locale pLocale, String pValue) throws JAXRException {
		final LocalizedString ls = new LocalizedStringImpl();
		ls.setCharsetName(LocalizedString.DEFAULT_CHARSET_NAME);
		ls.setLocale(pLocale == null ? Locale.getDefault() : pLocale);
		ls.setValue(pValue);
		addLocalizedString(ls);
	}
}
