/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.infomodel;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;

import javax.activation.DataHandler;
import javax.xml.registry.BulkResponse;
import javax.xml.registry.InvalidRequestException;
import javax.xml.registry.JAXRException;
import javax.xml.registry.LifeCycleManager;
import javax.xml.registry.RegistryService;
import javax.xml.registry.UnsupportedCapabilityException;
import javax.xml.registry.infomodel.Association;
import javax.xml.registry.infomodel.Classification;
import javax.xml.registry.infomodel.ClassificationScheme;
import javax.xml.registry.infomodel.Concept;
import javax.xml.registry.infomodel.EmailAddress;
import javax.xml.registry.infomodel.ExternalIdentifier;
import javax.xml.registry.infomodel.ExternalLink;
import javax.xml.registry.infomodel.ExtrinsicObject;
import javax.xml.registry.infomodel.InternationalString;
import javax.xml.registry.infomodel.Key;
import javax.xml.registry.infomodel.LocalizedString;
import javax.xml.registry.infomodel.Organization;
import javax.xml.registry.infomodel.PersonName;
import javax.xml.registry.infomodel.PostalAddress;
import javax.xml.registry.infomodel.RegistryObject;
import javax.xml.registry.infomodel.RegistryPackage;
import javax.xml.registry.infomodel.Service;
import javax.xml.registry.infomodel.ServiceBinding;
import javax.xml.registry.infomodel.Slot;
import javax.xml.registry.infomodel.SpecificationLink;
import javax.xml.registry.infomodel.TelephoneNumber;
import javax.xml.registry.infomodel.User;

import org.apache.labs.jaxmas.registry.accessor.AssociationAccessor;
import org.apache.labs.jaxmas.registry.accessor.ClassificationAccessor;
import org.apache.labs.jaxmas.registry.accessor.ConceptAccessor;
import org.apache.labs.jaxmas.registry.accessor.OrganizationAccessor;
import org.apache.labs.jaxmas.registry.accessor.ROAccessor;
import org.apache.labs.jaxmas.registry.accessor.ROAccessors;


/**
 * Default implementation of {@link LifeCycleManager}.
 */
public abstract class LifeCycleManagerImpl implements LifeCycleManager {
	private final RegistryServiceImpl registryService;

	/**
	 * Creates a new instance with the given registry service.
	 */
	protected LifeCycleManagerImpl(RegistryServiceImpl pRegistryService) {
		registryService = pRegistryService;
	}

	@Override
	public Association createAssociation(RegistryObject pTargetObject, Concept pAssociationType) throws JAXRException {
	    final Association assoc = AssociationAccessor.getInstance().create(getRegistryService());
	    assoc.setTargetObject(pTargetObject);
	    assoc.setAssociationType(pAssociationType);
	    return assoc;
	}

	@Override
	public Classification createClassification(ClassificationScheme pScheme,
			InternationalString pName, String pValue) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public Classification createClassification(ClassificationScheme pScheme, String pName, String pValue) throws JAXRException {
	    throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public Classification createClassification(Concept pConcept) throws JAXRException, InvalidRequestException {
	    final Classification cl = ClassificationAccessor.getInstance().create(getRegistryService());
	    cl.setConcept(pConcept);
	    return cl;
	}

	@Override
	public ClassificationScheme createClassificationScheme(Concept pConcept)
			throws JAXRException, InvalidRequestException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public ClassificationScheme createClassificationScheme(InternationalString pName, InternationalString pDescription)
			throws JAXRException, InvalidRequestException {
		final ClassificationSchemeImpl cs = new ClassificationSchemeImpl(registryService, null);
		cs.setKey(cs.getROLoader().newKey(registryService));
		cs.setName(pName);
		cs.setDescription(pDescription);
		return cs;
	}

	@Override
	public ClassificationScheme createClassificationScheme(String pName, String pDescription)
			throws JAXRException, InvalidRequestException {
		final InternationalString name = pName == null ? null : createInternationalString(pName);
		final InternationalString desc = pDescription == null ? null : createInternationalString(pDescription);
		return createClassificationScheme(name, desc);
	}

	@Override
	public Concept createConcept(RegistryObject pParent,
			InternationalString pName, String pValue) throws JAXRException {
	    final Concept concept = ConceptAccessor.getInstance().create(registryService);
	    concept.setName(pName);
	    concept.setValue(pValue);
	    if (pParent instanceof ClassificationScheme) {
	        ((ClassificationScheme) pParent).addChildConcept(concept);
	    } else if (pParent instanceof Concept) {
	        ((Concept) pParent).addChildConcept(concept);
	    } else {
	        throw new IllegalStateException("Invalid parent: " + pParent.getKey() //$NON-NLS-1$
	                + "[" + pParent.getClass().getName() + "]"); //$NON-NLS-1$ //$NON-NLS-2$
	    }
	    ((ConceptImpl) concept).setOwner(pParent);
	    return concept;
	}

	@Override
	public Concept createConcept(RegistryObject pParent, String pName,
			String pValue) throws JAXRException {
	    final InternationalString name = pName == null ? null : createInternationalString(pName);
	    return createConcept(pParent, name, pValue);
	}

	@Override
	public EmailAddress createEmailAddress(String pAddress, String pType) throws JAXRException {
		final EmailAddress ea = new EmailAddressImpl();
		ea.setType(pType);
		ea.setAddress(pAddress);
		return ea;
	}

	@Override
	public EmailAddress createEmailAddress(String pAddress) throws JAXRException {
	    final EmailAddressImpl emailAddress = new EmailAddressImpl();
	    emailAddress.setAddress(pAddress);
	    return emailAddress;
	}

	@Override
	public ExternalIdentifier createExternalIdentifier(
			ClassificationScheme pIdentificationScheme,
			InternationalString pName, String pValue) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public ExternalIdentifier createExternalIdentifier(
			ClassificationScheme pIdentificationScheme, String pName,
			String pValue) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public ExternalLink createExternalLink(String pExternalURI,
			InternationalString pDescription) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public ExternalLink createExternalLink(String pExternalURI,
			String pDescription) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public ExtrinsicObject createExtrinsicObject(DataHandler pRepositoryItem)
			throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public InternationalString createInternationalString() throws JAXRException {
		return new InternationalStringImpl();
	}

	@Override
	public InternationalString createInternationalString(Locale pLocale, String pValue) throws JAXRException {
		final InternationalString is = createInternationalString();
		is.setValue(pLocale, pValue);
		return is;
	}

	@Override
	public InternationalString createInternationalString(String pValue) throws JAXRException {
		final InternationalString is = createInternationalString();
		is.setValue(pValue);
		return is;
	}

	@Override
	public Key createKey(String pId) throws JAXRException {
		final KeyImpl key = new KeyImpl();
		key.setId(pId);
		return key;
	}

	@Override
	public LocalizedString createLocalizedString(Locale pLocale, String pValue) throws JAXRException {
		final LocalizedStringImpl ls = new LocalizedStringImpl();
		ls.setLocale(pLocale);
		ls.setValue(pValue);
		return ls;
	}

	@Override
	public LocalizedString createLocalizedString(Locale pLocale, String pValue, String pCharsetName) throws JAXRException {
		final LocalizedString ls = createLocalizedString(pLocale, pValue);
		ls.setCharsetName(pCharsetName);
		return ls;
	}

	@Override
	public Object createObject(String pInterfaceName) throws JAXRException,
			InvalidRequestException, UnsupportedCapabilityException {
	    final ROAccessor<?> roAcc = ROAccessors.getROAccessor(pInterfaceName);
	    if (roAcc == null) {
	        if (LifeCycleManager.EMAIL_ADDRESS.equals(pInterfaceName)) {
	            return new EmailAddressImpl();
	        }
	        if (LifeCycleManager.INTERNATIONAL_STRING.equals(pInterfaceName)) {
	            return createInternationalString();
	        }
	        if (LifeCycleManager.KEY.equals(pInterfaceName)) {
	            return ConceptAccessor.getInstance().newKey(getRegistryService());
	        }
	        if (LifeCycleManager.LOCALIZED_STRING.equals(pInterfaceName)) {
	            final LocalizedStringImpl ls = new LocalizedStringImpl();
	            ls.setLocale(Locale.getDefault());
	            return ls;
	        }
	        if (LifeCycleManager.PERSON_NAME.equals(pInterfaceName)) {
	            return new PersonNameImpl();
	        }
	        if (LifeCycleManager.SLOT.equals(pInterfaceName)) {
	            return new SlotImpl();
	        }
            if (LifeCycleManager.TELEPHONE_NUMBER.equals(pInterfaceName)) {
                return new TelephoneNumberImpl();
            }
            if (LifeCycleManager.VERSIONABLE.equals(pInterfaceName)  ||  LifeCycleManager.REGISTRY_ENTRY.equals(pInterfaceName)) {
                return ROAccessors.getROAccessor(ROAccessors.ObjectTypes.REGISTRY_ENTRY).create(getRegistryService());
            }
            throw new UnsupportedCapabilityException("Not implemented: " + pInterfaceName); //$NON-NLS-1$
	    }
	    return roAcc.create(getRegistryService());
	}

	@Override
	public Organization createOrganization(InternationalString pName) throws JAXRException {
	    final Organization org = OrganizationAccessor.getInstance().create(getRegistryService());
	    org.setName(pName);
	    return org;
	}

	@Override
	public Organization createOrganization(String pName) throws JAXRException {
	    return createOrganization(createInternationalString(pName));
	}

	@Override
	public PersonName createPersonName(String pFullName) throws JAXRException {
		final PersonName pn = new PersonNameImpl();
		pn.setFullName(pFullName);
		return pn;
	}

	@Override
	public PersonName createPersonName(String pFirstName, String pMiddleName, String pLastName) throws JAXRException {
		final PersonName pn = new PersonNameImpl();
		pn.setFirstName(pFirstName);
		pn.setMiddleName(pMiddleName);
		pn.setLastName(pLastName);
		return pn;
	}

	@Override
	public PostalAddress createPostalAddress(String pStreetNumber,
			String pStreet, String pCity, String pStateOrProvince,
			String pCountry, String pPostalCode, String pType)
			throws JAXRException {
		final PostalAddress pa = new PostalAddressImpl((RegistryServiceImpl) getRegistryService());
		pa.setStreetNumber(pStreetNumber);
		pa.setStreet(pStreet);
		pa.setCity(pCity);
		pa.setStateOrProvince(pStateOrProvince);
		pa.setCountry(pCountry);
		pa.setPostalCode(pPostalCode);
		pa.setType(pType);
		return pa;
	}

	@Override
	public RegistryPackage createRegistryPackage(InternationalString pName) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public RegistryPackage createRegistryPackage(String pName) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public Service createService(InternationalString pName) throws JAXRException {
	    throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public Service createService(String pName) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public ServiceBinding createServiceBinding() throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

    @SuppressWarnings("rawtypes")
	@Override
	public Slot createSlot(String pName, Collection pValues, String pSlotType) throws JAXRException {
		final Slot slot = new SlotImpl();
		slot.setName(pName);
		slot.setSlotType(pSlotType);
		slot.setValues(pValues);
		return slot;
	}

	@Override
	public Slot createSlot(String pName, String pValue, String pSlotType) throws JAXRException {
		return createSlot(pName, Collections.singleton(pValue), pSlotType);
	}

	@Override
	public SpecificationLink createSpecificationLink() throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public TelephoneNumber createTelephoneNumber() throws JAXRException {
		return new TelephoneNumberImpl();
	}

	@Override
	public User createUser() throws JAXRException {
		return new UserImpl(registryService, null);
	}

    @SuppressWarnings("rawtypes")
	@Override
	public BulkResponse deleteObjects(Collection pKeys, String pObjectType) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

    @SuppressWarnings({ "rawtypes", "unchecked" })
	@Override
	public BulkResponse deleteObjects(Collection pKeys) throws JAXRException {
	    for (Object o : pKeys) {
	        final Key key = (Key) o;
	        final RegistryObject ro = getRegistryService().getBusinessQueryManager().getRegistryObject(key.getId());
	        ((RegistryObjectImpl) ro).getROLoader().remove(getRegistryService(), ro);
	    }
	    return new BulkResponseImpl(pKeys);
	}

    @SuppressWarnings("rawtypes")
	@Override
	public BulkResponse deprecateObjects(Collection pKeys) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public RegistryService getRegistryService() throws JAXRException {
		return registryService;
	}

	/**
	 * Saves a single object.
	 */
	protected void saveObject(RegistryObject pObject) throws JAXRException {
		if (pObject == null) {
			throw new IllegalArgumentException(getNLSStrings().format(NLSStrings.NULL_PARAMETER, "pObject")); //$NON-NLS-1$
		}
		((RegistryObjectImpl<?>) pObject).getROLoader().save(getRegistryService(), pObject);
	}

    @SuppressWarnings("rawtypes")
	@Override
	public BulkResponse saveObjects(Collection pObjects) throws JAXRException {
		if (pObjects == null) {
			throw new IllegalArgumentException(getNLSStrings().format(NLSStrings.NULL_PARAMETER, "pObjects")); //$NON-NLS-1$
		}
		final List<Key> keys = new ArrayList<Key>();
		for (Object o : pObjects) {
			saveObject((RegistryObject) o);
		}
		return new BulkResponseImpl(keys);
	}

    @SuppressWarnings("rawtypes")
	@Override
	public BulkResponse unDeprecateObjects(Collection pKeys) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	/**
	 * Returns the managers {@link Locale}.
	 */
	protected Locale getLocale() {
		return registryService.getConnection().getLocale();
	}

	/**
	 * Returns the managers {@link NLSStrings}.
	 */
	protected NLSStrings getNLSStrings() {
		return registryService.getConnection().getNLSStrings();
	}
}
