/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.infomodel;

import java.util.Formatter;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;


/**
 * This class provides access to the localized strings.
 */
public class NLSStrings {
	private static final Map<Locale,ResourceBundle> bundles = new HashMap<Locale,ResourceBundle>();

	/**
	 * This string constant indicates, that the user attempts to perform
	 * an operation, which is not implemented.
	 */
	public static final String OPERATION_NOT_IMPLEMENTED = "OPERATION_NOT_IMPLEMENTED"; //$NON-NLS-1$

	/**
	 * This string constant indicates, that a property value is not set.
	 */
	public static final String PROPERTY_NOT_SET = "PROPERTY_NOT_SET"; //$NON-NLS-1$

	/**
	 * This string constant indicates, that a required parameter is not set, or was null.
	 */
	public static final String NULL_PARAMETER = "NULL_PARAMETER";  //$NON-NLS-1$

	/**
	 * This string constant indicates, that a parameter is not supported.
	 */
	public static final String UNSUPPORTED_PARAMETER = "UNSUPPORTED_PARAMETER"; //$NON-NLS-1$

	/**
	 * This string constant indicates, that a concepts parent is not set.
	 */
	public static final String CONCEPT_PARENT_MISSING = "CONCEPT_PARENT_MISSING"; //$NON-NLS-1$

	/**
	 * This string constant indicates, that a concepts value is not set.
	 */
	public static final String CONCEPT_VALUE_MISSING = "CONCEPT_VALUE_MISSING"; //$NON-NLS-1$

    /**
     * This string constant indicates, that multiple result elements have been found,
     * where only one was expected.
     */
    public static final String MULTIPLE_INSTANCES_FOUND = "MULTIPLE_INSTANCES_FOUND"; //$NON-NLS-1$

	private final Locale locale;
	private final ResourceBundle bundle;

	private String getPackageName() {
		final String className = NLSStrings.class.getName();
		final int offset = className.lastIndexOf('.');
		if (offset == -1) {
			throw new IllegalStateException("Failed to parse package name: " + className); //$NON-NLS-1$
		}
		return className.substring(0, offset);
	}

	/**
	 * Creates a new instance with the given locale.
	 */
	public NLSStrings(Locale pLocale) {
		ResourceBundle myBundle;
		synchronized(bundles) {
			myBundle = bundles.get(pLocale);
			if (myBundle == null) {
				myBundle = ResourceBundle.getBundle(getPackageName().replace('.','/') + "/NLSStrings", pLocale); //$NON-NLS-1$
				bundles.put(pLocale, myBundle);
			}
		}
		bundle = myBundle;
		locale = pLocale;
	}

	/**
	 * Reads a string with the given key from the bundle.
	 */
	public String get(String pKey) {
		return (String) bundle.getObject(pKey);
	}

	/**
	 * Returns a formatted string.
	 * @param pKey Key of the format, which is being passed to {@link #get(String)}.
	 * @param pArgs Format string arguments.
	 */
	public String format(String pKey, Object... pArgs) {
		final StringBuilder sb = new StringBuilder();
		final Formatter formatter = new Formatter(sb);
		formatter.format(get(pKey), pArgs);
		return sb.toString();
	}

	/**
	 * Returns the bundles locale.
	 */
	public Locale getLocale() {
		return locale;
	}
}
