/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.infomodel;

import javax.xml.registry.BusinessQueryManager;
import javax.xml.registry.JAXRException;
import javax.xml.registry.infomodel.Key;
import javax.xml.registry.infomodel.RegistryObject;


/**
 * Abstract base class for {@link ConceptImpl}, {@link AssociationImpl}, and
 * {@link ClassificationImpl}.
 */
public abstract class OwnedRegistryObject<RO extends RegistryObject> extends RegistryObjectImpl<RO> {
    private Key ownerKey;
    private RegistryObject owner;
    private int position;

    /**
     * Returns, whether this object blocks deleting the owner.
     */
    public abstract boolean isRestricting();
    
    /**
     * Returns the objects position within its parent objects list of childs.
     */
    public int getPosition() {
        return position;
    }

    /**
     * Sets the objects position within its parent objects list of childs.
     */
    public void setPosition(int pPosition) {
        position = pPosition;
    }

    /**
     * Creates a new instance with the given {@link RegistryServiceImpl registry service}.
     */
    public OwnedRegistryObject(RegistryServiceImpl pRegistryService, Key pKey) {
        super(pRegistryService, pKey);
    }

    /**
     * Sets the registry objects owner.
     */
    public void setOwner(RegistryObject pOwner) throws JAXRException {
        ensureLoaded();
        owner = pOwner;
        ownerKey = pOwner == null ? null : pOwner.getKey();
    }

    /**
     * Sets the registry objects owner.
     */
    public void setOwner(Key pOwner) throws JAXRException {
        ensureLoaded();
        if (pOwner == null) {
            owner = null;
        } else {
            if (owner != null  &&  !owner.getKey().equals(pOwner)) {
                owner = null;
            }
        }
        ownerKey = pOwner;
    }
    
    /**
     * Returns the registry objects owner.
     */
    public RegistryObject getOwner() throws JAXRException {
        ensureLoaded();
        if (owner == null  &&  ownerKey != null) {
            final BusinessQueryManager bqm = getRegistryService().getBusinessQueryManager();
            owner = bqm.getRegistryObject(ownerKey.getId());
            assert(owner != null);
        }
        return owner;
    }
}
