/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.infomodel;

import java.lang.reflect.UndeclaredThrowableException;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.xml.registry.BulkResponse;
import javax.xml.registry.JAXRException;
import javax.xml.registry.QueryManager;
import javax.xml.registry.RegistryService;
import javax.xml.registry.infomodel.RegistryObject;

import org.apache.labs.jaxmas.registry.sql.FilterPredicate;
import org.apache.labs.jaxmas.registry.sql.KeyPredicate;
import org.apache.labs.jaxmas.registry.sql.RegistryObjectLoader;


/**
 * Default implementation of {@link QueryManager}.
 */
public abstract class QueryManagerImpl implements QueryManager {
	private static final RegistryObjectLoader rol;
	static {
	    try {
	        rol = new RegistryObjectLoader(null, null, null);
	    } catch (JAXRException e) {
	        throw new UndeclaredThrowableException(e);
        } catch (SQLException e) {
            throw new UndeclaredThrowableException(e);
	    }
	}

	private final RegistryServiceImpl registryService;

	/**
	 * Creates a new instance with the given registry service.
	 */
	protected QueryManagerImpl(RegistryServiceImpl pRegistryService) {
		registryService = pRegistryService;
	}
	
	@Override
	public RegistryObject getRegistryObject(String pId) throws JAXRException {
	    final FilterPredicate predicate = new KeyPredicate(Collections.singleton(new KeyImpl(pId)));
	    return rol.getResultObject(getRegistryService(), Collections.singleton(predicate));
	}

	@Override
	public RegistryObject getRegistryObject(String pId, String pObjectType) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public BulkResponse getRegistryObjects() throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public BulkResponse getRegistryObjects(@SuppressWarnings("rawtypes") Collection pObjectKeys) throws JAXRException {
	    final FilterPredicate predicate = new KeyPredicate(pObjectKeys);
	    final List<RegistryObject> list = rol.getResultList(getRegistryService(), Collections.singleton(predicate), null);
	    return new BulkResponseImpl(list);
	}

	@Override
	public BulkResponse getRegistryObjects(String pObjectTypes) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public BulkResponse getRegistryObjects(@SuppressWarnings("rawtypes") Collection pObjectKeys, String pObjectTypes) throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public RegistryService getRegistryService() throws JAXRException {
		return registryService;
	}
}
