/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.infomodel;

import java.util.Date;

import javax.xml.registry.JAXRException;
import javax.xml.registry.infomodel.Classification;
import javax.xml.registry.infomodel.Concept;
import javax.xml.registry.infomodel.Key;
import javax.xml.registry.infomodel.RegistryEntry;

import org.apache.labs.jaxmas.registry.accessor.ROAccessor;
import org.apache.labs.jaxmas.registry.accessor.ROAccessors;
import org.apache.labs.jaxmas.registry.accessor.RegistryEntryAccessor;


/**
 * Default implementation of {@link RegistryEntry}.
 */
public class RegistryEntryImpl<RE extends RegistryEntry> extends RegistryObjectImpl<RE> implements RegistryEntry {
    private Date expiration;
	private int stability, majorVersion, minorVersion, status;
	private String userVersion, customType;
	private Concept customTypeObject;

	/**
	 * Creates a new instance with the given registry service.
	 */
	public RegistryEntryImpl(RegistryServiceImpl pRegistryService, Key pKey) {
		super(pRegistryService, pKey);
	}

	@Override
	public Date getExpiration() throws JAXRException {
		ensureLoaded();
		return expiration;
	}

	@Override
	public int getStability() throws JAXRException {
		ensureLoaded();
		return stability;
	}

	@Override
	public int getStatus() throws JAXRException {
		ensureLoaded();
		return status;
	}

	@Override
	public void setExpiration(Date pExpiration) throws JAXRException {
		ensureLoaded();
		expiration = pExpiration;
	}

	@Override
	public void setStability(int pStability) throws JAXRException {
		ensureLoaded();
		stability = pStability;
	}

	@Override
	public int getMajorVersion() throws JAXRException {
		ensureLoaded();
		return majorVersion;
	}

	@Override
	public int getMinorVersion() throws JAXRException {
		ensureLoaded();
		return minorVersion;
	}

	@Override
	public String getUserVersion() throws JAXRException {
		ensureLoaded();
		return userVersion;
	}

	@Override
	public void setMajorVersion(int pVersion) throws JAXRException {
		ensureLoaded();
		majorVersion = pVersion;
	}

	@Override
	public void setMinorVersion(int pVersion) throws JAXRException {
		ensureLoaded();
		minorVersion = pVersion;
	}

	@Override
	public void setUserVersion(String pVersion) throws JAXRException {
		ensureLoaded();
		userVersion = pVersion;
	}

    @SuppressWarnings("unchecked")
    @Override
    public ROAccessor<RE> getROLoader() {
        return (ROAccessor<RE>) RegistryEntryAccessor.getInstance();
    }

    public String getCustomType() throws JAXRException {
    	ensureLoaded();
    	return customType;
    }

    public void setCustomType(String pCustomType) throws JAXRException {
    	ensureLoaded();
    	customType = pCustomType;
    	customTypeObject = null;
    }

    @Override
    public void addClassification(Classification pClassification) throws JAXRException {
    	final Concept concept = pClassification.getConcept();
    	if (concept.getParent().getKey().getId().equals(ROAccessors.ClassificationSchemes.OBJECT_TYPE.getId())) {
    		setCustomType(concept.getKey().getId());
    	} else {
    		super.addClassification(pClassification);
    	}
    }

    @Override
    public Concept getObjectType() throws JAXRException {
    	final String type = getCustomType();
    	if (type == null) {
    		return super.getObjectType();
    	}
    	if (customTypeObject == null) {
    		customTypeObject = (Concept) getRegistryService().getBusinessQueryManager().getRegistryObject(type);
    	}
    	return customTypeObject;
    }
}
