/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.infomodel;

import java.lang.reflect.UndeclaredThrowableException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import javax.xml.registry.JAXRException;
import javax.xml.registry.LifeCycleManager;
import javax.xml.registry.infomodel.Association;
import javax.xml.registry.infomodel.Classification;
import javax.xml.registry.infomodel.Concept;
import javax.xml.registry.infomodel.ExternalIdentifier;
import javax.xml.registry.infomodel.ExternalLink;
import javax.xml.registry.infomodel.InternationalString;
import javax.xml.registry.infomodel.Key;
import javax.xml.registry.infomodel.Organization;
import javax.xml.registry.infomodel.RegistryObject;
import javax.xml.registry.infomodel.Slot;

import org.apache.labs.jaxmas.registry.accessor.AssociationAccessor;
import org.apache.labs.jaxmas.registry.accessor.ClassificationAccessor;
import org.apache.labs.jaxmas.registry.accessor.ConceptAccessor;
import org.apache.labs.jaxmas.registry.accessor.ROAccessor;
import org.apache.labs.jaxmas.registry.accessor.ROAccessors;
import org.apache.labs.jaxmas.registry.sql.AssociationNumOrderPredicate;
import org.apache.labs.jaxmas.registry.sql.ClassificationNumOrderPredicate;
import org.apache.labs.jaxmas.registry.sql.FilterPredicate;
import org.apache.labs.jaxmas.registry.sql.OrderPredicate;
import org.apache.labs.jaxmas.registry.sql.OwnerPredicate;
import org.apache.labs.jaxmas.registry.sql.RegistryObjectLoader;


/**
 * Default implementation of {@link RegistryObject}.
 */
public abstract class RegistryObjectImpl<RO extends RegistryObject> extends ExtensibleObjectImpl implements RegistryObject {
    private ROState state = ROState.created;
	private LifeCycleManager lifeCycleManager;
	private final InternationalStringController name;
	private final InternationalStringController description;
	private Key key;
	private List<Association> associations;
	private Collection<Key> loadedClassificationKeys, loadedAssociationKeys;
	private List<Classification> classifications;
	private Collection<ExternalIdentifier> externalIdentifiers = new ArrayList<ExternalIdentifier>();
	private Collection<ExternalLink> externalLinks = new ArrayList<ExternalLink>();

	@Override
    public boolean equals(Object pObj) {
	    if (pObj != null  &&  pObj.getClass() == getClass()) {
	        try {
	            return ((RegistryObject) pObj).getKey().equals(getKey());
	        } catch (JAXRException e) {
	            throw new UndeclaredThrowableException(e);
	        }
	    }
	    return false;
    }

    @Override
    public int hashCode() {
        try {
            return getKey().hashCode();
        } catch (JAXRException e) {
            throw new UndeclaredThrowableException(e);
        }
    }

    /**
	 * Creates a new instance with the given registry service.
	 */
	public RegistryObjectImpl(RegistryServiceImpl pRegistryService, Key pKey) {
		super(pRegistryService);
		key = pKey;
		state = pKey == null ? ROState.created : ROState.referenced;
		name = new InternationalStringController(this, InternationalStringImpl.Type.name, state);
		description = new InternationalStringController(this, InternationalStringImpl.Type.description, state);
	}

	/**
	 * Returns the registry objects state.
	 */
	public ROState getState() {
	    return state;
	}

	/**
     * Sets the registry objects state.
     */
    public void setState(ROState pState) {
        state = pState;
    }

    /**
     * Ensures that the objects state is {@link ROState#created}, or
     * {@link ROState#loaded}.
     */
    protected void ensureLoaded() throws JAXRException {
        switch (state) {
            case created:
            case loaded:
            case deleted:
                // Nothing to do
                break;
            case referenced:
                getROLoader().load(getRegistryService(), this);
                break;
            default:
                throw new IllegalStateException("Invalid state: " + state); //$NON-NLS-1$
        }
    }
        
	/**
	 * Returns the registry object loader.
	 */
    public abstract ROAccessor<RO> getROLoader();

	/**
	 * Loads the registry objects slots.
	 */
	@SuppressWarnings("unchecked")
    @Override
	protected Map<String,Slot> loadSlots() throws JAXRException {
		final ROAccessor<RO> roLoader = getROLoader();
		final RO ro = (RO) this;
		return roLoader.getSlots(getRegistryService(), ro);
	}

	@Override
	public Key getKey() throws JAXRException {
		return key;
	}

	@Override
	public void setKey(Key pKey) throws JAXRException {
		key = pKey;
	}

	@Override
	public void addAssociation(Association pAssociation) throws JAXRException {
	    ensureAssociationsLoaded();
	    pAssociation.setSourceObject(this);
	    ((AssociationImpl) pAssociation).setNum(associations.size());
	    associations.add(pAssociation);
	}

    @SuppressWarnings("rawtypes")
	@Override
	public void addAssociations(Collection pAssociations) throws JAXRException {
		for (Object o : pAssociations) {
			addAssociation((Association) o);
		}
	}

	private void ensureClassificationsLoaded() throws JAXRException {
        if (classifications == null) {
            classifications = new ArrayList<Classification>();
            switch (getState()) {
                case created:
                    // Nothing to do
                    break;
                case referenced:
                case loaded:
                    final Set<FilterPredicate> predicates = Collections.singleton((FilterPredicate) new OwnerPredicate(getKey(), false));
                	final Set<OrderPredicate> orderPredicates = Collections.singleton((OrderPredicate) ClassificationNumOrderPredicate.getInstance());
                    final RegistryObjectLoader rol = ClassificationAccessor.getInstance().getRegistryObjectLoader();
                    loadedClassificationKeys = new HashSet<Key>();
                    final Collection<RegistryObject> storedClassifications = rol.getResultList(getRegistryService(), predicates, orderPredicates);
                    for (RegistryObject ro : storedClassifications) {
                        loadedClassificationKeys.add(ro.getKey());
                    }
                    addClassifications(storedClassifications);
                    break;
                case deleted:
                    // Nothing to do
                    break;
            }
            if (loadedClassificationKeys == null) {
            	loadedClassificationKeys = new HashSet<Key>();
            }
        }
	}

	private void ensureAssociationsLoaded() throws JAXRException {
        if (associations == null) {
            associations = new ArrayList<Association>();
            switch (getState()) {
                case created:
                    // Nothing to do
                    break;
                case referenced:
                case loaded:
                    final Set<FilterPredicate> filterPredicates = Collections.singleton((FilterPredicate) new OwnerPredicate(getKey(), false));
                    final Set<OrderPredicate> orderPredicates = Collections.singleton((OrderPredicate) AssociationNumOrderPredicate.getInstance());
                    final RegistryObjectLoader rol = AssociationAccessor.getInstance().getRegistryObjectLoader();
                    loadedAssociationKeys = new HashSet<Key>();
                    final Collection<RegistryObject> storedAssociations = rol.getResultList(getRegistryService(), filterPredicates, orderPredicates);
                    for (RegistryObject ro : storedAssociations) {
                        loadedAssociationKeys.add(ro.getKey());
                    }
                    addAssociations(storedAssociations);
                    break;
                case deleted:
                    // Nothing to do
                    break;
            }
            if (loadedAssociationKeys == null) {
            	loadedAssociationKeys = new HashSet<Key>();
            }
        }
    }

	@Override
	public void addClassification(Classification pClassification) throws JAXRException {
	    ensureClassificationsLoaded();
	    pClassification.setClassifiedObject(this);
	    ((ClassificationImpl) pClassification).setNum(classifications.size());
		classifications.add(pClassification);
	}

	@SuppressWarnings("rawtypes")
	@Override
	public void addClassifications(Collection pClassifications) throws JAXRException {
		for (Object o : pClassifications) {
			addClassification((Classification) o);
		}
	}

	@Override
	public void addExternalIdentifier(ExternalIdentifier pExternalIdentifier) throws JAXRException {
		externalIdentifiers.add(pExternalIdentifier);
	}

    @SuppressWarnings("rawtypes")
	@Override
	public void addExternalIdentifiers(Collection pExternalIdentifiers) throws JAXRException {
		for (Object o : pExternalIdentifiers) {
			addExternalIdentifier((ExternalIdentifier) o);
		}
	}

	@Override
	public void addExternalLink(ExternalLink pExternalLink) throws JAXRException {
		externalLinks.add(pExternalLink);
	}

    @SuppressWarnings("rawtypes")
	@Override
	public void addExternalLinks(Collection pExternalLinks) throws JAXRException {
		for (Object o : pExternalLinks) {
			addExternalLink((ExternalLink) o);
		}
	}

    @SuppressWarnings("rawtypes")
	@Override
	public Collection getAssociatedObjects() throws JAXRException {
		final Set<Object> targets = new HashSet<Object>();
		for (Object o : getAssociations()) {
		    targets.add(((Association) o).getTargetObject());
		}
		return targets;
	}

    @SuppressWarnings("rawtypes")
	@Override
	public Collection getAssociations() throws JAXRException {
	    ensureAssociationsLoaded();
	    return associations;
	}

    @SuppressWarnings("rawtypes")
	@Override
	public Collection getAuditTrail() throws JAXRException {
	    throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

    @SuppressWarnings("rawtypes")
	@Override
	public Collection getClassifications() throws JAXRException {
        ensureClassificationsLoaded();
		return classifications;
	}

	@Override
	public InternationalString getDescription() throws JAXRException {
		return description.getString();
	}

    @SuppressWarnings("rawtypes")
	@Override
	public Collection getExternalIdentifiers() throws JAXRException {
		return externalIdentifiers;
	}

    @SuppressWarnings("rawtypes")
	@Override
	public Collection getExternalLinks() throws JAXRException {
		return externalLinks;
	}

	@Override
	public LifeCycleManager getLifeCycleManager() throws JAXRException {
		return lifeCycleManager;
	}

	@Override
	public InternationalString getName() throws JAXRException {
		return name.getString();
	}

    @SuppressWarnings("rawtypes")
	@Override
	public Collection getRegistryPackages() throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public Organization getSubmittingOrganization() throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public void removeAssociation(Association pAssociation) throws JAXRException {
	    ensureAssociationsLoaded();
	    associations.remove(pAssociation);
	}

    @SuppressWarnings("rawtypes")
	@Override
	public void removeAssociations(Collection pAssociations) throws JAXRException {
		for (Object o : pAssociations) {
			removeAssociation((Association) o);
		}
	}

	@Override
	public void removeClassification(Classification classification) throws JAXRException {
        ensureClassificationsLoaded();
		classifications.remove(classification);
	}

    @SuppressWarnings("rawtypes")
	@Override
	public void removeClassifications(Collection pClassifications) throws JAXRException {
		for (Object o : pClassifications) {
			removeClassification((Classification) o);
		}
	}

	@Override
	public void removeExternalIdentifier(ExternalIdentifier pExternalIdentifier) throws JAXRException {
		externalIdentifiers.remove(pExternalIdentifier);
	}

    @SuppressWarnings("rawtypes")
	@Override
	public void removeExternalIdentifiers(Collection pExternalIdentifiers) throws JAXRException {
		for (Object o : pExternalIdentifiers) {
			removeExternalIdentifier((ExternalIdentifier) o);
		}
	}

	@Override
	public void removeExternalLink(ExternalLink pExternalLink) throws JAXRException {
		externalLinks.remove(pExternalLink);
	}

    @SuppressWarnings("rawtypes")
	@Override
	public void removeExternalLinks(Collection pExternalLinks) throws JAXRException {
		for (Object o : pExternalLinks) {
			removeExternalLink((ExternalLink) o);
		}
	}

    @SuppressWarnings("rawtypes")
	@Override
	public void setAssociations(Collection pAssociations) throws JAXRException {
		ensureAssociationsLoaded();
		associations.clear();
		int i = 0;
		for (Object o : pAssociations) {
			final AssociationImpl assoc = (AssociationImpl) o;
			assoc.setNum(i++);
			associations.add(assoc);
		}
	}

    @SuppressWarnings("rawtypes")
	@Override
	public void setClassifications(Collection pClassifications) throws JAXRException {
		ensureClassificationsLoaded();
		classifications.clear();
		int i = 0;
		for (Object o : pClassifications) {
			final ClassificationImpl cl = (ClassificationImpl) o;
			cl.setNum(i++);
			classifications.add(cl);
		}
	}

	@Override
	public void setDescription(InternationalString pDescription) throws JAXRException {
		description.setString(pDescription);
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	@Override
	public void setExternalIdentifiers(Collection pExternalIdentifiers) throws JAXRException {
		if (pExternalIdentifiers == null) {
			externalIdentifiers.clear();
		} else {
			externalIdentifiers = pExternalIdentifiers;
		}
	}

	@SuppressWarnings({"rawtypes", "unchecked"})
	@Override
	public void setExternalLinks(Collection pExternalLinks) throws JAXRException {
		if (pExternalLinks == null) {
			externalLinks.clear();
		} else {
		    externalLinks = pExternalLinks;
		}
	}

	@Override
	public void setName(InternationalString pName) throws JAXRException {
		name.setString(pName);
	}

	@Override
	public String toXML() throws JAXRException {
        throw new JAXRException(NLSStrings.OPERATION_NOT_IMPLEMENTED);
	}

	@Override
	public Concept getObjectType() throws JAXRException {
	    ensureLoaded();
	    if (getKey().getId().equals(ROAccessors.ObjectTypes.CONCEPT.getId())) {
	        return (Concept) this;
	    }
	    final ROAccessors.ObjectTypes type = ROAccessors.ObjectTypes.values()[getROLoader().getRegistryObjectType()];
	    return ConceptAccessor.getInstance().create(getRegistryService(), new KeyImpl(type.getId()));
	}

	/**
	 * Returns the registry objects locale.
	 */
	protected Locale getLocale() {
		return ConnectionImpl.getLocale(getRegistryService().getConnection());
	}

	/**
	 * Returns the registry objects string bundle.
	 */
	protected NLSStrings getNLSStrings() {
		return ConnectionImpl.getNLSStrings(getRegistryService().getConnection());
	}

	/**
	 * Returns the name controller.
	 */
	public InternationalStringController getNameController() {
		return name;
	}

	/**
	 * Returns the description controller.
	 */
	public InternationalStringController getDescriptionController() {
		return description;
	}

	/**
	 * Sets the current set of classification keys.
	 */
    public void setLoadedClassificationKeys(Collection<Key> loadedClassificationKeys) {
        this.loadedClassificationKeys = loadedClassificationKeys;
    }

    /**
     * Returns the current set of classification keys.
     */
    public Collection<Key> getLoadedClassificationKeys() {
        return loadedClassificationKeys;
    }

    /**
     * Sets the current set of association keys.
     */
    public void setLoadedAssociationKeys(Collection<Key> loadedAssociationKeys) {
        this.loadedAssociationKeys = loadedAssociationKeys;
    }

    /**
     * Returns the current set of association keys.
     */
    public Collection<Key> getLoadedAssociationKeys() {
        return loadedAssociationKeys;
    }
}
