/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.infomodel;

import java.sql.SQLException;
import java.util.Locale;

import javax.xml.registry.BulkResponse;
import javax.xml.registry.BusinessLifeCycleManager;
import javax.xml.registry.BusinessQueryManager;
import javax.xml.registry.CapabilityProfile;
import javax.xml.registry.DeclarativeQueryManager;
import javax.xml.registry.InvalidRequestException;
import javax.xml.registry.JAXRException;
import javax.xml.registry.RegistryService;
import javax.xml.registry.UnsupportedCapabilityException;
import javax.xml.registry.infomodel.ClassificationScheme;

import org.apache.labs.jaxmas.registry.sql.DbDriver;


/**
 * Default implementation of {@link RegistryService}.
 */
public class RegistryServiceImpl implements RegistryService {
	private final ConnectionImpl connection;
	private final BusinessLifeCycleManager blm = new BusinessLifeCycleManagerImpl(this);
	private final BusinessQueryManager bqm = new BusinessQueryManagerImpl(this);
	private final DeclarativeQueryManager dqm = new DeclarativeQueryManagerImpl(this);
	private final DbDriver dbDriver;

	/**
	 * Creates a new instance with the given connection.
	 */
	public RegistryServiceImpl(ConnectionImpl pConnection) throws SQLException {
		connection = pConnection;
		final String databaseProductName = pConnection.getConnection().getMetaData().getDatabaseProductName();
		if ("Apache Derby".equals(databaseProductName)) { //$NON-NLS-1$
	        dbDriver = new org.apache.labs.jaxmas.registry.sql.DerbyDbDriver(this);
		} else if ("HSQL Database Engine".equals(databaseProductName)) { //$NON-NLS-1$
		    dbDriver = new org.apache.labs.jaxmas.registry.sql.HsqlDbDriver(this);
		} else {
		    throw new IllegalStateException("Invalid database product name: " + databaseProductName); //$NON-NLS-1$
		}
	}

	@Override
	public BulkResponse getBulkResponse(String pRequestId) throws JAXRException, InvalidRequestException {
		throw new IllegalStateException("Not implemented"); //$NON-NLS-1$
	}

	@Override
	public BusinessLifeCycleManager getBusinessLifeCycleManager() throws JAXRException {
		return blm;
	}

	@Override
	public BusinessQueryManager getBusinessQueryManager() throws JAXRException {
		return bqm;
	}

	@Override
	public CapabilityProfile getCapabilityProfile() throws JAXRException {
		return CapabilityProfileImpl.getInstance();
	}

	@Override
	public DeclarativeQueryManager getDeclarativeQueryManager() throws JAXRException, UnsupportedCapabilityException {
		return dqm;
	}

	@Override
	public ClassificationScheme getDefaultPostalScheme() throws JAXRException {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public String makeRegistrySpecificRequest(String pRequest) throws JAXRException {
		throw new IllegalStateException("Not implemented"); //$NON-NLS-1$
	}

	/**
	 * Returns the {@link Connection}.
	 */
	public ConnectionImpl getConnection() {
		return connection;
	}

	/**
	 * Returns the registry services {@link Locale}.
	 */
	public static Locale getLocale(RegistryService pRegistryService) {
		return ((RegistryServiceImpl) pRegistryService).getConnection().getLocale();
	}

	/**
	 * Returns the registry services {@link NLSStrings}.
	 */
	public static NLSStrings getNLSStrings(RegistryService pRegistryService) {
		return ((RegistryServiceImpl) pRegistryService).getConnection().getNLSStrings();
	}

	/**
	 * Returns the registry services {@link DbDriver}.
	 */
	public static DbDriver getDbDriver(RegistryService pRegistryService) {
		return ((RegistryServiceImpl) pRegistryService).getDbDriver();
	}

	/**
	 * Returns the {@link DbDriver}.
	 */
	public DbDriver getDbDriver() {
		return dbDriver;
	}
}
