/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.schema;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URL;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;

import javax.xml.registry.JAXRException;
import javax.xml.registry.RegistryService;

import org.apache.labs.jaxmas.registry.infomodel.ConnectionImpl;
import org.apache.labs.jaxmas.registry.infomodel.RegistryServiceImpl;
import org.apache.labs.jaxmas.registry.sql.DbDriver;
import org.apache.labs.jaxmas.registry.util.DefaultLoggerFactory;
import org.apache.labs.jaxmas.registry.util.Logger;
import org.apache.labs.jaxmas.registry.util.Strings;

/**
 * The database initializer is invoked at startup. It checks,
 * whether the database structure is as expected and updates
 * it, if required.
 */
public class DbInitializer {
	private static final Logger log = DefaultLoggerFactory.getInstance().newLogger(DbInitializer.class);

	private ConnectionImpl connection;

	/**
	 * Creates a new instance, which uses the given connection to
	 * initialize the database.
	 */
	public DbInitializer(ConnectionImpl pConnection) {
		connection = pConnection;
	}

	/**
	 * Returns the connection.
	 */
	protected ConnectionImpl getConnection() {
		return connection;
	}

	/**
	 * Returns the registry service.
	 */
	protected RegistryService getRegistryService() throws JAXRException {
		return connection.getRegistryService();
	}

	/**
	 * Returns the database driver.
	 */
	protected DbDriver getDbDriver() throws JAXRException {
		return ((RegistryServiceImpl) getRegistryService()).getDbDriver();
	}

	/**
	 * Returns, whether the character sequence contains a string, which
	 * ends with the character ';'. Terminating white space is ignored.
	 */
	private boolean endsWithSemicolon(CharSequence pCharSequence) {
		for (int i = pCharSequence.length()-1;  i >= 0;  i--) {
			char c = pCharSequence.charAt(i);
			if (!Character.isWhitespace(c)) {
				return c == ';';
			}
		}
		return false; // Only white space found
	}

	private String replaceProperties(String pCommand) throws JAXRException {
	    String command = pCommand;
	    for (;;) {
	        int offset = command.indexOf("${"); //$NON-NLS-1$
	        if (offset == -1) {
	            return command;
	        }
	        int end = command.indexOf('}', offset+2);
	        if (offset == -1) {
	            throw new IllegalStateException("Failed to parse command: " + command); //$NON-NLS-1$
	        }
	        final String property = command.substring(offset+2, end);
	        command = command.substring(0, offset) + getDbDriver().getDbProperty(property) + command.substring(end+1);
	    }
	}

	/**
	 * Runs a single SQL command.
	 */
	protected void sql(Connection pConnection, String pCommand) throws SQLException, JAXRException {
		final String mName = "sql"; //$NON-NLS-1$
		if (Strings.isTrimmedEmpty(pCommand)) {
			return;
		}
		final String command = replaceProperties(pCommand);
		final int offset = command.lastIndexOf(';');
		final String cmd = command.substring(0, offset);
		log.debug(mName, cmd);
		PreparedStatement stmt = null;
		try {
			stmt = pConnection.prepareStatement(cmd);
			stmt.executeUpdate();
			stmt.close();
			stmt = null;
		} catch (SQLException e) {
            // Ignore "Table does not exist" messages in DROP TABLE statements.
		    if (RegistryServiceImpl.getDbDriver(connection.getRegistryService()).isUnknownTableError(e)) {
		        return;
		    }
            throw e;
		} finally {
			if (stmt != null) { try { stmt.close(); } catch (Throwable t) { /* Ignore me */ } }
		}
	}

	/**
	 * Updates the database by applying a schema update.
	 */
	protected void initialize(Connection pConnection, BufferedReader pReader)
	        throws IOException, SQLException, JAXRException {
	    final StringBuilder sb = new StringBuilder();
		for (;;) {
			final String s = pReader.readLine();
			if (s == null) {
				sql(pConnection, sb.toString());
				return;
			}
			if (s.trim().startsWith("--")) { //$NON-NLS-1$
			    continue;
			}
			sb.append(s);
			sb.append('\n');
			if (endsWithSemicolon(sb)) {
				sql(pConnection, sb.toString());
				sb.setLength(0);
			}
		}
	}

	/**
	 * Updates the database by applying a schema update.
	 */
	protected void initialize(SchemaUpdater pSchemaUpdater, URL pSchemaFile) throws JAXRException {
		Connection conn = connection.getConnection();
		InputStream stream = null;
		Reader reader = null;
		BufferedReader bReader = null;
		try {
			conn.setAutoCommit(false);
			stream = pSchemaFile.openStream();
			reader = new InputStreamReader(stream, "UTF-8"); //$NON-NLS-1$
			bReader = new BufferedReader(MacroProcessor.newInstance(reader));
			if (pSchemaUpdater != null) {
				pSchemaUpdater.beforeUpdate(conn, connection.getRegistryService());
			}
			initialize(conn, bReader);
			if (pSchemaUpdater != null) {
				pSchemaUpdater.afterUpdate(conn, connection.getRegistryService());
			}
			bReader.close();
			bReader = null;
			reader.close();
			reader = null;
			stream.close();
			stream = null;
			conn.commit();
			conn = null;
		} catch (IOException e) {
			throw new JAXRException(e);
		} catch (SQLException e) {
			throw new JAXRException(e);
		} finally {
			if (bReader != null) { try { bReader.close(); } catch (Throwable t) { /* Ignore me */ } }
			if (reader != null) { try { reader.close(); } catch (Throwable t) { /* Ignore me */ } }
			if (stream != null) { try { stream.close(); } catch (Throwable t) { /* Ignore me */ } }
			if (conn != null) { try { conn.rollback(); } catch (Throwable t) { /* Ignore me */ } }
		}
	}

	private String getSchemaUpdaterClassName(int pNum) {
		final String className = getClass().getName();
		int offset = className.lastIndexOf('.');
		if (offset == -1) {
			throw new IllegalStateException("Failed to parse package name: " + className); //$NON-NLS-1$
		}
		return className.substring(0, offset) + ".SchemaUpdater" + pNum; //$NON-NLS-1$
	}

	private SchemaUpdater getSchemaUpdater(int pNum) throws JAXRException {
		final Class<?> cl;
		try {
			cl = Class.forName(getSchemaUpdaterClassName(pNum+1));
		} catch (ClassNotFoundException e) {
			return null;
		}
		try {
			return (SchemaUpdater) cl.newInstance();
		} catch (InstantiationException e) {
			throw new JAXRException(e);
		} catch (IllegalAccessException e) {
			throw new JAXRException(e);
		}
	}

	/**
	 * Initializes the database schema.
	 */
	public void initialize() throws JAXRException {
		int schemaVersion = getDbDriver().getSchemaVersion();
		for (int i = schemaVersion;  ;  i++) {
			final URL url = getClass().getResource("schema-update." + (i+1) + ".sql"); //$NON-NLS-1$ //$NON-NLS-2$
			if (url == null) {
				break;
			}
			initialize(getSchemaUpdater(i), url);
		}
	}
}
