/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.schema;

import java.io.FilterReader;
import java.io.IOException;
import java.io.PushbackReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * The macro processor is processing macros like &{macro, ...}
 * and allows to perform multiple or complex statements by specifying
 * a simple macro call.
 */
public class MacroProcessor extends FilterReader {
    private static class LineCountingReader extends FilterReader {
    	private int lineNum, colNum;

    	protected LineCountingReader(Reader pIn) {
			super(pIn);
		}

		@Override
		public int read(char[] pCbuf, int pOff, int pLen) throws IOException {
	        int num = 0;
	        for (int len = pLen;  len-- > 0;  ++num) {
	            int c = read();
	            if (c == -1) {
	                return num == 0 ? -1 : num;
	            }
	            pCbuf[pOff+num] = (char) c;
	        }
	        return num;
		}

		@Override
		public int read() throws IOException {
			int res = super.read();
			if (res == '\n') {
				lineNum++;
				colNum = 0;
			} else {
				colNum++;
			}
			return res;
		}

		public String format(String pMsg) {
			return "At line " + lineNum + ", column " + colNum + ": " + pMsg; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
    }

    private LineCountingReader lcr;
    private final Map<String,String[]> macros = new HashMap<String,String[]>();

    /**
     * Creates a new instance, which performs macro processing on the given
     * readers output.
     */
    public static MacroProcessor newInstance(Reader pReader) {
    	final LineCountingReader lcr = new LineCountingReader(pReader);
    	final MacroProcessor mp = new MacroProcessor(lcr);
    	mp.lcr = lcr;
    	return mp;
    }

    private MacroProcessor(Reader pReader) {
    	super(new PushbackReader(pReader, 8192));
    }

    private String processUnescapedParameter() throws IOException {
        final StringBuilder sb = new StringBuilder();
        for (;;) {
            int c = super.read();
            switch (c) {
                case -1:
                    throw new IllegalStateException(lcr.format("Unexpected EOF in input stream.")); //$NON-NLS-1$
                case '}':
                case ',':
                    ((PushbackReader) in).unread(c);
                    return sb.toString();
                case '\'':
                case '"':
                	throw new IllegalStateException(lcr.format("Invalid character in input stream.")); //$NON-NLS-1$
                case '\\':
                    int c1 = super.read();
                    if (c1 == -1) {
                        throw new IllegalStateException(lcr.format("Unexpected EOF in input stream.")); //$NON-NLS-1$
                    }
                    if (c1 >= '0'  &&  c1 <= '9') {
                        sb.append((char) c);
                    }
                    sb.append((char) c1);
                    break;
                default:
                    if (Character.isWhitespace(c)) {
                        return sb.toString();
                    }
                    sb.append((char) c);
                    break;
            }
        }
    }

    private String processEscapedParameter(int pTerminator) throws IOException {
        final StringBuilder sb = new StringBuilder();
        for (;;) {
            int c = super.read();
            switch (c) {
                case -1:
                    throw new IllegalStateException(lcr.format("Unexpected EOF in input stream.")); //$NON-NLS-1$
                case '\\':
                    int c1 = super.read();
                    if (c1 == -1) {
                        throw new IllegalStateException(lcr.format("Unexpected EOF in input stream.")); //$NON-NLS-1$
                    }
                    if (c1 >= '0'  &&  c1 <= '9') {
                        sb.append((char) c);
                    }
                    sb.append((char) c1);
                    break;
                default:
                    if (c == pTerminator) {
                        return ((char) pTerminator) + sb.toString() + ((char) pTerminator);
                    }
                    sb.append((char) c);
                    break;
            }
        }
    }

    private String processParameters(String pStatement, String[] pArgs) throws IOException {
        final StringBuilder sb = new StringBuilder();
        for (int i = 0;  i < pStatement.length();  i++) {
            int c = pStatement.charAt(i);
            if (c == '\\'  &&  i < pStatement.length()-1) {
                int c1 = pStatement.charAt(i+1);
                if (c1 >= '0'  &&  c1 <= '9') {
                    int num = c1 - '0';
                    if (num >= pArgs.length) {
                        throw new IllegalStateException(lcr.format("Invalid parameter reference \\" + num)); //$NON-NLS-1$
                    }
                    sb.append(pArgs[num]);
                    i++;
                } else {
                    ((PushbackReader) in).unread(c1);
                    sb.append((char) c);
                }
            } else {
                sb.append((char) c);
            }
        }
        return sb.toString();
    }

    private String unEscape(String pValue) {
        if (pValue.length() > 2) {
            int c = pValue.charAt(0);
            if ((c == '\''  ||  c == '\"')  &&  pValue.charAt(pValue.length()-1) == c) {
                return pValue.substring(1, pValue.length()-1);
            }
        }
        return pValue;
    }

    private void processMacro(List<String> pArgs) throws IOException {
        if (pArgs.size() == 0) {
            throw new IllegalStateException(lcr.format("Invalid macro call: Missing macro name")); //$NON-NLS-1$
        }
        final String macroName = pArgs.remove(0);
        if ("macro".equals(macroName)) { //$NON-NLS-1$
            if (pArgs.size() == 0) {
                throw new IllegalStateException(lcr.format("Invalid macro specification: Missing macro name")); //$NON-NLS-1$
            }
            final String newMacroName = pArgs.remove(0);
            if (macros.containsKey(newMacroName)) {
                throw new IllegalStateException(lcr.format("Macro redefined: " + newMacroName)); //$NON-NLS-1$
            }
            for (int i = 0;  i < pArgs.size();  i++) {
                pArgs.set(i, unEscape(pArgs.get(i)));
            }
            final String[] args = pArgs.toArray(new String[pArgs.size()]);
            macros.put(newMacroName, args);
        } else {
            final String[] statements = macros.get(macroName);
            if (statements == null) {
            	throw new IllegalStateException(lcr.format("Unknown macro: " + macroName)); //$NON-NLS-1$
            }
            final String[] args = pArgs.toArray(new String[pArgs.size()]);
            for (int i = statements.length-1;  i >= 0;  i--) {
                final String stmt = processParameters(statements[i], args);
                ((PushbackReader) in).unread((stmt + ";\n").toCharArray()); //$NON-NLS-1$
            }
        }
    }
    
    private void processMacro() throws IOException {
        final List<String> parameters = new ArrayList<String>();
        boolean commaExpected = false;
        for (;;) {
            int c = super.read();
            if (Character.isWhitespace(c)) {
                continue;
            }
            switch (c) {
                case '}':
                    processMacro(parameters);
                    return;
                case ',':
                    if (!commaExpected) {
                    	throw new IllegalStateException(lcr.format("Unexpected comma in input stream.")); //$NON-NLS-1$
                    }
                    commaExpected = false;
                    break;
                case '\'':
                case '"':
                    if (commaExpected) {
                        throw new IllegalStateException(lcr.format("Expected comma, got parameter in input stream.")); //$NON-NLS-1$
                    }
                    parameters.add(processEscapedParameter(c));
                    commaExpected = true;
                    break;
                default:
                    if (commaExpected) {
                    	throw new IllegalStateException(lcr.format("Expected comma, got parameter in input stream.")); //$NON-NLS-1$
                    }
                    ((PushbackReader) in).unread(c);
                    parameters.add(processUnescapedParameter());
                    commaExpected = true;
                    break;
            }
        }
    }

    
    @Override
    public int read(char[] pCbuf, int pOff, int pLen) throws IOException {
        int num = 0;
        for (int len = pLen;  len-- > 0;  ++num) {
            int c = read();
            if (c == -1) {
                return num == 0 ? -1 : num;
            }
            pCbuf[pOff+num] = (char) c;
        }
        return num;
    }

    @Override
    public int read() throws IOException {
        int res = super.read();
        if (res == '&') {
            int c = super.read();
            if (c == '{') {
                processMacro();
                return read();
            }
            ((PushbackReader) in).unread(c);
        }
        return res;
    }

    
}
