/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.schema;

import java.io.IOException;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import javax.xml.registry.BusinessLifeCycleManager;
import javax.xml.registry.JAXRException;
import javax.xml.registry.RegistryService;
import javax.xml.registry.infomodel.ClassificationScheme;
import javax.xml.registry.infomodel.InternationalString;


/**
 * This schema updater creates the default classification schemes
 * and association types.
 */
public class SchemaUpdater1 implements SchemaUpdater {
	@Override
	public void beforeUpdate(Connection pConnection, RegistryService pRegistryService) throws JAXRException, SQLException, IOException {
		// Nothing to do, the schema is still to be created.
	}

	@Override
	public void afterUpdate(Connection pConnection, RegistryService pRegistryService) throws JAXRException, SQLException, IOException {
		final List<ClassificationScheme> clSchemes = new ArrayList<ClassificationScheme>();
		pRegistryService.getBusinessLifeCycleManager().saveClassificationSchemes(clSchemes);
	}

	/**
	 * Creates an international string with the given value for the default locale and the
	 * US locale.
	 */
	protected InternationalString createInternationalString(RegistryService pRegistryService,
			String pValue) throws JAXRException {
		final BusinessLifeCycleManager blcm = pRegistryService.getBusinessLifeCycleManager();
		final InternationalString is = blcm.createInternationalString(pValue);
		is.addLocalizedString(blcm.createLocalizedString(Locale.ENGLISH, pValue));
		if (!Locale.getDefault().equals(Locale.ENGLISH)) {
			is.addLocalizedString(blcm.createLocalizedString(Locale.getDefault(), pValue));
		}
		return is;
	}

	/**
	 * Creates a new classification scheme with the given name and description.
	 */
	protected ClassificationScheme createClassificationScheme(RegistryService pRegistryService,
			String pName, String pDescription) throws JAXRException {
		final InternationalString name = createInternationalString(pRegistryService, pName);
		final InternationalString descr = createInternationalString(pRegistryService, pDescription);
		return pRegistryService.getBusinessLifeCycleManager().createClassificationScheme(name, descr);
	}
}
