/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.sql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import javax.xml.registry.JAXRException;
import javax.xml.registry.RegistryService;
import javax.xml.registry.infomodel.Association;
import javax.xml.registry.infomodel.Classification;
import javax.xml.registry.infomodel.Concept;
import javax.xml.registry.infomodel.InternationalString;
import javax.xml.registry.infomodel.Key;
import javax.xml.registry.infomodel.LocalizedString;
import javax.xml.registry.infomodel.RegistryEntry;
import javax.xml.registry.infomodel.RegistryObject;
import javax.xml.registry.infomodel.Slot;

import org.apache.labs.jaxmas.registry.accessor.AssociationAccessor;
import org.apache.labs.jaxmas.registry.accessor.ClassificationAccessor;
import org.apache.labs.jaxmas.registry.infomodel.AssociationImpl;
import org.apache.labs.jaxmas.registry.infomodel.ClassificationImpl;
import org.apache.labs.jaxmas.registry.infomodel.ConceptImpl;
import org.apache.labs.jaxmas.registry.infomodel.InternationalStringController;
import org.apache.labs.jaxmas.registry.infomodel.InternationalStringImpl;
import org.apache.labs.jaxmas.registry.infomodel.LocalizedStringImpl;
import org.apache.labs.jaxmas.registry.infomodel.NLSStrings;
import org.apache.labs.jaxmas.registry.infomodel.OwnedRegistryObject;
import org.apache.labs.jaxmas.registry.infomodel.ROState;
import org.apache.labs.jaxmas.registry.infomodel.RegistryEntryImpl;
import org.apache.labs.jaxmas.registry.infomodel.RegistryObjectImpl;
import org.apache.labs.jaxmas.registry.infomodel.RegistryServiceImpl;
import org.apache.labs.jaxmas.registry.infomodel.SlotImpl;
import org.apache.labs.jaxmas.registry.util.Locales;
import org.apache.log4j.Logger;


/**
 * Default implementation of {@link DbDriver}.
 */
public abstract class AbstractDbDriver implements DbDriver {
    private static final Logger log = Logger.getLogger(AbstractDbDriver.class);
    private final RegistryService registryService;

	/**
	 * Creates a new instance with the given registry service.
	 */
	protected AbstractDbDriver(RegistryService pRegistryService) {
		registryService = pRegistryService;
	}

	/**
	 * Returns the drivers registry service.
	 */
	protected RegistryService getRegistryService() {
		return registryService;
	}

	/**
	 * Called to perform the given prepared statement with the given parameters.
	 */
	protected void run(String pStmt, Object... pParams) throws JAXRException {
		Sql.run(getRegistryService(), pStmt, pParams);
	}

	/**
	 * Called to run the given {@link ConnUser}.
	 */
	protected void run(ConnUser<?> pConnUser) throws JAXRException {
		pConnUser.run(getRegistryService());
	}

	@Override
	public long newId(String pName) throws JAXRException {
		run("UPDATE ids SET nextValue = nextValue + 1 WHERE name = ?", pName); //$NON-NLS-1$
		final QueryUser<Long> query = new QueryUser<Long>("SELECT nextValue FROM ids WHERE name = ?", pName){ //$NON-NLS-1$
			@Override
			protected void action(ResultSet pResultSet) throws JAXRException, SQLException {
				boolean hasNext = pResultSet.next();
				assert(hasNext);
				final long l = pResultSet.getLong(1);
				assert(!pResultSet.wasNull());
				hasNext = pResultSet.next();
				assert(!hasNext);
				setResult(new Long(l));
				if (pResultSet.next()) {
					throw new JAXRException("Unexpected result row"); //$NON-NLS-1$
				}
			}
			@Override
			protected void action(Connection pConnection) throws JAXRException, SQLException {
				super.action(pConnection);
			}
		};
		query.run(getRegistryService());
		return query.getResult().longValue();
	}

	@Override
	public void deleteSlots(RegistryObject pRegistryObject) throws JAXRException {
	    // No need to delete anything, because the slots are deleted by ON DELETE CASCASE.
	}

	private void saveSlots(final RegistryObjectImpl<?> pRegistryObject) throws JAXRException {
		if (!pRegistryObject.hasSlotsLoaded()) {
			return;
		}
		new ObjStmtUser("DELETE FROM RegistryObjectSlots WHERE roKey=?", pRegistryObject.getKey()){ //$NON-NLS-1$
            @Override
            protected void action(PreparedStatement pStatement) throws JAXRException, SQLException {
                pStatement.executeUpdate();
                final Collection<?> slots = pRegistryObject.getSlots();
                if (slots != null  &&  slots.size() > 0) {
                    final String q = "INSERT INTO RegistryObjectSlots (roKey, id, name, slotType) VALUES (?, ?, ?, ?)"; //$NON-NLS-1$
                    final String q2 = "INSERT INTO RegistryObjectSlotValues (id, val) VALUES (?, ?)"; //$NON-NLS-1$
                    PreparedStatement stmt = pStatement.getConnection().prepareStatement(q);
                    PreparedStatement stmt2 = null;
                    try {
                        stmt.setString(1, pRegistryObject.getKey().getId());
                        for (Object o : slots) {
                            final Slot slot = (Slot) o;
                            final long id = newId("Slots"); //$NON-NLS-1$
                            stmt.setLong(2, id);
                            stmt.setString(3, slot.getName());
                            stmt.setString(4, slot.getSlotType());
                            log.debug(q + ", " + id + ", " + slot.getName() + ", " + slot.getSlotType());  //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
                            stmt.executeUpdate();
                            for (Object v : slot.getValues()) {
                                if (stmt2 == null) {
                                    stmt2 = pStatement.getConnection().prepareStatement(q2);
                                }
                                stmt2.setLong(1, id);
                                stmt2.setString(2, v == null ? null : v.toString());
                                log.debug(q2 + ", " + id + ", " + v);  //$NON-NLS-1$//$NON-NLS-2$
                                stmt2.executeUpdate();
                            }
                        }
                        if (stmt2 != null) {
                            stmt2.close();
                            stmt2 = null;
                        }
                        stmt.close();
                        stmt = null;
                    } finally {
                        if (stmt2 != null) { try { stmt2.close(); } catch (Throwable t) { /* Ignore me */ } }
                        if (stmt != null) { try { stmt.close(); } catch (Throwable t) { /* Ignore me */ } }
                    }
                }
            }
	    }.run(getRegistryService());
	}
	
	@SuppressWarnings("unchecked")
    private static Collection<String> asStringCollection(Collection<?> pCollection) {
	    return (Collection<String>) pCollection;
	}

	@Override
	public Map<String, Slot> getSlots(RegistryObject pRegistryObject) throws JAXRException {
	    final Map<Long,Slot> slots = new HashMap<Long,Slot>();

	    final String q1 = "SELECT id, name, slotType FROM RegistryObjectSlots WHERE roKey=?"; //$NON-NLS-1$
		new ObjQueryUser(q1, pRegistryObject.getKey()){
			@Override
			protected void action(ResultSet pResultSet) throws JAXRException, SQLException {
			    while (pResultSet.next()) {
			        final Long id = new Long(pResultSet.getLong(1));
			        assert(!pResultSet.wasNull());
			        Slot slot = slots.get(id);
			        if (slot == null) {
			            slot = new SlotImpl();
			            slots.put(id, slot); 
			            slot.setName(pResultSet.getString(2));
			            assert(slot.getName() != null);
			            slot.setSlotType(pResultSet.getString(3));
			            slot.setValues(new ArrayList<String>());
			        }
			    }
			}
		}.run(getRegistryService());

		final String q2 = "SELECT ros.id, rosv.val FROM RegistryObjectSlots ros JOIN RegistryObjectSlotValues rosv ON ros.id=rosv.id WHERE ros.roKey=?"; //$NON-NLS-1$
		new ObjQueryUser(q2, pRegistryObject.getKey()) {
            @Override
            protected void action(ResultSet pResultSet) throws JAXRException, SQLException {
                while (pResultSet.next()) {
                    final Long id = new Long(pResultSet.getLong(1));
                    assert(!pResultSet.wasNull());
                    final Slot slot = slots.get(id);
                    assert(slot != null);
                    asStringCollection(slot.getValues()).add(pResultSet.getString(2));
                }
            }
		}.run(getRegistryService());

		final Map<String,Slot> result = new HashMap<String,Slot>();
		for (Slot slot : slots.values()) {
		    result.put(slot.getName(), slot);
		}
		return result;
	}

	@Override
    public void deleteInternationalString(Key pKey, InternationalStringImpl.Type pType) throws JAXRException {
		run("DELETE FROM LocalizedStrings WHERE roKey=? AND lsType=?", pKey, pType); //$NON-NLS-1$
	}

	@Override
    public void updateInternationalString(Key pKey, InternationalStringImpl.Type pType, final InternationalString pValue) throws JAXRException {
	    deleteInternationalString(pKey, pType);
        insertInternationalString(pKey, pType, pValue);
	}

	@Override
    public void insertInternationalString(final Key pKey, final InternationalStringImpl.Type pType,
	        final InternationalString pValue) throws JAXRException {
	    if (pValue != null) {
	        new ObjStmtUser("INSERT INTO LocalizedStrings (roKey, lsType, locale, charset, val) VALUES (?, ?, ?, ?, ?)"){ //$NON-NLS-1$
	            @Override
	            protected void action(PreparedStatement pStatement) throws JAXRException, SQLException {
	                pStatement.setString(1, pKey.getId());
	                pStatement.setInt(2, pType.ordinal());
	                for (Object o : pValue.getLocalizedStrings()) {
	                    LocalizedString ls = (LocalizedString) o;
	                    pStatement.setString(3, ls.getLocale().toString());
	                    pStatement.setString(4, ls.getCharsetName());
	                    pStatement.setString(5, ls.getValue());
	                    pStatement.executeUpdate();
	                }
                }
	        }.run(getRegistryService());
	    }
    }

	@SuppressWarnings("unchecked")
    private static final Collection<Classification> asClassificationCollection(Collection<?> pCollection) {
	    return (Collection<Classification>) pCollection;
	}

	@SuppressWarnings("unchecked")
	private static final Collection<Association> asAssociationCollection(Collection<?> pCollection) {
	    return (Collection<Association>) pCollection;
    }

	@Override
	public void insert(RegistryObject pObject, int pRegistryObjectType)
			throws JAXRException {
		final RegistryObjectImpl<?> ro = (RegistryObjectImpl<?>) pObject;
		assert(ro.getState() == ROState.created);
		final Key key = pObject.getKey();
		final Key ownerRestricting, ownerCascading;
		final Integer pos;
		if (ro instanceof OwnedRegistryObject<?>) {
		    final OwnedRegistryObject<?> oro = (OwnedRegistryObject<?>) ro;
		    final RegistryObject roOwner = oro.getOwner();
		    final Key owner = roOwner == null ? null : roOwner.getKey();
		    if (oro.isRestricting()) {
		    	ownerCascading = null;
		    	ownerRestricting = owner;
		    } else {
		    	ownerCascading = owner;
		    	ownerRestricting = null;
		    }
		    pos = Integer.valueOf(oro.getPosition());
		} else {
		    ownerRestricting = ownerCascading = null;
		    pos = null;
		}
		final String s = "INSERT INTO RegistryObjects (roKey, roType, pos, roOwnerRestricting, roOwnerCascading) VALUES (?, ?, ?, ?, ?)"; //$NON-NLS-1$
		run(s, key, Integer.valueOf(pRegistryObjectType), pos, ownerRestricting, ownerCascading);

		final InternationalStringController name = ro.getNameController();
        name.save();
        final InternationalStringController description = ro.getDescriptionController();
        description.save();

        saveSlots(ro);

        if (pObject instanceof RegistryEntry) {
			RegistryEntryImpl<?> re = (RegistryEntryImpl<?>) pObject;
			run("INSERT INTO RegistryEntries (roKey, customType, expiration, status, stability, majorVersion, minorVersion, userVersion) VALUES (?, ?, ?, ?, ?, ?, ?, ?)", //$NON-NLS-1$
					key, re.getCustomType(), re.getExpiration(), new Integer(re.getStatus()),
					new Integer(re.getStability()), new Integer(re.getMajorVersion()),
					new Integer(re.getMinorVersion()), re.getUserVersion());
		}

		if (pObject instanceof Concept) {
			final ConceptImpl concept = (ConceptImpl) pObject;
			final String value = concept.getValue();
			if (value == null) {
				throw new JAXRException(RegistryServiceImpl.getNLSStrings(getRegistryService()).format(NLSStrings.CONCEPT_VALUE_MISSING));
			}
			run("INSERT INTO Concepts (roKey, value) VALUES (?, ?)", //$NON-NLS-1$
			        concept.getKey(), value);
		}

		if (pObject instanceof Classification) {
		    final ClassificationImpl cl = (ClassificationImpl) pObject;
            final Concept concept = cl.getConcept();
            assert(concept != null);
            run("INSERT INTO Classifications (roKey, roKeyConcept, num) VALUES (?, ?, ?)", //$NON-NLS-1$
                    cl.getKey(), concept.getKey(), Integer.valueOf(cl.getNum()));
		}

		if (pObject instanceof Association) {
		    final AssociationImpl assoc = (AssociationImpl) pObject;
		    final Concept assocType = assoc.getAssociationType();
		    assert(assocType != null);
		    final RegistryObject target = assoc.getTargetObject();
		    assert(target != null);
		    run("INSERT INTO Associations (roKey, roKeyType, roKeyTarget, num) VALUES (?, ?, ?, ?)", //$NON-NLS-1$
		            assoc.getKey(), assocType.getKey(), target.getKey(), Integer.valueOf(assoc.getNum()));
		}
		
		int num = 0;
		for (Classification cl : asClassificationCollection(ro.getClassifications())) {
			final ClassificationImpl clImpl =  (ClassificationImpl) cl;
			clImpl.setNum(num++);
			clImpl.getROLoader().save(getRegistryService(), cl);
		}

		num = 0;
		for (Association assoc : asAssociationCollection(ro.getAssociations())) {
			final AssociationImpl assocImpl = (AssociationImpl) assoc;
			assocImpl.setNum(num++);
			assocImpl.getROLoader().save(getRegistryService(), assoc);
        }
	}

	@Override
	public void update(RegistryObject pObject) throws JAXRException {
		final RegistryObjectImpl<?> ro = (RegistryObjectImpl<?>) pObject;
		final String key = pObject.getKey().getId();

		final InternationalStringController name = ro.getNameController();
		name.save();
		final InternationalStringController description = ro.getDescriptionController();
		description.save();

		this.saveSlots(ro);
		
		if (pObject instanceof RegistryEntry) {
			RegistryEntryImpl<?> re = (RegistryEntryImpl<?>) pObject;
			run("UPDATE RegistryEntries SET customType=?, expiration=?, status=?, stability=?, majorVersion=?, minorVersion=?, userVersion=? WHERE roKey=?", //$NON-NLS-1$
					re.getCustomType(), re.getExpiration(), new Integer(re.getStatus()),
					new Integer(re.getStability()), new Integer(re.getMajorVersion()),
					new Integer(re.getMinorVersion()), re.getUserVersion(), key);
		}

		if (pObject instanceof Concept) {
		    final ConceptImpl concept = (ConceptImpl) pObject;
		    switch (concept.getState()) {
		        case created:
		            throw new IllegalStateException("This object must be inserted, not updated."); //$NON-NLS-1$
		        case loaded:
	                final String value = concept.getValue();
	                if (value == null) {
	                    throw new JAXRException(RegistryServiceImpl.getNLSStrings(getRegistryService()).format(NLSStrings.CONCEPT_VALUE_MISSING));
	                }
	                run("UPDATE Concepts SET value=? WHERE roKey=?", //$NON-NLS-1$
	                        value, concept.getKey());
	                break;
		        case referenced:
		            // Nothing to do
		            break;
		        case deleted:
		            throw new IllegalStateException("A deleted object cannot be updated."); //$NON-NLS-1$
		    }
		}

		if (pObject instanceof Classification) {
		    final ClassificationImpl cl = (ClassificationImpl) pObject;
            final Concept concept = cl.getConcept();
            assert(concept != null);
            run("UPDATE Classifications SET roKeyConcept=?, num=? WHERE roKey=?", //$NON-NLS-1$
                    concept.getKey(), Integer.valueOf(cl.getNum()), cl.getKey());
        }

		if (pObject instanceof Association) {
		    final AssociationImpl assoc = (AssociationImpl) pObject;
		    final Concept assocType = assoc.getAssociationType();
		    assert(assocType != null);
		    final RegistryObject target = assoc.getTargetObject();
		    assert(target != null);
		    run("UPDATE Associations SET roKeyType=?, roKeyTarget=?, num=? WHERE roKey=?", //$NON-NLS-1$
                    assocType.getKey(), target.getKey(), Integer.valueOf(assoc.getNum()), assoc.getKey());
        }

		final Collection<Key> loadedClassificationKeys = ro.getLoadedClassificationKeys();
		if (loadedClassificationKeys != null) {
		    final Set<Key> currentClassificationKeys = new HashSet<Key>();
	        for (Classification cl : asClassificationCollection(ro.getClassifications())) {
	            ClassificationAccessor.getInstance().save(getRegistryService(), cl);
	            currentClassificationKeys.add(cl.getKey());
	            loadedClassificationKeys.remove(cl.getKey());
	        }
	        ro.setLoadedClassificationKeys(currentClassificationKeys);
	        for (Key k : loadedClassificationKeys) {
	            deleteRegistryObject(k);
	        }
		}

		final Collection<Key> loadedAssociationKeys = ro.getLoadedAssociationKeys();
		if (loadedAssociationKeys != null) {
		    final Set<Key> currentAssociationKeys = new HashSet<Key>();
		    for (Association assoc : asAssociationCollection(ro.getAssociations())) {
		        AssociationAccessor.getInstance().save(getRegistryService(), assoc);
		        currentAssociationKeys.add(assoc.getKey());
		        loadedAssociationKeys.remove(assoc.getKey());
            }
		    ro.setLoadedAssociationKeys(currentAssociationKeys);
		    for (Key k : loadedAssociationKeys) {
		        deleteRegistryObject(k);
            }
        }
	} 

	@Override
	public InternationalString loadInternationalString(Key pKey, InternationalStringImpl.Type pType) throws JAXRException {
		final InternationalString is = new InternationalStringImpl();
		new ObjQueryUser("SELECT locale, charset, val FROM LocalizedStrings WHERE roKey=? AND lsType=?", pKey, pType) { //$NON-NLS-1$
			@Override
			protected void action(ResultSet pResultSet) throws JAXRException, SQLException {
				while (pResultSet.next()) {
					final Locale locale = Locales.getLocale(pResultSet.getString(1));
					final String charset = pResultSet.getString(2);
					final String val = pResultSet.getString(3);
					final LocalizedStringImpl ls = new LocalizedStringImpl();
					ls.setLocale(locale);
					ls.setCharsetName(charset);
					ls.setValue(val);
					is.addLocalizedString(ls);
				}
			}
		}.run(getRegistryService());
		return is;
	}

    @Override
    public void deleteRegistryObject(Key pKey) throws JAXRException {
        Sql.run(getRegistryService(), "DELETE FROM RegistryObjects WHERE roKey=?", pKey); //$NON-NLS-1$
    }

    @Override
    public int getSchemaVersion() throws JAXRException {
        final String s = "SELECT version FROM DbInfo"; //$NON-NLS-1$
        try {
            return Sql.intQuery(getRegistryService(), s);
        } catch (JAXRException e) {
            Throwable t = e.getCause();
            if (t != null  &&  t instanceof SQLException) {
                if (isUnknownTableError((SQLException) t)) {
                    return 0;
                }
            }
            throw e;
        }
    }
}
