/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.sql;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;

import javax.xml.registry.JAXRException;
import javax.xml.registry.RegistryService;

import org.apache.labs.jaxmas.registry.infomodel.RegistryServiceImpl;
import org.apache.labs.jaxmas.registry.util.DefaultLoggerFactory;
import org.apache.labs.jaxmas.registry.util.Logger;

/**
 * Abstract base class of an object, which performs an operation
 * by using a database connection.
 */
public abstract class ConnUser<O extends Object> {
	private static final Logger log = DefaultLoggerFactory.getInstance().newLogger(ConnUser.class);

	private O result;

	/**
	 * Sets the result object.
	 */
	protected void setResult(O pResult) {
		result = pResult;
	}

	/**
	 * Returns the result object.
	 */
	public O getResult() {
		return result;
	}

	/**
	 * This method must be implemented to perform the actual operation.
	 * It is called from within {@link #run(Connection)}.
	 */
	protected abstract void action(Connection pConnection) throws JAXRException, SQLException;
	
	/**
	 * Called to perform the database connection.
	 */
	public void run(RegistryService pRegistryService) throws JAXRException {
		run(((RegistryServiceImpl) pRegistryService).getConnection().getConnection());
	}

	/**
	 * Logs, that the connection user is entering the method with the given name.
	 */
	protected void logEntering(Logger pLog, String pName) {
		pLog.entering(pName);
	}

    /**
     * Logs, that the connection user is leaving the method with the given name.
     */
	protected void logExiting(Logger pLog, String pName) {
		pLog.exiting(pName);
	}

	/**
     * Logs, that the connection user is leaving the method with the given name.
     * The given result is returned.
	 */
	protected void logExiting(Logger pLog, String pName, String pMsg) {
	    pLog.exiting(pName, pMsg);
	}

	/**
	 * Returns the result with the given index as a Long.
	 */
	protected Long getLong(ResultSet pResultSet, int pIndex) throws SQLException {
		long l = pResultSet.getLong(pIndex);
		return pResultSet.wasNull() ? null : new Long(l);
	}
	
	/**
	 * Called to perform the database connection.
	 */
	public void run(Connection pConnection) throws JAXRException {
		final String mName = "run"; //$NON-NLS-1$
		logEntering(log, mName);
		try {
			action(pConnection);
		} catch (SQLException e) {
			throw new JAXRException(e);
		}
		final Object o = getResult();
		if (o == null) {
		    logExiting(log, mName);
		} else {
		    logExiting(log, mName, String.valueOf(o));
		}
	}
}
