/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.sql;

import java.sql.SQLException;
import java.util.Collection;
import java.util.List;

import javax.xml.registry.JAXRException;

import org.apache.labs.jaxmas.registry.accessor.ROAccessors.ClassificationSchemes;
import org.apache.labs.jaxmas.registry.accessor.ROAccessors.ObjectTypes;


/**
 * Predicate for restricting a search to a particular registry object type.
 */
public class CustomObjectTypePredicate extends RegistryObjectTypePredicate {
    private final String prefix;
    private final Collection<String> types;

    /**
     * Creates a new instance, which restricts the search to the given
     * type.
     */
    public CustomObjectTypePredicate(Collection<String> pTypes, String pPrefix) {
        super(ObjectTypes.REGISTRY_ENTRY, pPrefix);
    	if (pTypes == null  ||  pTypes.isEmpty()) {
    		throw new IllegalArgumentException("The list of custom object types must not be null."); //$NON-NLS-1$
    	}
    	types = pTypes;
        prefix = pPrefix;
    }

    /**
     * Creates a new instance, which restricts the search to the given
     * type.
     */
    public CustomObjectTypePredicate(Collection<String> pTypes) {
        this(pTypes, null);
    }

    /**
     * Returns the collection of custom object types.
     */
    public Collection<String> getTypes() {
        return types;
    }

    @Override
    public void add(StringBuilder pBuffer, List<Object> pParameters) throws SQLException, JAXRException {
        super.add(pBuffer, pParameters);
        final String reTable = RegistryObjectLoader.getTable(prefix, "re");
        pBuffer.append(" AND ");
    	final Collection<String> customTypes = getTypes();
    	pBuffer.append(reTable);
    	pBuffer.append(".customType IN (SELECT ctro.roKey FROM RegistryObjects ctro JOIN Concepts ctco ON ctro.roType=? AND ctro.roKey=ctco.roKey WHERE ctro.roOwnerRestricting=? AND ");
    	pParameters.add(new Integer(ObjectTypes.CONCEPT.ordinal()));
    	pParameters.add(ClassificationSchemes.OBJECT_TYPE.getId());
    	if (customTypes.size() == 1) {
    		pBuffer.append("ctco.value=?"); //$NON-NLS-1$
    		pParameters.add(customTypes.iterator().next());
    	} else {
    		pBuffer.append("ctco.value IN ("); //$NON-NLS-1$
    		boolean first = true;
    		for (String s : customTypes) {
    			if (first) {
    				first = false;
    			} else {
    				pBuffer.append(',');
    			}
    			pBuffer.append('?');
    			pParameters.add(s);
    		}
    		pBuffer.append(')');
    	}
    	pBuffer.append(')');
    }
}
