/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.sql;

import java.sql.SQLException;
import java.util.Map;

import javax.xml.registry.JAXRException;
import javax.xml.registry.infomodel.InternationalString;
import javax.xml.registry.infomodel.Key;
import javax.xml.registry.infomodel.RegistryObject;
import javax.xml.registry.infomodel.Slot;

import org.apache.labs.jaxmas.registry.infomodel.InternationalStringImpl;


/**
 * Database drivers are used to hide database specific details.
 * The intention is to support different databases by using different
 * drivers.
 */
public interface DbDriver {
	/**
	 * Creates a new id.
	 */
	long newId(String pName) throws JAXRException;

	/**
	 * Reads a registry objects slots.
	 */
	Map<String, Slot> getSlots(RegistryObject pRegistryObject) throws JAXRException;

	/**
	 * Returns the current schema version.
	 */
	int getSchemaVersion() throws JAXRException;

	/**
	 * Inserts the given object into the registry.
	 */
	void insert(RegistryObject pObject, int pRegistryObjectType) throws JAXRException;

	/**
	 * Updates the given object in the registry.
	 */
	void update(RegistryObject pObject) throws JAXRException;

	/**
	 * Loads the given {@link InternationalString} from the registry.
	 */
	InternationalString loadInternationalString(Key pKey, InternationalStringImpl.Type pType) throws JAXRException;

	/**
	 * Inserts the given {@link InternationalString} into the registry.
	 */
	void insertInternationalString(Key pKey, InternationalStringImpl.Type pType, InternationalString pString) throws JAXRException;

	/**
	 * Deletes the given {@link InternationalString} from the registry.
	 */
	void deleteInternationalString(Key pKey, InternationalStringImpl.Type pType) throws JAXRException;

	/**
	 * Updates the given {@link InternationalString} in the registry.
	 */
	void updateInternationalString(Key pKey, InternationalStringImpl.Type pType, InternationalString pString) throws JAXRException;

	/**
	 * Removes the given {@link RegistryObject} from the registry.
	 */
    void deleteRegistryObject(Key pKey) throws JAXRException;

    /**
     * Removes the given registry objects {@link Slot slots} from the registry.
     */
    void deleteSlots(RegistryObject pRegistryObject) throws JAXRException;

    /**
     * Called to shutdown the registry. This is mainly useful for unit tests.
     */
    void shutdown() throws JAXRException;

    /**
     * Returns, whether a DROP TABLE command failed, because the table does not exist.
     */
    boolean isUnknownTableError(SQLException pE);

    /**
     * Returns a database property, for example the database specific CLOB type.
     */
    String getDbProperty(String pProperty);
}
