/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.sql;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.xml.registry.JAXRException;
import javax.xml.registry.RegistryService;
import javax.xml.registry.infomodel.Association;
import javax.xml.registry.infomodel.Classification;
import javax.xml.registry.infomodel.Concept;
import javax.xml.registry.infomodel.RegistryEntry;
import javax.xml.registry.infomodel.RegistryObject;

import org.apache.labs.jaxmas.registry.accessor.ROAccessor;
import org.apache.labs.jaxmas.registry.accessor.ROAccessors;
import org.apache.labs.jaxmas.registry.accessor.ROAccessors.ObjectTypes;
import org.apache.labs.jaxmas.registry.infomodel.AssociationImpl;
import org.apache.labs.jaxmas.registry.infomodel.ClassificationImpl;
import org.apache.labs.jaxmas.registry.infomodel.KeyImpl;
import org.apache.labs.jaxmas.registry.infomodel.OwnedRegistryObject;
import org.apache.labs.jaxmas.registry.infomodel.ROState;
import org.apache.labs.jaxmas.registry.infomodel.RegistryEntryImpl;
import org.apache.labs.jaxmas.registry.infomodel.RegistryObjectImpl;


/**
 * An instance of this class is used for loading registry
 * objects.
 */
public class RegistryObjectLoader {
    private static final TableHandler[] EXTENSION_HANDLERS_REGISTRY_OBJECT = new TableHandler[]{
        new DefaultTableHandler()
    };
    private static final TableHandler[] EXTENSION_HANDLERS_ASSOCIATION = new TableHandler[]{
        new DefaultTableHandler(),
        new AssociationTableHandler(ROAccessors.ObjectTypes.ASSOCIATION)
    };
    private static final TableHandler[] EXTENSION_HANDLERS_CLASSIFICATION = new TableHandler[]{
        new DefaultTableHandler(),
        new ClassificationTableHandler(ROAccessors.ObjectTypes.CLASSIFICATION)
    };
    private static final TableHandler[] EXTENSION_HANDLERS_EXTRINSIC_OBJECT = new TableHandler[]{
        new DefaultTableHandler(),
        new RegistryEntryTableHandler(ROAccessors.ObjectTypes.EXTRINSIC_OBJECT)
    };
    private static final TableHandler[] EXTENSION_HANDLERS_SERVICE = new TableHandler[]{
        new DefaultTableHandler(),
        new RegistryEntryTableHandler(ROAccessors.ObjectTypes.SERVICE)
    };
    private static final TableHandler[] EXTENSION_HANDLERS_REGISTRY_PACKAGE = new TableHandler[]{
        new DefaultTableHandler(),
        new RegistryEntryTableHandler(ROAccessors.ObjectTypes.REGISTRY_PACKAGE)
    };
    private static final TableHandler[] EXTENSION_HANDLERS_REGISTRY_ENTRY = new TableHandler[]{
        new DefaultTableHandler(),
        new RegistryEntryTableHandler(ROAccessors.ObjectTypes.REGISTRY_ENTRY)
    };
    private static final TableHandler[] EXTENSION_HANDLERS_CLASSIFICATION_SCHEME = new TableHandler[]{
        new DefaultTableHandler(),
        new RegistryEntryTableHandler(ROAccessors.ObjectTypes.CLASSIFICATION_SCHEME)
    };
    private static final TableHandler[] EXTENSION_HANDLERS_CONCEPT = new TableHandler[]{
        new DefaultTableHandler(),
        new ConceptTableHandler(ROAccessors.ObjectTypes.CONCEPT)
    };
    private static final TableHandler[] EXTENSION_HANDLERS_NULL_TYPE = new TableHandler[]{
        new DefaultTableHandler(),
        new ConceptTableHandler(null),
        new RegistryEntryTableHandler(null),
        new AssociationTableHandler(null),
        new ClassificationTableHandler(null)
    };

    /**
     * Extension handler for concepts.
     */
    public static class ConceptTableHandler extends TableHandler {
        /**
         * Creates a new instance with the given object type.
         */
        public ConceptTableHandler(ROAccessors.ObjectTypes pObjectType) {
            super(pObjectType);
        }

        @Override
        protected int addColumns(StringBuilder pBuilder, String pPrefix, int pIndex) {
            final String table = getTable(pPrefix, "c");
            pBuilder.append(", ");
            pBuilder.append(table);
            pBuilder.append(".value");
            return pIndex + 1;
        }

        @Override
        protected void addJoin(StringBuilder pBuilder, String pPrefix, List<?> pParameters) {
            final String table = getTable(pPrefix, "c");
            final String roTable = getRoTable(pPrefix);
            if (getObjectType() == null) {
                pBuilder.append(" LEFT OUTER");
            }
            pBuilder.append(" JOIN Concepts ");
            pBuilder.append(table);
            pBuilder.append(" ON ");
            pBuilder.append(roTable);
            pBuilder.append(".roType="); //$NON-NLS-1$
            pBuilder.append(String.valueOf(ROAccessors.ObjectTypes.CONCEPT.ordinal()));
            pBuilder.append(" AND "); //$NON-NLS-1$
            pBuilder.append(roTable);
            pBuilder.append(".roKey="); //$NON-NLS-1$
            pBuilder.append(table);
            pBuilder.append(".roKey"); //$NON-NLS-1$
        }

        @Override
        protected int load(final ResultSet pResultSet, final RegistryObject pRegistryObject, int pIndex)
                throws SQLException, JAXRException {
            if (pRegistryObject instanceof Concept) {
                final Concept concept = (Concept) pRegistryObject;
                concept.setValue(pResultSet.getString(pIndex));
            }
            return pIndex+1;
        }
    }

    /**
     * Extension handler for registry entries.
     */
    public static class RegistryEntryTableHandler extends TableHandler {
        /**
         * Creates a new instance with the given object type.
         */
        public RegistryEntryTableHandler(ROAccessors.ObjectTypes pObjectType) {
            super(pObjectType);
        }

        @Override
        protected int addColumns(StringBuilder pBuilder, String pPrefix, int pIndex) {
            final String table = getTable(pPrefix, "re");
            pBuilder.append(", ");
            pBuilder.append(table);
            pBuilder.append(".customType, ");
            pBuilder.append(table);
            pBuilder.append(".expiration, ");
            pBuilder.append(table);
            pBuilder.append(".status, ");
            pBuilder.append(table);
            pBuilder.append(".stability, ");
            pBuilder.append(table);
            pBuilder.append(".majorVersion, ");
            pBuilder.append(table);
            pBuilder.append(".minorVersion, ");
            pBuilder.append(table);
            pBuilder.append(".userVersion");
            return pIndex + 7;
        }

        @Override
        protected void addJoin(StringBuilder pBuilder, String pPrefix, List<?> pParameters) {
            final String table = getTable(pPrefix, "re");
            final String roTable = getRoTable(pPrefix);
            final ROAccessors.ObjectTypes objectType = getObjectType();
            if (objectType == null) {
                pBuilder.append(" LEFT OUTER JOIN RegistryEntries ");
                pBuilder.append(table);
                pBuilder.append(" ON (");
                pBuilder.append(roTable);
                pBuilder.append(".roType=");
                pBuilder.append(ROAccessors.ObjectTypes.CLASSIFICATION_SCHEME.ordinal());
                pBuilder.append(" OR ");
                pBuilder.append(roTable);
                pBuilder.append(".roType=");
                pBuilder.append(ROAccessors.ObjectTypes.REGISTRY_ENTRY.ordinal());
                pBuilder.append(")");
            } else {
                pBuilder.append(" JOIN RegistryEntries ");
                pBuilder.append(table);
                pBuilder.append(" ON ");
                pBuilder.append(roTable);
                pBuilder.append(".roType=");
                pBuilder.append(String.valueOf(objectType.ordinal()));
            }
            pBuilder.append(" AND ");
            pBuilder.append(roTable);
            pBuilder.append(".roKey=");
            pBuilder.append(table);
            pBuilder.append(".roKey");
        }

        @Override
        protected int load(final ResultSet pResultSet, final RegistryObject pRegistryObject, int pIndex)
                throws SQLException, JAXRException {
            if (pRegistryObject instanceof RegistryEntry) {
                final RegistryEntryImpl<?> re = (RegistryEntryImpl<?>) pRegistryObject;
                re.setCustomType(pResultSet.getString(pIndex));
                re.setExpiration(pResultSet.getDate(pIndex+1));
                re.setStability(pResultSet.getInt(pIndex+3));
                re.setMajorVersion(pResultSet.getInt(pIndex+4));
                re.setMinorVersion(pResultSet.getInt(pIndex+5));
                re.setUserVersion(pResultSet.getString(pIndex+6));
            }
            return pIndex+7;
        }
    }

    /**
     * Table handlers for standard registry objects.
     */
    public static class DefaultTableHandler extends TableHandler {
        public DefaultTableHandler() {
            super(null);
        }

        @Override
        protected int addColumns(StringBuilder pBuilder, String pPrefix, int pIndex) {
            final String table = getRoTable(pPrefix);
            pBuilder.append(table);
            pBuilder.append(".roKey, ");
            pBuilder.append(table);
            pBuilder.append(".roType, ");
            pBuilder.append(table);
            pBuilder.append(".roOwnerRestricting, ");
            pBuilder.append(table);
            pBuilder.append(".roOwnerCascading, ");
            pBuilder.append(table);
            pBuilder.append(".pos");
            return pIndex + 5;
        }

        @Override
        protected void addJoin(StringBuilder pBuilder, String pPrefix, List<?> pParameters) {
            throw new IllegalStateException("Join not supported"); //$NON-NLS-1$
        }

        @Override
        protected int load(ResultSet pResultSet, RegistryObject pRegistryObject, int pIndex)
                throws SQLException, JAXRException {
            final RegistryObjectImpl<?> ro = (RegistryObjectImpl<?>) pRegistryObject;
            ro.setState(ROState.loaded);

            if (pRegistryObject instanceof OwnedRegistryObject<?>) {
                final OwnedRegistryObject<?> oro = (OwnedRegistryObject<?>) ro;
                final String ownerKeyRestricting = pResultSet.getString(pIndex+2);
                final String ownerKeyCascading = pResultSet.getString(pIndex+3);
                assert(ownerKeyRestricting == null  ||  ownerKeyCascading == null);
                final String ownerKey = ownerKeyCascading == null ? ownerKeyRestricting : ownerKeyCascading;
                assert(ownerKey != null);
                oro.setOwner(new KeyImpl(ownerKey));
                int pos = pResultSet.getInt(pIndex+4);
                assert(!pResultSet.wasNull());
                oro.setPosition(pos);
            }

            return pIndex+5;
        }
    }

    /**
     * Extension handler for associations.
     */
    public static class AssociationTableHandler extends TableHandler {
        /**
         * Creates a new instance with the given object type.
         */
        public AssociationTableHandler(ROAccessors.ObjectTypes pObjectType) {
            super(pObjectType);
        }

        @Override
        protected int addColumns(StringBuilder pBuilder, String pPrefix, int pIndex) {
            final String table = getTable(pPrefix, "assoc");
            pBuilder.append(", ");
            pBuilder.append(table);
            pBuilder.append(".roKeyTarget, ");
            pBuilder.append(table);
            pBuilder.append(".roKeyType, ");
            pBuilder.append(table);
            pBuilder.append(".num");
            return pIndex + 3;
        }

        @Override
        protected void addJoin(StringBuilder pBuilder, String pPrefix, List<?> pParameters) {
            final String table = getTable(pPrefix, "assoc");
            final String roTable = getRoTable(pPrefix);
            if (getObjectType() == null) {
                pBuilder.append(" LEFT OUTER"); //$NON-NLS-1$
            }
            pBuilder.append(" JOIN Associations ");
            pBuilder.append(table);
            pBuilder.append(" ON ");
            pBuilder.append(roTable);
            pBuilder.append(".roType=");
            pBuilder.append(ROAccessors.ObjectTypes.ASSOCIATION.ordinal());
            pBuilder.append(" AND ");
            pBuilder.append(roTable);
            pBuilder.append(".roKey=");
            pBuilder.append(table);
            pBuilder.append(".roKey");
        }

        @Override
        protected int load(final ResultSet pResultSet, final RegistryObject pRegistryObject, int pIndex)
                throws SQLException, JAXRException {
            if (pRegistryObject instanceof Association) {
                final AssociationImpl assoc = (AssociationImpl) pRegistryObject;
                final String targetKey = pResultSet.getString(pIndex);
                assert(targetKey != null);
                assoc.setTargetObjectKey(new KeyImpl(targetKey));
                final String typeKey = pResultSet.getString(pIndex+1);
                assert(typeKey != null);
                assoc.setAssociationTypeKey(new KeyImpl(typeKey));
                assoc.setNum(pResultSet.getInt(pIndex+2));
                assert(!pResultSet.wasNull());
            }
            return pIndex+3;
        }
    }

    /**
     * Extension handler for classifications.
     */
    public static class ClassificationTableHandler extends TableHandler {
        /**
         * Creates a new instance with the given object type.
         */
        public ClassificationTableHandler(ROAccessors.ObjectTypes pObjectType) {
            super(pObjectType);
        }

        @Override
        protected int addColumns(StringBuilder pBuilder, String pPrefix, int pIndex) {
            final String table = getTable(pPrefix, "cl");
            pBuilder.append(", ");
            pBuilder.append(table);
            pBuilder.append(".roKeyConcept, ");
            pBuilder.append(table);
            pBuilder.append(".num");
            return pIndex + 2;
        }

        @Override
        protected void addJoin(StringBuilder pBuilder, String pPrefix, List<?> pParameters) {
            final String table = getTable(pPrefix, "cl");
            final String roTable = getRoTable(pPrefix);
            if (getObjectType() == null) {
                pBuilder.append(" LEFT OUTER"); //$NON-NLS-1$
            }
            pBuilder.append(" JOIN Classifications ");
            pBuilder.append(table);
            pBuilder.append(" ON ");
            pBuilder.append(roTable);
            pBuilder.append(".roType=");
            pBuilder.append(ROAccessors.ObjectTypes.CLASSIFICATION.ordinal());
            pBuilder.append(" AND ");
            pBuilder.append(roTable);
            pBuilder.append(".roKey=");
            pBuilder.append(table);
            pBuilder.append(".roKey");
        }

        @Override
        protected int load(final ResultSet pResultSet, final RegistryObject pRegistryObject, int pIndex)
                throws SQLException, JAXRException {
            if (pRegistryObject instanceof Classification) {
                final ClassificationImpl cl = (ClassificationImpl) pRegistryObject;
                final String conceptKey = pResultSet.getString(pIndex);
                assert(conceptKey != null);
                cl.setConceptKey(new KeyImpl(conceptKey));
                cl.setNum(pResultSet.getInt(pIndex+1));
                assert(!pResultSet.wasNull());
            }
            return pIndex+2;
        }
    }

    public static RegistryObjectQuery getRegistryObjectQuery(String pQName, String pPrefix) {
        ObjectTypes registryObjectType = null;
        Collection<FilterPredicate> predicates = null;
        if (pQName != null) {
            for (ObjectTypes objectType : ROAccessors.ObjectTypes.values()) {
                if (objectType.getInterfaceName().equals(pQName)) {
                    registryObjectType = objectType;
                    break;
                }
            }
            if (registryObjectType == null) {
                registryObjectType = ObjectTypes.REGISTRY_ENTRY;
                final Collection<String> objectTypes = Collections.singleton(pQName);
                predicates = Collections.singleton((FilterPredicate) new CustomObjectTypePredicate(objectTypes, pPrefix));
            }
        }
        return new RegistryObjectQuery(registryObjectType, pPrefix, predicates);
    }
    
    public static class RegistryObjectQuery {
        private final ROAccessors.ObjectTypes objectType;
        private final String prefix;
        private final Collection<FilterPredicate> predicates;
        private final TableHandler[] tableHandlers;

        public RegistryObjectQuery(ROAccessors.ObjectTypes pObjectType, String pPrefix, Collection<FilterPredicate> pPredicates) {
            objectType = pObjectType;
            prefix = pPrefix;
            predicates = pPredicates;
            tableHandlers = getExtensionHandlers(pObjectType);
        }

        public TableHandler[] getTableHandlers() {
            return tableHandlers;
        }

        public String getPrefix() {
            return prefix;
        }

        public ROAccessors.ObjectTypes getObjectType() {
            return objectType;
        }

        public Collection<FilterPredicate> getPredicates() {
            return predicates;
        }
    }
    
    private final String query;
    private final Object[] parameters;
    private final RegistryObjectQuery[] queries;
    private String sep = " WHERE ";

    private static TableHandler[] getExtensionHandlers(ROAccessors.ObjectTypes pObjectType) {
        if (pObjectType == null) {
            return EXTENSION_HANDLERS_NULL_TYPE;
        }
        switch (pObjectType) {
            case CONCEPT:
                return EXTENSION_HANDLERS_CONCEPT;
            case CLASSIFICATION_SCHEME:
                return EXTENSION_HANDLERS_CLASSIFICATION_SCHEME;
            case REGISTRY_ENTRY:
                return EXTENSION_HANDLERS_REGISTRY_ENTRY;
            case REGISTRY_PACKAGE:
                return EXTENSION_HANDLERS_REGISTRY_PACKAGE;
            case SERVICE:
                return EXTENSION_HANDLERS_SERVICE;
            case EXTRINSIC_OBJECT:
                return EXTENSION_HANDLERS_EXTRINSIC_OBJECT;
            case CLASSIFICATION:
                return EXTENSION_HANDLERS_CLASSIFICATION;
            case ASSOCIATION:
                return EXTENSION_HANDLERS_ASSOCIATION;
            case AUDITABLE_EVENT:
            case EXTERNAL_IDENTIFIER:
            case EXTERNAL_LINK:
            case ORGANIZATION:
            case SERVICE_BINDING:
            case SPECIFICATION_LINK:
            case USER:
                return EXTENSION_HANDLERS_REGISTRY_OBJECT;
            default:
                return null;
        }
    }


    public static String getAliasAssociations(RegistryObjectQuery pSource) {
        return getTable(pSource.getPrefix(), "assoc");
    }

    public static String getAliasRegistryObjects(RegistryObjectQuery pSource) {
        return getTable(pSource.getPrefix(), "ro");
    }

    public static String getAliasRegistryEntries(RegistryObjectQuery pSource) {
        return getTable(pSource.getPrefix(), "re");
    }

    public static String getAliasSlots(RegistryObjectQuery pSource) {
        return getTable(pSource.getPrefix(), "ros");
    }

    public static String getAliasSlotValues(RegistryObjectQuery pSource) {
        return getTable(pSource.getPrefix(), "rosv");
    }

    static String getTable(String pPrefix, String pSuffix) {
        if (pPrefix == null || pPrefix.length() == 0) {
            return pSuffix;
        }
        return pPrefix + pSuffix;
    }

    static String getRoTable(String pPrefix) {
        return getTable(pPrefix, "ro");
    }

    /**
     * Creates a new instance, which is searching for the given object types.
     */
    public RegistryObjectLoader(RegistryObjectQuery[] pQueries, Collection<OrderPredicate> pOrderPredicates)
            throws JAXRException, SQLException {
        queries = pQueries;
        final StringBuilder sb = new StringBuilder();
        sb.append("SELECT ");
        int index = 1;
        for (int i = 0;  i < queries.length;  i++) {
            final RegistryObjectQuery q = queries[i];
            if (i > 0) {
                sb.append(", ");
            }
            final TableHandler[] tableHandlers = q.getTableHandlers();
            index = tableHandlers[0].addColumns(sb, q.getPrefix(), index);
        }
        for (RegistryObjectQuery q : queries) {
            final TableHandler[] tableHandlers = q.getTableHandlers();
            final String prefix = q.getPrefix();
            for (int j = 1;  j < tableHandlers.length;  j++) {
                index = tableHandlers[j].addColumns(sb, prefix, index);
            }
        }
        final List<Object> parameterList = new ArrayList<Object>();
        for (int i = 0;  i < queries.length;  i++) {
            final RegistryObjectQuery q = queries[i];
            final String table = getRoTable(q.getPrefix());
            if (i > 0) {
                sb.append(" JOIN RegistryObjects ");
                sb.append(table);
                final ObjectTypes objectType = q.getObjectType();
                String objectTypeSep = " ON ";
                if (objectType != null) {
                    sb.append(objectTypeSep);
                    objectTypeSep = " AND ";
                    sb.append(table);
                    sb.append(".roType=");
                    sb.append(objectType.ordinal());
                }
            } else {
                sb.append(" FROM RegistryObjects ");
                sb.append(table);
            }
        }
        for (RegistryObjectQuery q : queries) {
            final TableHandler[] otTableHandlers = q.getTableHandlers();
            for (int j = 1;  j < otTableHandlers.length;  j++) {
                otTableHandlers[j].addJoin(sb, q.getPrefix(), parameterList);
            }
        }
        {
            final RegistryObjectQuery q = queries[0];
            final ObjectTypes objectType = q.getObjectType();
            if (objectType != null) {
                final String table = getRoTable(q.getPrefix());
                sb.append(sep);
                sep = " AND ";
                sb.append(table);
                sb.append(".roType=");
                sb.append(objectType.ordinal());
            }
        }
        for (RegistryObjectQuery q : queries) {
            final Collection<FilterPredicate> predicates = q.getPredicates();
            appendToQuery(predicates, parameterList, sep, sb, true);
        }
        appendToQuery(pOrderPredicates, parameterList, sb);
        parameters = parameterList.toArray();
        query = sb.toString();
    }

    /**
     * Creates a new instance, which is searching for the given object type.
     */
    public RegistryObjectLoader(ROAccessors.ObjectTypes pObjectType, Collection<FilterPredicate> pFilterPredicates, Collection<OrderPredicate> pOrderPredicates)
            throws JAXRException, SQLException {
        this(new RegistryObjectQuery[]{new RegistryObjectQuery(pObjectType, null, pFilterPredicates)}, pOrderPredicates);
    }

    /**
     * Called to read multiple result objects by specifying additional predicates.
     */
    public <RO extends RegistryObject> List<RegistryObject> getResultList(RegistryService pRegistryService,
            Collection<FilterPredicate> pFilterPredicates, Collection<OrderPredicate> pOrderPredicates) throws JAXRException {
        if ((pFilterPredicates == null  ||  pFilterPredicates.size() == 0)  &&
            (pOrderPredicates == null  ||  pOrderPredicates.size() == 0)) {
            return getResultList(pRegistryService, query, parameters);
        }
        final StringBuilder sb = new StringBuilder(query);
        final List<Object> params = new ArrayList<Object>();
        params.addAll(Arrays.asList(parameters));
        try {
            appendToQuery(pFilterPredicates, params, sep, sb, false);
            appendToQuery(pOrderPredicates, params, sb);
            return getResultList(pRegistryService, sb.toString(), params.toArray());
        } catch (SQLException e) {
            throw new JAXRException(e);
        }
    }

    private void appendToQuery(Collection<OrderPredicate> pPredicates,
            final List<Object> pParams, final StringBuilder sb) throws JAXRException, SQLException {
        if (pPredicates != null) {
            String mySep = " ORDER BY ";
            for (OrderPredicate predicate : pPredicates) {
                sb.append(mySep);
                mySep = ", ";
                predicate.add(sb, pParams);
            }
        }
    }

    private void appendToQuery(Collection<FilterPredicate> pPredicates,
            final List<Object> pParams, String pSep, final StringBuilder sb,
            boolean pModifySep)
            throws JAXRException, SQLException {
        if (pPredicates != null) {
            String mySep = pSep;
            for (FilterPredicate predicate : pPredicates) {
                sb.append(mySep);
                mySep = " AND ";
                if (pModifySep) {
                    sep = mySep;
                }
                predicate.add(sb, pParams);
            }
        }
    }

    /**
     * Called to read multiple result objects.
     */
    public <RO extends RegistryObject> List<RegistryObject> getResultList(RegistryService pRegistryService)
            throws JAXRException {
        return getResultList(pRegistryService, query, parameters);
    }

    public List<RegistryObject> getResultList(final RegistryService pService, final PreparedStatement pStmt)
            throws JAXRException {
        final List<RegistryObject> list = new ArrayList<RegistryObject>();
        ResultSet rs = null;
        try {
            rs = pStmt.executeQuery();
            while (rs.next()) {
                int type = rs.getInt(2);
                assert(!rs.wasNull());
                final ROAccessor<?> roAcc = ROAccessors.getROAccessor(type);
                final String key = rs.getString(1);
                assert(key != null);
                final RegistryObject ro = roAcc.create(pService, new KeyImpl(key));
                load(ro, rs);
                list.add(ro);
            }
            rs.close();
            rs = null;
            return list;
        } catch (SQLException e) {
            throw new JAXRException(e);
        } finally {
            if (rs != null) { try { rs.close(); } catch (Throwable t) { /* Ignore me */ } }
        }
    }

    public List<RegistryObject[]> getResultArrayList(final RegistryService pService, final PreparedStatement pStmt)
            throws JAXRException {
        final List<RegistryObject[]> list = new ArrayList<RegistryObject[]>();
        ResultSet rs = null;
        try {
            rs = pStmt.executeQuery();
            while (rs.next()) {
                final RegistryObject[] row = new RegistryObject[queries.length];
                int index = 1;
                for (int i = 0;  i < row.length;  i++) {
                    int type = rs.getInt(index+1);
                    assert(!rs.wasNull());
                    final ROAccessor<?> roAcc = ROAccessors.getROAccessor(type);
                    final String key = rs.getString(index+1);
                    assert(key != null);

                    final RegistryObjectImpl<?> ro = (RegistryObjectImpl<?>) roAcc.create(pService, new KeyImpl(key));
                    ro.setState(ROState.loaded);
                    
                    final TableHandler[] tableHandlers = queries[i].tableHandlers;
                    index = tableHandlers[0].load(rs, ro, index);
                    row[i] = ro;
                }
                for (int i = 0;  i < row.length;  i++) {
                    final RegistryObject ro = row[i];
                    final TableHandler[] tableHandlers = queries[i].tableHandlers;
                    for (int j = 1;  j < tableHandlers.length;  j++) {
                        index = tableHandlers[j].load(rs, ro, index);
                    }
                }
                list.add(row);
            }
            rs.close();
            rs = null;
            return list;
        } catch (SQLException e) {
            throw new JAXRException(e);
        } finally {
            if (rs != null) { try { rs.close(); } catch (Throwable t) { /* Ignore me */ } }
        }
    }

    private <RO extends RegistryObject> List<RegistryObject> getResultList(final RegistryService pRegistryService,
            String pQuery, Object[] pParameters) throws JAXRException {
        final List<RegistryObject> list = new ArrayList<RegistryObject>();
        new ObjQueryUser(pQuery, pParameters){
            @Override
            protected void action(ResultSet pResultSet) throws JAXRException,
                    SQLException {
                while (pResultSet.next()) {
                    int type = pResultSet.getInt(2);
                    assert(!pResultSet.wasNull());
                    final ROAccessor<?> roAcc = ROAccessors.getROAccessor(type);
                    final String key = pResultSet.getString(1);
                    assert(key != null);
                    final RegistryObject ro = roAcc.create(pRegistryService, new KeyImpl(key));
                    load(ro, pResultSet);
                    list.add(ro);
                }
                setResult(Integer.valueOf(list.size()));
            }
        }.run(pRegistryService);
        return list;
    }

    /**
     * Called to read a single result object by applying additional parameters.
     */
    @SuppressWarnings("unchecked")
    public <RO extends RegistryObject> RO getResultObject(RegistryService pRegistryService,
            Collection<FilterPredicate> pPredicates) throws JAXRException {
        final RegistryObject ro;
        if (pPredicates == null  ||  pPredicates.isEmpty()) {
            ro = getResultObject(pRegistryService, query, parameters);
        } else {
            final StringBuilder sb = new StringBuilder(query);
            final List<Object> params = new ArrayList<Object>();
            try {
                appendToQuery(pPredicates, params, sep, sb, false);
                ro = getResultObject(pRegistryService, sb.toString(), params.toArray());
            } catch (SQLException e) {
                throw new JAXRException(e);
            }
        }
        return (RO) ro;
    }

    /**
     * Called to read a single result object.
     */
    @SuppressWarnings("unchecked")
    public <RO extends RegistryObject> RO getResultObject(RegistryService pRegistryService)
            throws JAXRException {
        final RegistryObject ro = getResultObject(pRegistryService, query, parameters);
        return (RO) ro;
    }

    @SuppressWarnings("unchecked")
    private <RO extends RegistryObject> ROAccessor<RO> asROAccessor(ROAccessor<?> pRO) {
        return (ROAccessor<RO>) pRO;
    }

    private RegistryObject getResultObject(final RegistryService pRegistryService,
            String pQuery, Object[] pParameters) throws JAXRException {
        final QueryUser<RegistryObject> qu = new QueryUser<RegistryObject>(pQuery, pParameters){
            @Override
            protected void action(ResultSet pResultSet) throws JAXRException,
                    SQLException {
                if (pResultSet.next()) {
                    final int type = pResultSet.getInt(2);
                    assert(!pResultSet.wasNull());
                    final ROAccessor<?> roAcc = asROAccessor(ROAccessors.getROAccessor(type));
                    final String key = pResultSet.getString(1);
                    assert(key != null);
                    final RegistryObject ro = roAcc.create(pRegistryService, new KeyImpl(key));
                    load(ro, pResultSet);
                    setResult(ro);
                }
            }
        };
        qu.run(pRegistryService);
        return qu.getResult();
    }

    private int load(RegistryObject pRegistryObject, ResultSet pResultSet) throws SQLException, JAXRException {
        final RegistryObjectImpl<?> ro = (RegistryObjectImpl<?>) pRegistryObject;
        ro.setState(ROState.loaded);

        final TableHandler[] tableHandlers = queries[0].tableHandlers;
        int index = 1;
        for (int i = 0;  i < tableHandlers.length;  i++) {
            index = tableHandlers[i].load(pResultSet, pRegistryObject, index);
        }
        return index;
    }

    /**
     * Called to load the given objects data from the registry.
     */
    public void load(RegistryService pRegistryService, final RegistryObject pRegistryObject)
            throws JAXRException {
        final FilterPredicate predicate = new KeyPredicate(Collections.singleton(pRegistryObject.getKey()));
        final StringBuilder sb = new StringBuilder(query);
        final List<Object> params = new ArrayList<Object>();
        params.addAll(Arrays.asList(parameters));
        try {
            appendToQuery(Collections.singleton(predicate), params, sep, sb, false);
        } catch (SQLException e) {
            throw new JAXRException(e);
        }
        new ObjQueryUser(sb.toString(), params.toArray()){
            @Override
            protected void action(ResultSet pResultSet) throws JAXRException,
                    SQLException {
                if (!pResultSet.next()) {
                    throw new IllegalStateException("Object not found: " + pRegistryObject.getKey()); //$NON-NLS-1$
                }
                final int type = pResultSet.getInt(2);
                assert(!pResultSet.wasNull()  &&  type == queries[0].objectType.ordinal());
                load(pRegistryObject, pResultSet);
                if (pResultSet.next()) {
                    throw new IllegalStateException("Multiple objects found: " + pRegistryObject.getKey()); //$NON-NLS-1$
                }
            }
        }.run(pRegistryService);
    }

    public String getQuery() {
        return query;
    }

    public Object[] getParameters() {
        return parameters;
    }
}
