/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.sql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import javax.xml.registry.JAXRException;
import javax.xml.registry.RegistryService;

import org.apache.labs.jaxmas.registry.infomodel.RegistryServiceImpl;


/**
 * Utility class for performing database operations.
 */
public class Sql {
	/**
	 * Called to perform the given prepared statement with the given parameters.
	 * @return The result of {@link PreparedStatement#executeUpdate()}.
	 */
	public static int count(RegistryService pRegistryService, String pStmt, Object... pParams) throws JAXRException {
		return count(((RegistryServiceImpl) pRegistryService).getConnection().getConnection(), pStmt, pParams);
	}

	/**
	 * Called to perform the given prepared statement with the given parameters.
	 * @return The result of {@link PreparedStatement#executeUpdate()}.
	 */
	public static int count(Connection pConnection, String pStmt, Object... pParams) throws JAXRException {
		final StmtUser<Integer> stmtUser = new StmtUser<Integer>(pStmt, pParams){
			@Override
			protected void action(PreparedStatement pStatement) throws JAXRException, SQLException {
				setResult(new Integer(pStatement.executeUpdate()));
			}
		};
		stmtUser.run(pConnection);
		return stmtUser.getResult().intValue();
	}

	/**
	 * Called to perform the given prepared statement with the given parameters.
	 * @return The result of {@link PreparedStatement#executeQuery()}.
	 */
	public static int intQuery(RegistryService pRegistryService, String pStmt, Object... pParams) throws JAXRException {
		return intQuery(((RegistryServiceImpl) pRegistryService).getConnection().getConnection(), pStmt, pParams);
	}

	/**
	 * Called to perform the given prepared statement with the given parameters.
	 * @return The result of {@link PreparedStatement#executeQuery()}.
	 */
	public static int intQuery(Connection pConnection, String pStmt, Object... pParams) throws JAXRException {
		final QueryUser<Integer> queryUser = new QueryUser<Integer>(pStmt, pParams){
			@Override
			protected void action(ResultSet pResultSet) throws JAXRException, SQLException {
				boolean hasResult = pResultSet.next();
				assert(hasResult);
				int i = pResultSet.getInt(1);
				setResult(new Integer(i));
				assert(!pResultSet.wasNull());
				hasResult = pResultSet.next();
				assert(!hasResult);
			}
		};
		queryUser.run(pConnection);
		return queryUser.getResult().intValue();
	}

	/**
	 * Called to perform the given prepared statement with the given parameters.
	 */
	public static void run(RegistryService pRegistryService, String pStmt, Object... pParams) throws JAXRException {
		run(((RegistryServiceImpl) pRegistryService).getConnection().getConnection(), pStmt, pParams);
	}

	/**
	 * Called to perform the given prepared statement with the given parameters.
	 */
	public static void run(Connection pConnection, String pStmt, Object... pParams) throws JAXRException {
		new ObjStmtUser(pStmt, pParams){
			@Override
			protected void action(PreparedStatement pStatement) throws JAXRException, SQLException {
				pStatement.executeUpdate();
			}
		}.run(pConnection);
	}
}
