/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.sql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.Calendar;
import java.util.Date;

import javax.xml.registry.JAXRException;
import javax.xml.registry.infomodel.Key;

import org.apache.labs.jaxmas.registry.util.Calendars;
import org.apache.labs.jaxmas.registry.util.Logger;



/**
 * Extension of {@link ConnUser} for database operations, which are
 * executing a prepared statement.
 */
public abstract class StmtUser<O extends Object> extends ConnUser<O> {
	private final String statement;
	private Object[] params;

	/**
	 * Creates a new instance with the given statement and parameters.
	 */
	public StmtUser(String pStatement, Object... pParams) {
		statement = pStatement;
		params = pParams;
	}

	/**
	 * Creates a new instance with the given statement and no parameters.
	 */
	public StmtUser(String pStatement) {
		statement = pStatement;
		params = null;
	}

	/**
	 * This method must be implemented to perform the actual operation.
	 * It is called from within {@link #run(Connection)}.
	 */
	protected abstract void action(java.sql.PreparedStatement pStatement) throws JAXRException, SQLException;

	/**
	 * Called to prepare a statement by setting the parameters.
	 */
	public static void setParams(PreparedStatement pStmt, Object[] pParams) throws SQLException, JAXRException {
		if (pParams != null) {
			for (int i = 0;  i < pParams.length;  i++) {
				Object o = pParams[i];
				if (o == null) {
					pStmt.setNull(i+1, Types.VARCHAR);
				} else if (o instanceof String) {
					pStmt.setString(i+1, (String) o);
				} else if (o instanceof Integer) {
					pStmt.setInt(i+1, ((Integer) o).intValue());
				} else if (o instanceof Long) {
					pStmt.setLong(i+1, ((Long) o).longValue());
				} else if (o instanceof Boolean) {
					pStmt.setBoolean(i+1, ((Boolean) o).booleanValue());
				} else if (o instanceof Short) {
					pStmt.setShort(i+1, ((Short) o).shortValue());
				} else if (o instanceof Byte) {
					pStmt.setByte(i+1, ((Byte) o).byteValue());
				} else if (o instanceof Float) {
					pStmt.setFloat(i+1, ((Float) o).floatValue());
				} else if (o instanceof Double) {
					pStmt.setDouble(i+1, ((Double) o).doubleValue());
				} else if (o instanceof Date) {
					final Calendar cal = Calendars.nowUTC();
					final Timestamp ts = (o instanceof Timestamp) ? ((Timestamp) o) : new Timestamp(((Date) o).getTime());
					pStmt.setTimestamp(i+1, ts, cal);
				} else if (o instanceof Key) {
				    pStmt.setString(i+1, ((Key) o).getId());
				} else if (o instanceof Enum<?>) {
				    pStmt.setInt(i+1, ((Enum<?>) o).ordinal());
				} else {
					throw new IllegalStateException("Invalid parameter type: " + o.getClass().getName()); //$NON-NLS-1$
				}
			}
		}
	}

	/**
	 * Returns the query parameters.
	 */
	protected Object[] getParams() {
		return params;
	}

	@Override
	protected void logEntering(Logger pLog, String pName) {
		if (pLog.isDebugEnabled()) {
			pLog.entering(pName, statement, getParams());
		}
	}

	@Override
	protected void action(Connection pConnection) throws JAXRException, SQLException {
		PreparedStatement stmt = pConnection.prepareStatement(statement);
		try {
			setParams(stmt, getParams());
			action(stmt);
			stmt.close();
			stmt = null;
		} finally {
			if (stmt != null) { try { stmt.close(); } catch (Throwable t) { /* Ignore me */ } }
		}
	}
}
