/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.util;

import java.text.DateFormat;


/**
 * Default implementation of {@link Logger}, which is writing to {@link System#err}.
 */
public class DefaultLogger implements Logger {
    /**
     * The trace level.
     */
    public static final int LEVEL_TRACE = 5;
    /**
     * The debug level.
     */
	public static final int LEVEL_DEBUG = 4;
    /**
     * The info level.
     */
	public static final int LEVEL_INFO = 3;
    /**
     * The warning level.
     */
	public static final int LEVEL_WARN = 2;
    /**
     * The error level.
     */
	public static final int LEVEL_ERR = 1;
    /**
     * A log level, which suppresses logging at all.
     */
	public static final int LEVEL_NONE = 0;

	private static final String DEBUG = "DEBUG"; //$NON-NLS-1$
    private static final String ERROR = "ERROR"; //$NON-NLS-1$

	private final DateFormat df = DateFormat.getDateTimeInstance();
	private final String id;
	private int level;

	/**
	 * Creates a new instance.
	 */
	public DefaultLogger(Class<?> pClass, int pLevel) {
		this(pClass.getCanonicalName(), pLevel);
	}

	/**
	 * Creates a new instance.
	 */
	public DefaultLogger(String pId, int pLevel) {
		id = pId;
		level = pLevel;
	}

	@Override
	public boolean isDebugEnabled() {
		return level >= LEVEL_DEBUG;
	}

	@Override
	public boolean isErrorEnabled() {
        return level >= LEVEL_ERR;
    }

	private void asString(StringBuilder pSb, Object[] pArgs) {
		if (pArgs != null) {
			for (int i = 0;  i < pArgs.length;  i++) {
				if (i > 0) {
					pSb.append(", "); //$NON-NLS-1$
				}
				pSb.append(pArgs[i]);
			}
		}
	}

	private StringBuilder fName(String pLevel, String pName) {
		final StringBuilder sb = new StringBuilder();
		synchronized (df) {
			sb.append(df.format(new java.util.Date()));
		}
		sb.append(" ["); //$NON-NLS-1$
		sb.append(pLevel);
		sb.append(',');
		sb.append(Thread.currentThread().getName());
		sb.append("] "); //$NON-NLS-1$
		sb.append(id);
		sb.append(' ');
		sb.append(pName);
		return sb;
	}

	@Override
	public void entering(String pName) {
		if (isDebugEnabled()) {
			System.err.println(fName(DEBUG, pName).append(": ->")); //$NON-NLS-1$
		}
	}

	@Override
	public void entering(String pName, String pMsg) {
		if (isDebugEnabled()) {
			System.err.println(fName(DEBUG, pName).append(": -> ").append(pMsg)); //$NON-NLS-1$
		}
	}

	@Override
	public void entering(String pName, String pMsg, Object... pArgs) {
		if (isDebugEnabled()) {
			final StringBuilder sb = fName(DEBUG, pName).append(": -> ").append(pMsg); //$NON-NLS-1$
			asString(sb, pArgs);
			System.err.println(sb);
		}
	}

	@Override
	public void exiting(String pName) {
		if (isDebugEnabled()) {
			System.err.println(fName(DEBUG, pName).append(": <-")); //$NON-NLS-1$
		}
	}

	@Override
	public void exiting(String pName, String pMsg) {
		if (isDebugEnabled()) {
			System.err.println(fName(DEBUG, pName).append(": <- ").append(pMsg)); //$NON-NLS-1$
		}
	}

	@Override
	public void debug(String pName, String pMsg) {
		if (isDebugEnabled()) {
			System.err.println(fName(DEBUG, pName).append(": ").append(pMsg)); //$NON-NLS-1$
		}
	}

    @Override
    public void error(String pName, Throwable pThrowable) {
        if (isErrorEnabled()) {
            final String msg = pThrowable.getMessage() == null ? pThrowable.getClass().getName() : pThrowable.getMessage();
            System.err.println(fName(ERROR, pName).append(": ").append(msg)); //$NON-NLS-1$
        }
    }
}
