/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry.util;

import org.apache.log4j.Level;


/**
 * Implementation of {@link Logger}, which is based on logj.
 */
public class Log4jLogger implements Logger {
	private final org.apache.log4j.Logger log;

	/**
	 * Creates a new instance.
	 */
	public Log4jLogger(Class<?> pClass) {
		log = org.apache.log4j.Logger.getLogger(pClass);
	}

	/**
	 * Creates a new instance.
	 */
	public Log4jLogger(String pId) {
		log = org.apache.log4j.Logger.getLogger(pId);
	}

	private String asString(Object[] pParams) {
		final StringBuilder sb = new StringBuilder();
		if (pParams != null) {
			for (int i = 0;  i < pParams.length;  i++) {
				if (i > 0) {
					sb.append(", "); //$NON-NLS-1$
				}
				sb.append(pParams[i]);
			}
		}
		return sb.toString();
	}

	@Override
	public boolean isDebugEnabled() {
		return log.isDebugEnabled();
	}

	@Override
	public boolean isErrorEnabled() {
        return log.isEnabledFor(Level.ERROR);
    }

	@Override
	public void entering(String pName) {
		if (log.isDebugEnabled()) {
			log.debug(pName + ": ->"); //$NON-NLS-1$
		}
	}

	@Override
	public void entering(String pName, String pMsg) {
		if (log.isDebugEnabled()) {
			log.debug(pName + ": -> " + pMsg); //$NON-NLS-1$
		}
	}

	@Override
	public void entering(String pName, String pMsg, Object... pArgs) {
		if (log.isDebugEnabled()) {
			log.debug(pName + ": -> " + pMsg + ", " + asString(pArgs)); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}

	@Override
	public void exiting(String pName) {
		if (log.isDebugEnabled()) {
			log.debug(pName + ": <-"); //$NON-NLS-1$
		}
	}

	@Override
	public void exiting(String pName, String pMsg) {
		if (log.isDebugEnabled()) {
			log.debug(pName + ": <- " + pMsg); //$NON-NLS-1$
		}
	}

	@Override
	public void debug(String pName, String pMsg) {
		if (log.isDebugEnabled()) {
			log.debug(pName + ": " + pMsg); //$NON-NLS-1$
		}
	}

    @Override
    public void error(String pName, Throwable pThrowable) {
        if (isErrorEnabled()) {
            final String msg = pThrowable.getMessage() == null ? pThrowable.getClass().getName() : pThrowable.getMessage();
            log.error(pName + ": " + msg, pThrowable); //$NON-NLS-1$
        }
    }
}
