/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry;

import java.util.StringTokenizer;

import javax.xml.registry.BusinessLifeCycleManager;
import javax.xml.registry.BusinessQueryManager;
import javax.xml.registry.Connection;
import javax.xml.registry.JAXRException;
import javax.xml.registry.infomodel.ClassificationScheme;
import javax.xml.registry.infomodel.Concept;

import org.apache.labs.jaxmas.registry.infomodel.RegistryServiceImpl;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import static org.junit.Assert.*;

/**
 * Abstract base class for test cases, where an initialized database
 * and an open registry connection are expected.
 */
public abstract class AbstractInitializedJaxMasTestCase extends AbstractJaxMasTestCase {
    /**
     * Called to ensure that the database is initialized.
     */
    @BeforeClass
    public static void setUpBeforeClass() throws Exception {
        initDatabase();
    }

    /**
     * Called to open a database connection.
     */
    @Before
    public void setUp() throws Exception {
        initConnection();
    }

    /**
     * Called to close the database connection, which has been
     * opened by {@link #setUp()}.
     */
    @After
    public void tearDown() throws Exception {
        closeConnection();
    }

    /**
     * Called after the test is finished in order to shutdown the database.
     */
    @AfterClass
    public static void tearDownAfterClass() throws Exception {
        final Connection connection = newConnection();
        RegistryServiceImpl.getDbDriver(connection.getRegistryService()).shutdown();
    }
    
    /**
     * Returns the current connections business query manager.
     */
    protected BusinessQueryManager getBusinessQueryManager() throws JAXRException {
        return getConnection().getRegistryService().getBusinessQueryManager();
    }

    /**
     * Returns the current connections business lifecycle manager.
     */
    protected BusinessLifeCycleManager getBusinessLifecycleManager() throws JAXRException {
        return getConnection().getRegistryService().getBusinessLifeCycleManager();
    }

    /**
     * Returns the classification scheme with the given name.
     */
    protected ClassificationScheme findClassificationScheme(String pName) throws JAXRException {
        return getBusinessQueryManager().findClassificationSchemeByName(null, pName);
    }

    /**
     * Returns the concept with the given value, the given classification scheme, and no parent concept.
     */
    protected Concept findConcept(ClassificationScheme pClassificationScheme, String pValue) throws JAXRException {
        for (Object o : pClassificationScheme.getChildrenConcepts()) {
            final Concept concept = (Concept) o;
            if (pValue.equals(concept.getValue())) {
                return concept;
            }
        }
        return null;
    }

    /**
     * Returns the concept with the given value, and the given parent concept.
     */
    protected Concept findConcept(Concept pParent, String pValue) throws JAXRException {
        for (Object o : pParent.getChildrenConcepts()) {
            final Concept concept = (Concept) o;
            if (pValue.equals(concept.getValue())) {
                return concept;
            }
        }
        return null;
    }

    /**
     * Searches a concept by its path.
     */
    public Concept getConceptByPath(String pPath) throws JAXRException {
        final StringTokenizer st = new StringTokenizer(pPath, "/"); //$NON-NLS-1$
        if (!st.hasMoreTokens()) {
            fail("Invalid concept path: " + pPath); //$NON-NLS-1$
        }
        final String classificationSchemeName = st.nextToken();
        if (!st.hasMoreTokens()) {
            fail("Invalid concept path: " + pPath); //$NON-NLS-1$
        }
        final ClassificationScheme cs = findClassificationScheme(classificationSchemeName);
        if (cs == null) {
            return null;
        }
        Concept concept = findConcept(cs, st.nextToken());
        if (concept == null) {
            return null;
        }
        while (st.hasMoreTokens()) {
            concept = findConcept(concept, st.nextToken());
            if (concept == null) {
                return null;
            }
        }
        return concept;
    }
}
