/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry;


import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Locale;
import java.util.Properties;

import javax.xml.registry.Connection;
import javax.xml.registry.ConnectionFactory;
import javax.xml.registry.JAXRException;
import javax.xml.registry.infomodel.InternationalString;

import org.apache.labs.jaxmas.registry.infomodel.ConnectionImpl;
import org.apache.labs.jaxmas.registry.schema.DbInitializer;
import org.apache.labs.jaxmas.registry.util.DefaultLoggerFactory;
import org.apache.labs.jaxmas.registry.util.Logger;


/**
 * Abstract base class for deriving JaxMas test cases.
 */
public abstract class AbstractJaxMasTestCase {
    private static final Logger log = DefaultLoggerFactory.getInstance().newLogger(AbstractJaxMasTestCase.class);
    private static Properties properties;
    private static ConnectionFactory connFactory;
    private Connection connection;

    
    /**
     * Creates a new international string with the given value for the
     * locale {@link Locale#ENGLISH}.
     */
    protected InternationalString newString(String pValue) throws JAXRException {
        final InternationalString is = getConnection().getRegistryService().getBusinessLifeCycleManager().createInternationalString();
        is.setValue(Locale.ENGLISH, pValue);
        return is;
    }

    /**
     * Returns the given international strings value for the locale
     * {@link Locale#ENGLISH}.
     */
    protected String asString(InternationalString pValue) throws JAXRException {
        return pValue == null ? null : pValue.getValue(Locale.ENGLISH);
    }
    
    private static File getDbDir() {
        return new File(new File("target"), "db");  //$NON-NLS-1$//$NON-NLS-2$
    }

    private static void initProperties() throws IOException {
        final File dir = getDbDir();
        if (!dir.isDirectory()) {
            dir.mkdirs();
        }
        final String p = "JaxmasRegistryTestCase.properties"; //$NON-NLS-1$
        final URL url = AbstractJaxMasTestCase.class.getResource(p);
        if (url == null) {
            throw new IllegalStateException("Missing resource file: " + p); //$NON-NLS-1$
        }
        properties = new Properties();
        properties.load(url.openStream());
    }

    private static void initConnectionFactory() throws JAXRException {
        connFactory = new ConnectionFactoryImpl();
        connFactory.setProperties(properties);
    }

    /**
     * Creates a new database connection.
     */
    protected static Connection newConnection() throws JAXRException {
        if (connFactory == null) {
            try {
                initDatabase();
            } catch (JAXRException e) {
                throw e;
            } catch (Exception e) {
                throw new JAXRException(e);
            }
        }
        return connFactory.createConnection();
    }

    /**
     * Initializes the database schema.
     */
    public static void initDb() throws JAXRException {
        final String mName = "initDb"; //$NON-NLS-1$
        ConnectionImpl conn = null;
        try {
            conn = (ConnectionImpl) newConnection();
            new DbInitializer(conn).initialize();
            conn.close();
            conn = null;
        } catch (JAXRException e) {
            log.error(mName, e);
            throw e;
        } finally {
            if (conn != null) { try { conn.close(); } catch (Throwable t) { /* Ignore me */ } }
        }
    }

    /**
     * This method is used to initialize the database before running a test case.
     */
    public static void initDatabase() throws Exception {
        initProperties();
        initConnectionFactory();
        initDb();
    }

    /**
     * This method is used to open a database connection.
     */
    public Connection initConnection() throws Exception {
        connection = newConnection();
        return connection;
    }

    /**
     * Returns an open database connection.
     */
    protected Connection getConnection() {
        return connection;
    }

    /**
     * This method is used to close a database connection.
     */
    public void closeConnection() throws Exception {
        if (connection != null) {
            connection.close();
        }
    }
}
