package org.apache.labs.jaxmas.registry;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;

import javax.xml.registry.BulkResponse;
import javax.xml.registry.Connection;
import javax.xml.registry.FindQualifier;
import javax.xml.registry.JAXRException;
import javax.xml.registry.LifeCycleManager;
import javax.xml.registry.RegistryService;
import javax.xml.registry.infomodel.Classification;
import javax.xml.registry.infomodel.ClassificationScheme;
import javax.xml.registry.infomodel.Concept;
import javax.xml.registry.infomodel.RegistryEntry;
import javax.xml.registry.infomodel.RegistryObject;

import org.apache.labs.jaxmas.registry.infomodel.JaxMasQueryManager;
import static org.junit.Assert.*;
import org.junit.Test;


/**
 * Test case for custom object types.
 */
public class CustomTypeTest extends AbstractInitializedJaxMasTestCase {
	private static final String MY_TYPE1 = "{http://namespaces.foo.com/jaxr}MyType1"; //$NON-NLS-1$
	private static final String MY_TYPE2 = "{http://namespaces.foo.com/jaxr}MyType2"; //$NON-NLS-1$
	private static final String RO_NAME = CustomTypeTest.class.getName();

	private Concept findObjectType(Connection pConnection, String pName, String pValue) throws JAXRException {
		final ClassificationScheme objectTypes = findClassificationScheme("ObjectType"); //$NON-NLS-1$
		Concept concept = findConcept(objectTypes, pValue);
		if (concept == null) {
			concept = pConnection.getRegistryService().getBusinessLifeCycleManager().createConcept(objectTypes, pName, pValue);
			pConnection.getRegistryService().getBusinessLifeCycleManager().saveConcepts(Collections.singleton(concept));
		}
		return concept;
	}

	@SuppressWarnings("unchecked")
	private static Collection<RegistryEntry> asRegistryEntryCollection(Collection<?> pCollection) {
		return (Collection<RegistryEntry>) pCollection;
	}
	
	private RegistryEntry findObjectsByObjectType(RegistryService pService, Collection<String> pObjectTypes, String pName)
			throws JAXRException {
		final JaxMasQueryManager jqm = (JaxMasQueryManager) pService.getBusinessQueryManager();
		final BulkResponse br = jqm.findObjects(Collections.singleton(FindQualifier.EXACT_NAME_MATCH),
				pObjectTypes, Collections.singleton(pName), null,
				null, null, null);
		final Collection<RegistryEntry> list = asRegistryEntryCollection(br.getCollection());
		switch (list.size()) {
			case 0:
				return null;
			case 1:
				return list.iterator().next();
			default:
				throw new IllegalStateException("Multiple matching objects named " + pName + " found.");  //$NON-NLS-1$//$NON-NLS-2$
		}
	}
	
	@Test public void testCreateCustomObjectType() throws Exception {
		final String name = RO_NAME + '-' + System.currentTimeMillis();
		final Concept myType1, myType2;
		final RegistryEntry registryEntry;
		Connection conn = newConnection();
		try {
			myType1 = findObjectType(conn, "My Type 1", MY_TYPE1); //$NON-NLS-1$
			assertEquals(MY_TYPE1, myType1.getValue());

			myType2 = findObjectType(conn, "My Type 2", MY_TYPE2); //$NON-NLS-1$
			assertEquals(MY_TYPE2, myType2.getValue());
			assertTrue(!myType1.equals(myType2));
			assertEquals(myType1.getParent(), myType2.getParent());

			registryEntry = (RegistryEntry) conn.getRegistryService().getBusinessLifeCycleManager().createObject(LifeCycleManager.REGISTRY_ENTRY);
			registryEntry.setName(conn.getRegistryService().getBusinessLifeCycleManager().createInternationalString(name));
			final Concept registryEntryType = registryEntry.getObjectType();
			assertTrue(!registryEntryType.equals(myType1));
			assertTrue(!registryEntryType.equals(myType2));
			assertEquals(registryEntryType.getParent(), myType1.getParent());
			assertEquals(registryEntryType.getParent(), myType2.getParent());

			final Classification cl1 = conn.getRegistryService().getBusinessLifeCycleManager().createClassification(myType1);
			registryEntry.addClassification(cl1);
			assertEquals(myType1, registryEntry.getObjectType());
			conn.getRegistryService().getBusinessLifeCycleManager().saveObjects(Collections.singleton(registryEntry));
			conn.close();
			conn = null;
		} finally {
			if (conn != null) { try { conn.close(); } catch (Throwable t) { /* Ignore me */ } }
		}

		conn = newConnection();
		try {
			final RegistryObject ro1 = conn.getRegistryService().getBusinessQueryManager().getRegistryObject(registryEntry.getKey().getId());
			assertEquals(myType1, ro1.getObjectType());

			final Classification cl2 = conn.getRegistryService().getBusinessLifeCycleManager().createClassification(myType2);
			ro1.addClassification(cl2);
			assertEquals(myType2, ro1.getObjectType());
			conn.getRegistryService().getBusinessLifeCycleManager().saveObjects(Collections.singleton(ro1));
			conn.close();
			conn = null;
		} finally {
			if (conn != null) { try { conn.close(); } catch (Throwable t) { /* Ignore me */ } }
		}

		conn = newConnection();
		try {
			final RegistryService rs = conn.getRegistryService();
			final RegistryObject ro2 = rs.getBusinessQueryManager().getRegistryObject(registryEntry.getKey().getId());
			assertEquals(myType2, ro2.getObjectType());

			assertNull(findObjectsByObjectType(rs, Collections.singleton(MY_TYPE1), name));
			final RegistryObject ro3 = findObjectsByObjectType(rs, Arrays.asList(new String[]{MY_TYPE1, MY_TYPE2}), name);
			assertNotNull(ro3);
			assertEquals(ro2, ro3);
			final RegistryObject ro4 = findObjectsByObjectType(rs, Collections.singleton(MY_TYPE2), name);
			assertNotNull(ro4);
			assertEquals(ro2, ro4);
			
			conn.close();
			conn = null;
		} finally {
			if (conn != null) { try { conn.close(); } catch (Throwable t) { /* Ignore me */ } }
		}
	}
}
